#!/usr/bin/python

# Copyright (C) 2005 by Hans Ulrich Niedermann
# Portions Copyright (C) 2005 by Canonical Ltd

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#<<< code taken from bzr (C) Canonical

import os, sys

try:
    version_info = sys.version_info
except AttributeError:
    version_info = 1, 5 # 1.5 or older


REINVOKE = "__BZR_REINVOKE"    
NEED_VERS = (2, 3)

if version_info < NEED_VERS:
    if not os.environ.has_key(REINVOKE):
        # mutating os.environ doesn't work in old Pythons
        os.putenv(REINVOKE, "1")
        for python in 'python2.4', 'python2.3':
            try:
                os.execvp(python, [python] + sys.argv)
            except OSError:
                pass
    print >>sys.stderr, "bzr-man.py: error: cannot find a suitable python interpreter"
    print >>sys.stderr, "  (need %d.%d or later)" % NEED_VERS
    sys.exit(1)
if hasattr(os, "unsetenv"):
    os.unsetenv(REINVOKE)

import bzrlib, bzrlib.help

#>>> code taken from bzr (C) Canonical

#<<< code by HUN

import time
import re


def man_escape(string):
    result = string.replace("\\","\\\\")
    result = result.replace("`","\\`")
    result = result.replace("'","\\'")
    result = result.replace("-","\\-")
    return result


class Parser:

    def parse_line(self, line):
        pass


class CommandListParser(Parser):

    """Parser for output of "bzr help commands".

    The parsed content can then be used to
    - write a "COMMAND OVERVIEW" section into a man page
    - provide a list of all commands
    """

    def __init__(self,params):
        self.params = params
        self.command_usage = []
        self.all_commands = []
        self.usage_exp = re.compile("([a-z0-9-]+).*")
        self.descr_exp = re.compile("    ([A-Z].*)\s*")
        self.state = 0
        self.command = None
        self.usage = None
        self.descr = None

    def parse_line(self, line):
        m = self.usage_exp.match(line)
	if line == '':
		return
        if m:
            if self.state == 0:
                if self.usage:
                    self.command_usage.append((self.command,self.usage,self.descr))
                    self.all_commands.append(self.command)
                self.usage = " ".join(line.split(" ")[1:])
                self.command = m.groups()[0]
            else:
                raise RuntimeError, "matching usage line in state %d" % state
            self.state = 1
            return
        m = self.descr_exp.match(line)
        if m:
            if self.state == 1:
                self.descr = m.groups()[0]
            else:
                raise RuntimeError, "matching descr line in state %d" % state
            self.state = 0
            return
        raise RuntimeError, "Cannot parse this line ('%s')." % line

    def end_parse(self):
        if self.state == 0:
            if self.usage:
                self.command_usage.append((self.command,self.usage,self.descr))
                self.all_commands.append(self.command)
        else:
            raise RuntimeError, "ending parse in state %d" % state

    def write_to_manpage(self, outfile):
        bzrcmd = self.params["bzrcmd"]
        outfile.write('.SH "COMMAND OVERVIEW"\n')
        for (command,usage,descr) in self.command_usage:
            outfile.write('.TP\n.B "%s %s"\n%s\n\n' % (bzrcmd, usage, descr))


class HelpReader:

    def __init__(self, parser):
        self.parser = parser

    def write(self, data):
        if data[-1] == '\n':
            data = data[:-1]
        for line in data.split('\n'):
            self.parser.parse_line(line)


def write_command_details(params, command, usage, descr, outfile):
    x = ('.SS "%s %s"\n.B "%s"\n.PP\n.B "Usage:"\n%s %s\n\n' %
         (params["bzrcmd"],
          command,
          descr,
          params["bzrcmd"],
          usage))
    outfile.write(man_escape(x))


man_preamble = """\
.\\\" Man page for %(bzrcmd)s (bazaar-ng)
.\\\"
.\\\" Large parts of this file are autogenerated from the output of
.\\\"     \"%(bzrcmd)s help commands\"
.\\\"     \"%(bzrcmd)s help <cmd>\"
.\\\"
.\\\" Generation time: %(timestamp)s
.\\\"
"""

# The DESCRIPTION text was taken from http://www.bazaar-ng.org/
# and is thus (C) Canonical
man_head = """\
.TH bzr 1 "%(datestamp)s" "%(version)s" "bazaar-ng"
.SH "NAME"
%(bzrcmd)s - bazaar-ng next-generation distributed version control
.SH "SYNOPSIS"
.B "%(bzrcmd)s"
.I "command"
[
.I "command_options"
]
.br
.B "%(bzrcmd)s"
.B "help"
.br
.B "%(bzrcmd)s"
.B "help"
.I "command"
.SH "DESCRIPTION"
bazaar-ng (or
.B "%(bzrcmd)s"
) is a project of Canonical to develop an open source distributed version control system that is powerful, friendly, and scalable. Version control means a system that keeps track of previous revisions of software source code or similar information and helps people work on it in teams.
.SS "Warning"
bazaar-ng is at an early stage of development, and the design is still changing from week to week. This man page here may be inconsistent with itself, with other documentation or with the code, and sometimes refer to features that are planned but not yet written. Comments are still very welcome; please send them to bazaar-ng@lists.canonical.com.
"""

man_foot = """\
.SH "ENVIRONMENT"
.TP
.I "BZRPATH"
Path where
.B "%(bzrcmd)s"
is to look for external command.

.TP
.I "BZREMAIL"
E-Mail address of the user. Overrides
.I "~/.bzr.conf/email" and
.IR "EMAIL" .
Example content:
.I "John Doe <john@example.com>"

.TP
.I "EMAIL"
E-Mail address of the user. Overridden by the content of the file
.I "~/.bzr.conf/email"
and of the environment variable
.IR "BZREMAIL" .

.SH "FILES"
.TP
.I "~/.bzr.conf/"
Directory where all the user\'s settings are stored.
.TP
.I "~/.bzr.conf/email"
Stores name and email address of the user. Overrides content of
.I "EMAIL"
environment variable. Example content:
.I "John Doe <john@example.com>"

.SH "SEE ALSO"
.UR http://www.bazaar-ng.org/
.BR http://www.bazaar-ng.org/,
.UR http://www.bazaar-ng.org/doc/
.BR http://www.bazaar-ng.org/doc/
"""

def main():
    t = time.time()
    tt = time.gmtime(t)
    params = \
           { "bzrcmd": "bzr",
             "datestamp": time.strftime("%Y-%m-%d",tt),
             "timestamp": time.strftime("%Y-%m-%d %H:%M:%S +0000",tt),
             "version": bzrlib.__version__,
             }

    clp = CommandListParser(params)
    bzrlib.help.help("commands", outfile=HelpReader(clp))
    clp.end_parse()

    filename = "bzr.1"
    if len(sys.argv) == 2:
        filename = sys.argv[1]
    if filename == "-":
        outfile = sys.stdout
    else:
        outfile = open(filename,"w")

    outfile.write(man_preamble % params)
    outfile.write(man_escape(man_head % params))
    clp.write_to_manpage(outfile)

    # FIXME:
    #   This doesn't do more than the summary so far.
    #outfile.write('.SH "DETAILED COMMAND DESCRIPTION"\n')
    #for (command,usage,descr) in clp.command_usage:
    #    write_command_details(params, command, usage, descr, outfile = outfile)

    outfile.write(man_escape(man_foot % params))


if __name__ == '__main__':
    main()


#>>> code by HUN
