# Written by Henrik Nilsen Omma
# (C) Canonical, Ltd. Licensed under the GPL

import os
import time
from subprocess import Popen, PIPE
from bzrlib.bzrdir import BzrDir

DEFAULT_BRANCH = "http://bazaar.launchpad.net/~bugsquad/bughelper-data/main/"

class BzrError(Exception):
    def __init__(self, error):
        self.value = error
        
    def __str__(self):
        return self.value

def update_packages_data(path, intervall):
    """
    returns stdout and stderr of 'bzr branch' or 'bzr pull'
    """
    result = (None,None)
    if not os.path.exists(path):
        f = Popen(["bzr", "branch", DEFAULT_BRANCH, path], stderr=PIPE, stdout=PIPE)
        sout=f.stdout.readlines()
        eout=f.stderr.readlines()
        if not sout and eout[0].startswith("bzr: ERROR:"):
            raise BzrError, eout[0]
        result = (sout,eout)
    else:
        bi = branch_info(path)
        if bi.timestamp + time.timezone + float(intervall)*60*60 <= time.time():
            old_cwd = os.getcwd()
            os.chdir(path)
            f = Popen(["bzr", "pull", DEFAULT_BRANCH], stderr=PIPE, stdout=PIPE)
            os.chdir(old_cwd)
            result = (f.stdout.readlines(),f.stderr.readlines())
    return result

def update_branch(path, intervall, default_branch, branch_type="checkout"):
    """ update given branch
    
    this is more generic than update_packages_data and uses subprocess,
    will replace update_packages_data at some time
    """
    out, error = None, None
    assert branch_type in ["branch", "checkout"], "branch_type has to be 'branch' or 'checkout'"
    if not os.path.exists(path):
        p = Popen(["bzr", branch_type, default_branch, path], stderr=PIPE, stdout=PIPE)
        out = p.stdout.read()
        error = p.stderr.read()
        if not out and not len(error.split("bzr: ERROR:")) == 1:
            raise BzrError, error
    else:
        bi = branch_info(path)
        if branch_type == "checkout":
            operation = "update"
        else:
            operation = "pull"
        if isinstance(intervall, type(None)):
            intervall = 0
        if bi.timestamp + time.timezone + float(intervall)*60*60 <= time.time():
            p = Popen(["bzr", operation, default_branch], cwd=path, stderr=PIPE, stdout=PIPE)
            out = p.stdout.read()
            error = p.stderr.read()
    return out, error
        


class branch_info(object):
    def __init__(self, path):
        tree = BzrDir.open(path).open_workingtree()
        self.revno = tree.branch.revno()
        last_rev = tree.branch.repository.get_revision(tree.last_revision())
        self.timestamp = last_rev.timestamp

