/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * FILE   : FtpServer.cpp
 * OBJECT : FtpServer
 * INFO   : Dialog umozliwiajacy wprowadzanie/modyfikacje danych
 *          opisujacych polaczenie z konkretnym serwerem S/FTP.
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FtpServer.h"
#include "Shared.h"
#include "Config.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qvalidator.h>

/*------- local constants:
-------------------------------------------------------------------*/
const int     FtpServer::OVER_STRETCH  = 100;
const int     FtpServer::SPACING_WIDTH = 2;
const int     FtpServer::MARGIN_WIDTH  = 2;
const QString FtpServer::ANONYM_NAME   = "anonymous";
const QString FtpServer::ANONYM_PASS   = "bsc@beesoft.org";
//...................................................................
const QString FtpServer::CAPTION       = QT_TR_NOOP( "FTP server data" );
const QString FtpServer::BTN_OK        = QT_TR_NOOP( "&Ok" );
const QString FtpServer::SRV_SETTINGS  = QT_TR_NOOP( "Server settings" );
const QString FtpServer::NAME_LABEL    = QT_TR_NOOP( "&Name:" );
const QString FtpServer::PORT_LABEL    = QT_TR_NOOP( "&Port" );
const QString FtpServer::ADDR_LABEL    = QT_TR_NOOP( "&Address:" );
const QString FtpServer::DIR_LABEL     = QT_TR_NOOP( "&Directory:" );
const QString FtpServer::USR_SETTINGS  = QT_TR_NOOP( "User settings" );
const QString FtpServer::USER_LABEL    = QT_TR_NOOP( "&User name:" );
const QString FtpServer::PASS_LABEL    = QT_TR_NOOP( "&Password:" );
const QString FtpServer::ANONYM_CB     = QT_TR_NOOP( "Anonymou&s user" );
const QString FtpServer::SSL_CB        = QT_TR_NOOP( "&Secure FTP (sftp)" );

//*******************************************************************
// FtpServer                                             CONSTRUCTOR
//*******************************************************************
FtpServer::FtpServer( QWidget* const in_parent, FtpServerInfo& in_fsi )
: QDialog      ( in_parent )
, d_main_layout( new QHBoxLayout( this ))
//.....................................
, d_data_layout( new QVBoxLayout )
, d_data_box   ( new QGroupBox( 2, Qt::Horizontal, tr(SRV_SETTINGS), this ))
, d_name_lbl   ( new QLabel   ( tr(NAME_LABEL), d_data_box ))
, d_name_le    ( new QLineEdit( d_data_box ) )
, d_addr_lbl   ( new QLabel   ( tr(ADDR_LABEL), d_data_box ))
, d_addr_le    ( new QLineEdit( d_data_box ) )
, d_port_lbl   ( new QLabel   ( tr(PORT_LABEL), d_data_box ))
, d_port_le    ( new QLineEdit( d_data_box ) )
, d_dir_lbl    ( new QLabel   ( tr(DIR_LABEL), d_data_box ))
, d_dir_le     ( new QLineEdit( d_data_box ) )
, d_ssl_cb     ( new QCheckBox( tr(SSL_CB), this ))
, d_anonym_cb  ( new QCheckBox( tr(ANONYM_CB), this ))
, d_user_box   ( new QGroupBox( 2, Qt::Horizontal, tr(USR_SETTINGS), this ))
, d_user_lbl   ( new QLabel   ( tr(USER_LABEL), d_user_box ))
, d_user_le    ( new QLineEdit( d_user_box ))
, d_pass_lbl   ( new QLabel   ( tr(PASS_LABEL), d_user_box ))
, d_pass_le    ( new QLineEdit( d_user_box ))
//.....................................
, d_btn_layout ( new QVBoxLayout )
, d_ok_btn     ( new QPushButton( tr(BTN_OK), this ))
, d_cancel_btn ( new QPushButton( tr(Shared::CancelBtnLabel), this ))
//.....................................
, d_validator  ( new QIntValidator( 1, 100000, this ))
, d_fsi        ( in_fsi )
{
	setCaption( tr(CAPTION) );
	setFont( Config::instance()->lfs_default_font()  );
	d_ssl_cb->hide();

	d_main_layout->setSpacing( SPACING_WIDTH );
	d_main_layout->setMargin( MARGIN_WIDTH );
	
	d_name_lbl->setBuddy( d_name_le );
	d_addr_lbl->setBuddy( d_addr_le );
	d_port_lbl->setBuddy( d_port_le );
	d_dir_lbl->setBuddy( d_dir_le );
	d_user_lbl->setBuddy( d_user_le );
	d_pass_lbl->setBuddy( d_pass_le );
	
	d_port_le->setValidator( d_validator );
	d_pass_le->setEchoMode( QLineEdit::Password );
			
	d_data_layout->addWidget( d_data_box );
	d_data_layout->addWidget( d_ssl_cb );
	d_data_layout->addWidget( d_anonym_cb );
	d_data_layout->addWidget( d_user_box );
	d_main_layout->addLayout( d_data_layout );

	d_btn_layout->addStretch( OVER_STRETCH );
	d_btn_layout->addWidget( d_ok_btn );
	d_btn_layout->addWidget( d_cancel_btn );
	d_main_layout->addLayout( d_btn_layout );
	
	d_main_layout->setStretchFactor( d_data_layout, OVER_STRETCH );

	connect( d_ok_btn    , SIGNAL( clicked()     ), this, SLOT( slot_ok()         ));
	connect( d_cancel_btn, SIGNAL( clicked()     ), this, SLOT( slot_cancel()     ));
	connect( d_anonym_cb , SIGNAL( toggled(bool) ), this, SLOT( slot_anonym(bool) ));
	connect( d_ssl_cb    , SIGNAL( toggled(bool) ), this, SLOT( slot_ssl(bool)    ));
}
// end of FtpServer

//*******************************************************************
// ~FtpServer                                             DESTRUCTOR
//*******************************************************************
FtpServer::~FtpServer()
{
	delete d_validator;
	d_validator = 0;
}
// end of ~FtpServer

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void FtpServer::show()
{
	d_name_le->setText( d_fsi.name() );
	d_addr_le->setText( d_fsi.addr() );
	d_port_le->setText( QString::number( d_fsi.port() ));
	d_dir_le->setText( d_fsi.dir() );
	d_user_le->setText( d_fsi.user() );
	d_pass_le->setText( d_fsi.pass() );

	if( ANONYM_NAME == d_fsi.user() ) {
		d_anonym_cb->setChecked( TRUE );
	}
	d_ssl_cb->setChecked( d_fsi.ssl() );
	QDialog::show();
}
// end of polish

//*******************************************************************
// check                                                     PRIVATE
//-------------------------------------------------------------------
// Kontrola czy wprowadzono wszystkie dane.
//*******************************************************************
bool FtpServer::check()
{
	bool retval = TRUE;
	
	if( d_name_le->text().isEmpty() ) {
		retval = FALSE;
		d_name_le->setFocus();
	}
	else if( d_addr_le->text().isEmpty() ) {
		retval = FALSE;
		d_addr_le->setFocus();
	}
	else if( FALSE == d_ssl_cb->isChecked() ) {
		if( d_port_le->text().isEmpty() ) {
			retval = FALSE;
			d_port_le->setText( "21" );
			d_port_le->setFocus();
		}
		else if( d_dir_le->text().isEmpty() ) {
			retval = FALSE;
			d_dir_le->setText( "/" );
			d_dir_le->setFocus();
		}
	}
	else if( d_user_le->text().isEmpty() ) {
		retval = FALSE;
		d_user_le->setFocus();
	}
	else if( d_pass_le->text().isEmpty() ) {
		retval = FALSE;
		d_pass_le->setFocus();
	}

	return retval;
}
// end of check

//*******************************************************************
// slot_ok                                              PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik zaakceptowal aktualnie wprowadzone dane.
// Wracamy do listy serwerow.
//*******************************************************************
void FtpServer::slot_ok()
{
	if( check() ) {
		d_fsi.set_name( d_name_le->text() );
		d_fsi.set_ssl ( d_ssl_cb->isChecked() );
		d_fsi.set_addr( d_addr_le->text() );
		d_fsi.set_port( d_port_le->text().toInt() );
		d_fsi.set_dir ( d_dir_le->text()  );
		d_fsi.set_user( d_user_le->text() );
		d_fsi.set_pass( d_pass_le->text() );
		accept();
	}
}
// end of slot_ok

//*******************************************************************
// slot_cancel                                          PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik zrezygnowal ze wszystkich zmian danych.
//*******************************************************************
void FtpServer::slot_cancel()
{
	d_fsi.clear();
	reject();
}
// end of slot_cancel

//*******************************************************************
// slot_anonym                                          PRIVATE slot
//*******************************************************************
void FtpServer::slot_anonym( bool in_on )
{
	if( in_on ) {
		d_user_le->setEnabled( FALSE );
		d_pass_le->setEnabled( FALSE );
		d_user_le->setText( ANONYM_NAME );
		d_pass_le->setText( ANONYM_PASS );
	}
	else {
		d_user_le->setEnabled( TRUE );
		d_pass_le->setEnabled( TRUE );
		d_user_le->clear();
		d_pass_le->clear();		
	}
}
// end of slot_anonym

//*******************************************************************
// slot_ssl                                             PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik zdecydowal, ze chce skorzystac z SFTP.
//*******************************************************************
void FtpServer::slot_ssl( bool in_on )
{
	if( in_on ) {
		d_anonym_cb->setEnabled( FALSE );
		d_port_le->setEnabled( FALSE );
		d_dir_le->setEnabled( FALSE );
		d_user_le->setEnabled( TRUE );
		d_pass_le->setEnabled( TRUE );
		d_user_le->clear();
		d_pass_le->clear();
	}
	else {
		d_anonym_cb->setEnabled( TRUE );
		d_port_le->setEnabled( TRUE );
		d_dir_le->setEnabled( TRUE );
	}
}
// end of slot_anonym
