/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
#ifndef INCLUDED_CMPDIRS_H
#define INCLUDED_CMPDIRS_H

/*------- include files:
-------------------------------------------------------------------*/
#ifndef INCLUDED_SHARED_H
#include "Shared.h"
#endif // INCLUDED_SHARED_H

#ifndef INCLUDED_INFOFIELD_H
#include "InfoField.h"
#endif // INCLUDED_INFOFIELD_H

#ifndef INCLUDED_QLISTVIEW_H
#include <qlistview.h>
#define INCLUDED_QLISTVIEW_H
#endif // INCLUDED_QLISTVIEW_H

#ifndef INCLUDED_QHEADER_H
#include <qheader.h>
#define INCLUDED_QHEADER_H
#endif // INCLUDED_QHEADER_H

#ifndef INCLUDED_QDIALOG_H
#include <qdialog.h>
#define INCLUDED_QDIALOG_H
#endif // INCLUDED_QDIALOG_H

#ifndef INCLUDED_QDIR_H
#include <qdir.h>
#define INCLUDED_QDIR_H
#endif // INCLUDED_QDIR_H

#ifndef INCLUDED_MAP
#include <map>
#define INCLUDED_MAP
#endif // INCLUDED_MAP


/*------- forward declarations:
-------------------------------------------------------------------*/
class QPushButton;
class QListView;
class QFileInfo;

/*------- class declaration:
-------------------------------------------------------------------*/

class CmpDirs : public QDialog
{
	Q_OBJECT

// ******* TYPES *******
private:
	enum { ColsNumber = 3 };
	typedef std::map<const QString, const QFileInfo* const , std::greater<const QString> > DataMap;
	class ViewItem : public QListViewItem
	{
		public:
			ViewItem(	QListView* const,
							const QString&, const QString&, const QString&,
							const QString&, const QString& );
		private:
			ViewItem( const ViewItem& );
			ViewItem& operator=( const ViewItem& );
		private:
			const QString d_lft_path;
			const QString d_rgt_path;
		private:
			int  compare  ( QListViewItem* in_item, int in_column, bool in_ascending ) const;
			void paintCell( QPainter*, const QColorGroup&, int, int, int );
		public:
			const QString& lft_path() const;
			const QString& rgt_path() const;
	};

// ******* CONSTRUCTION *******
public:
	CmpDirs( QWidget* const, const QString&, const QString& );
private:
	CmpDirs( const CmpDirs& );
	CmpDirs& operator=( const CmpDirs );

// ******* CONSTANTS *******
private:
	static const QString Caption;
	static const QString InfoCaption;
	static const QString InfoMsg;
	static const QString GboxDirsCaption;
	static const QString GboxTblCaption;
	static const QString LeftDirLabel;
	static const QString RightDirLabel;
	static const QString ViewHdrCaption[ ColsNumber ];
	static const QString FileNotPresents;
	static const QString Equal;
	static const QString NotEqual;
	static const QString Error;
	static const QString Dir;

// ******* MEMBERS *******
private:
	const QString      d_src_dir;
	const QString      d_dst_dir;
	QPushButton* const d_run_btn;
	QPushButton* const d_close_btn;
	InfoField*         d_src_dir_path;
	InfoField*         d_dst_dir_path;
	QListView*         d_view;
	DataMap            d_lft_files;
	DataMap            d_rgt_files;
	QString            d_lft_name;
	QString            d_rgt_name;
	QString            d_lft_path;
	QString            d_rgt_path;

// ******* METHODS *******
private:
	void polish();
	void showEvent     ( QShowEvent* );
	bool compare       ( const QString&, const QString& );
	void files         ( const QFileInfoList* const, DataMap& );
	void dirs          ( const QFileInfoList* const, DataMap& );
	void compare_files ();

private slots:
	void slot_run      ();
	void slot_close    ();
	void info          ( QListViewItem*, const QPoint& );
};

//*******************************************************************
// ViewItem::ViewItem
//*******************************************************************
inline CmpDirs::ViewItem::ViewItem( QListView* const parent,
	const QString& label1, const QString& label2, const QString& label3,
	const QString& in_lft_path, const QString& in_rgt_path )
: QListViewItem( parent, label1, label2, label3 )
, d_lft_path( in_lft_path )
, d_rgt_path( in_rgt_path )
{}
// end of ViewItem::ViewItem

//*******************************************************************
// paintCell                                       PRIVATE inherited
//*******************************************************************
inline void CmpDirs::ViewItem::paintCell( QPainter* p, const QColorGroup& cg, int col, int w, int align )
{
	QColorGroup new_cg = cg;
	
	if( d_lft_path && d_rgt_path ) {
		if( NotEqual == text( 1 ) ){
			new_cg.setColor( QColorGroup::Base, QColor( 200, 200, 255 ) );
		}
	}
	else {
		new_cg.setColor( QColorGroup::Base, QColor( 255, 200, 200 ) );
	}

	QListViewItem::paintCell( p, new_cg, col, w, align );
}
// end of paintCell

//*******************************************************************
// compare                                         PRIVATE inherited
//*******************************************************************
inline int CmpDirs::ViewItem::compare( QListViewItem* in_item, int in_column, bool ) const
{
	bool retval = 0;
	const ViewItem* const item = dynamic_cast<const ViewItem*>( in_item );
	return lft_path().localeAwareCompare( item->lft_path() );
		
	if( 0 == in_column ) {
		QString my_dir, my_name;
		Shared::clip_path( lft_path(), my_dir, my_name );
		QString another_dir, another_name;
		Shared::clip_path( item->lft_path(), another_dir, another_name );
	
		if( my_dir == another_dir ) retval = my_name.localeAwareCompare( another_name );
		else                        retval = my_dir.localeAwareCompare ( another_dir  );
	}
	
	return retval;
}
// end of compare

//*******************************************************************
// ViewItem::lft_path
//*******************************************************************
inline const QString& CmpDirs::ViewItem::lft_path() const
{
	return d_lft_path;
}
// end of ViewItem::lft_path

//*******************************************************************
// ViewItem::rgt_path
//*******************************************************************
inline const QString& CmpDirs::ViewItem::rgt_path() const
{
	return d_rgt_path;
}
// end of ViewItem::path

#endif // INCLUDED_CMP_DIRS
