/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "DisplayConfig.h"
#include "Settings.h"
#include "Config.h"
#include <qfont.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qfontdialog.h>
#include <qcolordialog.h>
#include <qdatastream.h>
#include <qcstring.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qcheckbox.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString DisplayConfig::FontIcon        = "fonts.png";
const QString DisplayConfig::ColorIcon       = "colors.png";
//...................................................................
const QString DisplayConfig::FontBtnLabel    = QT_TR_NOOP( "&Font" );
const QString DisplayConfig::ColorBtnLabel   = QT_TR_NOOP( "&Color" );
const QString DisplayConfig::LfsLabel        = QT_TR_NOOP( "&Local file system" );
const QString DisplayConfig::FtpLabel        = QT_TR_NOOP( "&FTP client" );


//*******************************************************************
// DisplayConfig                                         CONSTRUCTOR
//*******************************************************************
DisplayConfig::DisplayConfig( QWidget* const in_parent )
: QFrame( in_parent )
, d_main_layout( new QHBoxLayout( this ))
, d_wrk_layout ( new QVBoxLayout )
, d_btn_layout ( new QVBoxLayout )
, d_view       ( new QListView( this ))
, d_cols_grpbox( new QGroupBox( 2, Qt::Vertical, "Columns", this ))
, d_resize_cbox( new QCheckBox( "&disable automatic resizing of column width", d_cols_grpbox ))
, d_font_btn   ( new QPushButton( tr(FontBtnLabel), this ))
, d_color_btn  ( new QPushButton( tr(ColorBtnLabel), this ))
, d_reset_btn  ( new QPushButton( tr(Shared::ResetBtnLabel), this ))
, d_apply_btn  ( new QPushButton( tr(Shared::ApplyBtnLabel), this ))
, d_lfs_rb     ( 0 )
, d_ftp_rb     ( 0 )
, d_selected   ( LfsIdx )
{
	Shared::add_icon( d_font_btn, FontIcon );
	Shared::add_icon( d_color_btn, ColorIcon );
	Shared::add_icon( d_reset_btn, Shared::ResetIcon );
	Shared::add_icon( d_apply_btn, Shared::ApplyIcon );
	
	d_view->setAllColumnsShowFocus( TRUE );
	d_view->addColumn( tr("Column 1") );
	d_view->addColumn( tr("Column 2") );
	d_view->addColumn( tr("Column 3") );
	d_view->setColumnAlignment( 0, Qt::AlignLeft );
	d_view->setColumnAlignment( 1, Qt::AlignRight );
	d_view->setColumnAlignment( 2, Qt::AlignHCenter );
	new QListViewItem( d_view, "ABCDEABCDE",    "111", "11:11:11" );
	new QListViewItem( d_view, "FGHIJFGHIJ",  "2,222", "22:22:22" );
	new QListViewItem( d_view, "KLMNOKLMNO", "33,333", "33:33:33" );

	d_main_layout->setSpacing( Shared::LayoutSpacing );
	d_main_layout->setMargin( Shared::LayoutMargin );
	
	// work
	d_wrk_layout->addWidget( d_view );
	d_wrk_layout->addWidget( d_cols_grpbox );
	d_main_layout->addLayout( d_wrk_layout );
	
	// radio buttons & push buttons
	QButtonGroup* const gbox = new QButtonGroup( 2, Qt::Vertical, this );
	gbox->setRadioButtonExclusive( TRUE );
	gbox->insert( d_lfs_rb = new QRadioButton( tr(LfsLabel), gbox ), LfsIdx );
	gbox->insert( d_ftp_rb = new QRadioButton( tr(FtpLabel), gbox ), FtpIdx );
	d_btn_layout->addWidget( gbox );

	d_btn_layout->addStretch( Shared::OverStretch );
	d_btn_layout->addWidget( d_font_btn );
	d_btn_layout->addWidget( d_color_btn );
	d_btn_layout->addStretch( 20 );
	d_btn_layout->addWidget( d_reset_btn );
	d_btn_layout->addWidget( d_apply_btn );
	d_btn_layout->addStretch( 20 );
	d_main_layout->addLayout( d_btn_layout );
	
	
	connect( d_font_btn , SIGNAL( clicked()      ), this, SLOT( slot_font()         ));
	connect( d_color_btn, SIGNAL( clicked()      ), this, SLOT( slot_color()        ));
	connect( d_reset_btn, SIGNAL( clicked()      ), this, SLOT( slot_reset()        ));
	connect( d_apply_btn, SIGNAL( clicked()      ), this, SLOT( slot_apply()        ));
	connect( gbox       , SIGNAL( clicked( int ) ), this, SLOT( slot_clicked( int ) ));
	connect( this       , SIGNAL( looks_refresh()), in_parent, SIGNAL( looks_refresh() ));
	
	gbox->setButton( d_selected );
	read_config();
}
// end of DisplayConfig

//*******************************************************************
// read_config                                          PRIVATE slot
//*******************************************************************
void DisplayConfig::read_config()
{
	Config::instance()->refresh();
	
	d_lfs_org_font  = d_lfs_selected_font  = Config::instance()->lfs_font();
	d_ftp_org_font  = d_ftp_selected_font  = Config::instance()->ftp_font();
	d_lfs_org_color = d_lfs_selected_color = Config::instance()->lfs_bkg_color();;
	d_ftp_org_color = d_ftp_selected_color = Config::instance()->ftp_bkg_color();

	update_view();
}
// end of read_config

//*******************************************************************
// update_view                                               PRIVATE
//*******************************************************************
void DisplayConfig::update_view()
{
	switch( d_selected ) {
		case LfsIdx:
			d_view->setPaletteBackgroundColor( d_lfs_selected_color );
			d_view->setFont( d_lfs_selected_font );
			break;
		case FtpIdx:
			d_view->setPaletteBackgroundColor( d_ftp_selected_color );
			d_view->setFont( d_ftp_selected_font );
			break;
	}
	for( int i = 0; i < 3; ++i ) {
		d_view->adjustColumn( i );
	}
	d_resize_cbox->setChecked( ( 1 == Config::instance()->disable_columns_resize() ) ? TRUE : FALSE );
}
// end of update_view

//*******************************************************************
// slot_font                                            PRIVATE slot
//*******************************************************************
void DisplayConfig::slot_font()
{
	const QFont current_font = ( LfsIdx == d_selected )
										? d_lfs_selected_font
										: d_ftp_selected_font;
	bool ok;
	const QFont new_font = QFontDialog::getFont( &ok, current_font, this );
	if ( ok ) {
		if( new_font != current_font ) {
			if( LfsIdx == d_selected ) d_lfs_selected_font = new_font;
			else                       d_ftp_selected_font = new_font;
			update_view();
		}
	}
}
// end of slot_font

//*******************************************************************
// slot_color                                           PRIVATE slot
//*******************************************************************
void DisplayConfig::slot_color()
{
	const QColor current_color = ( LfsIdx == d_selected )
											? d_lfs_selected_color
											: d_ftp_selected_color;
	
	QColor new_color = QColorDialog::getColor( current_color, this );
	if( new_color != current_color ) {
		if( LfsIdx == d_selected ) d_lfs_selected_color = new_color;
		else                       d_ftp_selected_color = new_color;
		update_view();
	}
}
// end of slot_color

//*******************************************************************
// slot_reset                                           PRIVATE slot
//*******************************************************************
void DisplayConfig::slot_reset()
{
	d_lfs_selected_font  = d_lfs_org_font;
	d_ftp_selected_font  = d_ftp_org_font;
	d_lfs_selected_color = d_lfs_org_color;
	d_ftp_selected_color = d_ftp_org_color;
	update_view();
}
// end of slot_reset

//*******************************************************************
// slot_apply                                           PRIVATE slot
//*******************************************************************
void DisplayConfig::slot_apply()
{
	Settings::save( Config::PanelConfigGroupName, Config::LFS_FontKey     , d_lfs_selected_font.toString() );
	Settings::save( Config::PanelConfigGroupName, Config::FTP_FontKey     , d_ftp_selected_font.toString() );
	Settings::save( Config::PanelConfigGroupName, Config::LFS_BkgColorKey , d_lfs_selected_color.name() );
	Settings::save( Config::PanelConfigGroupName, Config::FTP_BkgColorKey , d_ftp_selected_color.name() );
	Settings::save( Config::PanelConfigGroupName, Config::ColumnsResizeKey, d_resize_cbox->isChecked() ? 1 : 0 );
	
	read_config();
	emit looks_refresh();
}
// end of slot_apply

//*******************************************************************
// slot_clicked                                        PRIVATE slot
//*******************************************************************
void DisplayConfig::slot_clicked( const int in_idx )
{
	d_selected = in_idx;
	update_view();
}
// end of slot_clicked
