
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This will be a rotary potmeter. Takes a bitmap and rotates it according to
 * input from the mouse/keyboard. We need a few different parameters, and a
 * hefty include file. Where possible will try and keep X11 requests in a
 * separate set of files.
 */

#include <math.h>

#include "brightoninternals.h"
//#include "brightonX11.h"

int
destroyRotary(brightonDevice *dev)
{
	printf("destroyRotary()\n");

	if (dev->image)
		brightonFreeBitmap(dev->bwin, dev->image);
	dev->image = NULL;
}

static int
displayrotary(brightonDevice *dev)
{
	if (dev->bwin->app->resources[dev->panel].flags & BRIGHTON_WITHDRAWN)
		return;

	/*
	 * Build up a smooth position for the pot. We may need to adjust this based
	 * on the to/from values.
	 */
	dev->position = dev->value < 0.5?
		7 * M_PI * (1 - 2 * dev->value) / 8:
		M_PI * (23 - dev->value * 14) / 8;

	/*
	 * Only draw fixed number of steps.
	 */
	if ((int) (dev->value * 360 / M_PI) != (int) (dev->lastvalue * 360 / M_PI))
	{
		brightonIResource *panel;

		panel = &dev->bwin->app->resources[dev->panel];

		/*
		 * Rotate my image onto the parents canvas.
		 */
		brightonRotate(dev->bwin, dev->image,
			//dev->bwin->app->resources[dev->panel].canvas,
			dev->bwin->dlayer,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->bwin->app->resources[dev->panel].sy,
			dev->width, dev->height,
			dev->position);

		/*
		 * And request the panel to put this onto the respective image.
		 */
		brightonFinalRender(dev->bwin,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->bwin->app->resources[dev->panel].sy,
			dev->width, dev->height);
	}

	dev->lastvalue = dev->value;
	dev->lastposition = dev->position;
}

/*
 * This will go into brighton render
 */
static int
renderHighlights(brightonWindow *bwin, brightonDevice *dev)
{
	float d, ho2, streak;
	float ox, oy, dx, dy;

	if (dev->shadow.coords == 0)
		dev->shadow.coords = brightonmalloc(7 * sizeof(brightonCoord));
	dev->shadow.ccount = 7;
	dev->shadow.flags = BRIGHTON_STATIC;

	ho2 = dev->width / 2;
	ox = dev->x + ho2;
	oy = dev->y + ho2;

	/*
	 * We are going to render the shadow directly onto the background bitmap.
	 * We have X and Y for the source of the shadow, plus its height and 
	 * intensity. For now we will take a default relief, and highlight the
	 * background accordingly. This should be a 3D transform.....
	 *
	 * This can all be done with fractional distances, since we are going to
	 * be dealing with a number of similar triangles.
	 */
	dx = ox - bwin->lightX;
	dy = oy - bwin->lightY;
	d = sqrt((double) (dx * dx + dy * dy));

	dev->shadow.coords[0].x = ox + dy * ho2 / d;
	dev->shadow.coords[0].y = oy - dx * ho2 / d;
	dev->shadow.coords[1].x = ox - dy * ho2 / d;
	dev->shadow.coords[1].y = oy + dx * ho2 / d;

	streak = (dev->width * 1.6 * d / bwin->lightH)
		/ (1 - dev->width * 1.6 / bwin->lightH);

	dev->shadow.coords[2].x = dev->shadow.coords[1].x
		+ streak * dx / d + dy * ho2 * 0.4 / d;
	dev->shadow.coords[2].y = dev->shadow.coords[1].y
		+ streak * dy / d - dx * ho2 * 0.4 / d;

	dev->shadow.coords[6].x = dev->shadow.coords[0].x
		+ streak * dx / d - dy * ho2 * 0.4 / d;
	dev->shadow.coords[6].y = dev->shadow.coords[0].y
		+ streak * dy / d + dx * ho2 * 0.4 / d;

	streak = (dev->width * 2.07 * d / bwin->lightH)
		/ (1 - dev->width * 2.07 / bwin->lightH);

	dev->shadow.coords[3].x = ox + (streak * dx / d) - dy * ho2 * 0.5 / d;
	dev->shadow.coords[3].y = oy + (streak * dy / d) + dx * ho2 * 0.5 / d;
	dev->shadow.coords[5].x = ox + (streak * dx / d) + dy * ho2 * 0.5 / d;
	dev->shadow.coords[5].y = oy + (streak * dy / d) - dx * ho2 * 0.5 / d;

	streak = (dev->width * 2.2 * d / bwin->lightH)
		/ (1 - dev->width * 2.2 / bwin->lightH);

	dev->shadow.coords[4].x = ox + (streak * dx / d);
	dev->shadow.coords[4].y = oy + (streak * dy / d);

	/*
	printf("renderHighlights(%i) %i,%i-%i,%i-%i,%i-%i,%i-%i,%i-%i,%i-%i,%i\n",
		dev->index, dev->shadow.coords[0].x,dev->shadow.coords[0].y,
		dev->shadow.coords[1].x,dev->shadow.coords[1].y,
		dev->shadow.coords[2].x,dev->shadow.coords[2].y,
		dev->shadow.coords[3].x,dev->shadow.coords[3].y,
		dev->shadow.coords[4].x,dev->shadow.coords[4].y,
		dev->shadow.coords[5].x,dev->shadow.coords[5].y,
		dev->shadow.coords[6].x,dev->shadow.coords[6].y);
	 */

	dev->shadow.ccount = 7;
	/*
	 * rather than fill the polygon, we need to generate this shape, and lower
	 * the shading of pixels that are within this area.
	brightonRenderShadow(dev, 0);
	 */
}

static int considercallback(brightonDevice *dev)
{
	brightonIResource *panel = &dev->bwin->app->resources[dev->panel];

	if (dev->value < 0)
		dev->value = 0;

	/*
	 * We now need to consider rounding this to the resolution of this
	 * device. If the max value is not 1.0 then we need to put fixed steps
	 * into our new device value.
	 */
	if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].to != 1.0)
	{
		dev->value =
			(dev->value
			* dev->bwin->app->resources[dev->panel].devlocn[dev->index].to);

		if ((dev->value - ((int) dev->value)) > 0.5)
			dev->value = ((float) ((int) dev->value) + 1)
				/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;
		else
			dev->value = ((float) ((int) dev->value))
				/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;
	}

	if (dev->lastvalue != dev->value)
	{
		if (panel->devlocn[dev->index].callback)
		{
			panel->devlocn[dev->index].callback(dev->bwin,
				dev->panel, dev->index,
				dev->value * panel->devlocn[dev->index].to);
		} else if (panel->callback) {
			panel->callback(dev->bwin, dev->panel, dev->index,
				dev->value * panel->devlocn[dev->index].to);
		}
	}
}

static int
configure(brightonDevice *dev, brightonEvent *event)
{
//	printf("configureRotary(%i)\n", event->command);

	if (event->command == -1)
		return(-1);

	if (event->command == BRIGHTON_RESIZE)
	{
		dev->originx = event->x;
		dev->originy = event->y;

			/*
		if (event->w < event->h)
		{
			 * If width is less than height, then we need to configure
			 * some offsets. Also, we only want even number of pixel areas.
			dev->x = event->x;
			dev->y = event->y + ((event->h - event->w) / 2);
			dev->width = event->w & ~0x01;
			dev->height = event->w & ~0x01;
		} else if (event->w > event->h) {
			dev->x = event->x + ((event->w - event->h) / 2);
			dev->y = event->y;
			dev->width = event->h & ~0x01;
			dev->height = event->h & ~0x01;
		} else {
			dev->x = event->x;
			dev->y = event->y;
			dev->width = event->w & ~0x01;
			dev->height = event->h & ~0x01;
		}
			 */
		if (event->w < event->h)
		{
			/*
			 * If width is less than height, then we need to configure
			 * some offsets. Also, we only want even number of pixel areas.
			 */
			dev->x = event->x;
			dev->y = event->y;
			dev->width = event->w & ~0x01;
			dev->height = event->w & ~0x01;
		} else if (event->w > event->h) {
			dev->x = event->x;
			dev->y = event->y;
			dev->width = event->h & ~0x01;
			dev->height = event->h & ~0x01;
		} else {
			dev->x = event->x;
			dev->y = event->y;
			dev->width = event->w & ~0x01;
			dev->height = event->h & ~0x01;
		}

		/*
		 * We should now rework our parent understanding of our window, since
		 * it will have altered.
		 */
		brightonPanelLocation(dev->bwin,
			dev->panel, dev->index, dev->x, dev->y, dev->width, dev->height);

		considercallback(dev);

		/*
		 * We need to build in some shadow, to prevent the rotary from looking
		 * like it is hanging in mid air.
		 */
		brightonRenderShadow(dev, 0);

		dev->lastvalue = -1;
		displayrotary(dev);

		return(0);
	}

	if (event->command == BRIGHTON_KEYRELEASE)
	{
		switch(event->key) {
			default:
				break;
			case 50:
			case 62:
				dev->flags &= ~BRIGHTON_SHIFTKEY;
				break;
		}
	}

	if (event->command == BRIGHTON_KEYPRESS)
	{
		switch(event->key) {
			default:
				break;
			case 50:
			case 62:
				dev->flags |= BRIGHTON_SHIFTKEY;
				break;
			case 45:
			case 98:
				if (dev->flags & BRIGHTON_SHIFTKEY)
					dev->value += ((float) 10) / 16384;
				else
					dev->value += ((float) 1) / 16384;
				break;
			case 44:
			case 104:
				if (dev->flags & BRIGHTON_SHIFTKEY)
					dev->value -= ((float) 10) / 16384;
				else
					dev->value -= ((float) 1) / 16384;
				break;
		}

		considercallback(dev);

		displayrotary(dev);
	}

	if (event->command == BRIGHTON_MOTION)
	{
		double angle, diffx, diffy;

		diffx = event->x - (dev->width / 2 + dev->x);
		diffy = event->y - (dev->height / 2 + dev->y);
		angle = atan(diffy / diffx);

		/*
		 * Adjust this so that we get counterclock rotating angles from mid top.
		 */
		if (((diffx < 0) && (diffy < 0)) || ((diffx < 0) && (diffy >= 0)))
			angle = M_PI / 2 - angle;
		else
			angle = 3 * M_PI / 2 - angle;

		/*
		 * To correct to clock rotating angles, with stop points.
		 */
		if (angle < M_PI)
			dev->value = (7 * M_PI / 8 - angle) * 4 / (M_PI * 7);
		else
			dev->value = (23 * M_PI / 8 - angle) * 4 / (M_PI * 7);

		if (dev->value < 0)
			dev->value = 0;
		else if (dev->value > 1.0)
			dev->value = 1.0;

		considercallback(dev);

		displayrotary(dev);

		return(0);
	}

	if (event->command == BRIGHTON_PARAMCHANGE)
	{
		dev->value = event->value
			/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;

		considercallback(dev);

		displayrotary(dev);

		return(0);
	}
}

int *
createRotary(brightonWindow *bwin, brightonDevice *dev, int index, char *bitmap)
{
	//printf("createRotary(%s)\n", bitmap);

	dev->destroy = destroyRotary;
	dev->configure = configure;
	dev->bwin = bwin;
	dev->index = index;

	if (bitmap == 0)
	{
		if (dev->image)
			brightonFreeBitmap(bwin, dev->image);
		/*
		 * Open the default bitmap
		 */
		if (bwin->app->resources[dev->panel].devlocn[dev->index].image != 0)
			dev->image =
				bwin->app->resources[dev->panel].devlocn[dev->index].image;
		else
			dev->image = brightonReadImage(bwin, "bitmaps/knobs/knob.xpm");
	} else {
		if (dev->image)
			brightonFreeBitmap(bwin, dev->image);
		dev->image = brightonReadImage(bwin, bitmap);
	}

	/*
	 * These will force an update when we first display ourselves.
	 */
	dev->value = 0;
	dev->lastvalue = -1;
	dev->lastposition = -1;

	return(0);
}

