
/*
 *  Diverse Bristol midi routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This started out primarily to load arrays of values, they being used for
 * whatever purposes required by the caller. The midi interface uses them to
 * define the gain parameters for velocity and any other control settings.
 * The Hammond preacher algorithm uses them to load the the tonewheel settings,
 * not really a midi function however the code is reasonably general.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "bristolmidiapi.h"
#include "bristolmidieventnames.h"

#define NO_INTERPOLATE 0x01

static char *myHome = NULL;

/*
 * Search the file path given below to look for the desired file. Return the
 * base to that file: the bristol base irrespective of where we find the file
 * in our tree search. For two of the locations we should consider building 
 * a subtree infrastructure to allow the user to generate their own file 
 * copies.
 *
 * There are a couple of issues due to the tree structure. This might be a 
 * memory request or a profile request. One needs a valid tree the other needs
 * a valid file.
 */
char *
getBristolCache(char *file)
{
	struct stat statbuf;
	char *envcache;
	char path[1024];

	/*
	 * Find the file specified in the request. Use the search order given
	 * below. Create a local cache in either $BRISTOL_CACHE or $HOME if its
	 * tree does not exist since this will be used for our private memories.
	 *
	 * Search in the following order:
	 *
	 * 	$BRISTOL_CACHE/memory/profiles/<file> - return BRISTOL_CACHE
	 * 	$BRISTOL_CACHE/memory/<file> - return BRISTOL_CACHE
	 *	$HOME/.bristol/memory/profiles/<file> - return $HOME/.bristol
	 * 	$HOME/.bristol/memory/<file> - return $HOME/.bristol
	 *	$BRISTOL/memory/<file> - these should never fail
	 *	$BRISTOL/memory/profiles - return $BRISTOL
	 */
	if (myHome == NULL)
		myHome = (char *) calloc(1024, 1);

	/*
	 * See if we have an env configured:
	 *	if so see if it exists, create a tree if necessary.
	 */
	if ((envcache = getenv("BRISTOL_CACHE")) != NULL)
	{
		sprintf(path, "%s/memory/profiles/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			/*
			 * Path exists.
			 */
			sprintf(myHome, "%s", envcache);
			/* printf("opened BRISTOL_CACHE: %s\n", path); */
			return(myHome);
		} else {
			/*
			 * Ok, path was desired but not available, build the tree if we
			 * can and then carry on searching.
			 */
			sprintf(path, "%s", envcache);
			mkdir(path, 0755);
			sprintf(path, "%s/memory", envcache);
			mkdir(path, 0755);
			sprintf(path, "%s/memory/profiles", envcache);
			mkdir(path, 0755);
		}

		sprintf(path, "%s/memory/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			/*
			 * Path exists.
			 */
			sprintf(myHome, "%s", envcache);
			/* printf("opened BRISTOL_CACHE: %s\n", path); */
			return(myHome);
		}

		printf("create cache %s\n", path);
		mkdir(path, 0755);
	}

	/*
	 * So, no cache or the file did not exist. See about our default cache
	 * $HOME/.bristol
	 */
	if ((envcache = getenv("HOME")) != NULL)
	{
		sprintf(path, "%s/.bristol/memory/profiles/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			/*
			 * Path exists.
			 */
			sprintf(myHome, "%s/.bristol", envcache);
			/* printf("opened private profile cache: %s\n", path); */
			return(myHome);
		} else {
			/*
			 * Ok, path was desired but not available, build the tree if we
			 * can and then carry on searching.
			 */
			sprintf(path, "%s", envcache);
			mkdir(path, 0755);
			sprintf(path, "%s/memory", envcache);
			mkdir(path, 0755);
			sprintf(path, "%s/memory/profiles", envcache);
			mkdir(path, 0755);
		}

		sprintf(path, "%s/.bristol/memory/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			/*
			 * Path exists.
			 */
			sprintf(myHome, "%s/.bristol", envcache);
			/* printf("opened private memory cache: %s\n", path); */
			return(myHome);
		}
	}

	if ((envcache = getenv("BRISTOL")) != NULL)
	{
		sprintf(path, "%s/memory/profiles/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			sprintf(myHome, "%s", envcache);
			/* printf("opened factory cache: %s\n", path); */
			return(myHome);
		}

		sprintf(path, "%s/memory/%s", envcache, file);
		if (stat(path, &statbuf) == 0)
		{
			sprintf(myHome, "%s", envcache);
			/* printf("opened factory cache: %s\n", path); */
			return(myHome);
		}
	}

	return(myHome);
}

int
bristolGetMap(char *file, char *match, float *points, int count, int flags)
{
	int i, n = 0, mapped = 0;
	FILE *fd;
	float from, delta;
	char path[256];

	/*
	 * Open and read configuration. Should consider seaching
	 * $HOME/.bristol/memory and $BRISTOL_DB/memory.
	 */
	sprintf(path, "%s/memory/profiles/%s", getBristolCache("profiles"), file);
//printf("		%s\n", path);

	/*
	 * If we can open the file then clean the array and start again. If not
	 * then see if we have a factory copy available.
	 */
	if ((fd = fopen(path, "r")) == NULL)
	{
		sprintf(path, "%s/memory/profiles/%s", getenv("BRISTOL"), file);
//printf("attempt	%s\n", path);
		fd = fopen(path, "r");
	}

	if (fd != NULL)
	{
		char param[256];

		for (i = 0; i < count; i++)
			points[i] = 0;

		while (fgets(param, 256, fd) != NULL)
		{
			if (param[0] == '#')
				continue;
			if (strlen(param) < 5)
				continue;

			if (strncmp(param, match, strlen(match)) == 0)
			{
				int wheel;
				float value;
				char *offset;

				/*
				 * We should now have a line with something like:
				 * <match>: wheel value
				if ((offset = strpbrk(param, " 	")) == NULL)
				 */
				if ((offset = index(param, ' ')) == NULL)
					continue;

				if ((wheel = atoi(offset)) < 0)
					continue;
				if (wheel >= count)
					continue;

				/*if ((offset = strpbrk(param, " 	")) == NULL) */
				if ((offset = index(++offset, ' ')) == NULL)
					continue;

				value = atof(offset);

				if (value > 0)
				{
					points[wheel] = value;
					mapped++;
				}
			}
		}
	} else
		return(0);

	fclose(fd);

	if (flags & NO_INTERPOLATE)
		return(mapped);

	from = points[0];

	for (i = 1; i < count; i++)
	{
		if (points[i] != 0.0f)
		{
			if (i == (n + 1))
			{
				n = i;
				from = points[n];
				continue;
			}
			
			/* 
			 * If a start point was not defined then take the first definition
			 * and fill from there.
			 */
			if (from == 0)
				from = points[0] = points[i];

			delta = (points[i] - points[n]) / (i - n); 

			for (++n; n < i; n++)
				points[n] = (from += delta);

			from = points[i];
		}
	}

	/*
	 * If no end point defined then fill from last value.
	 */
	for (++n; n < count; n++)
		points[n] = points[n - 1];

/*
	i = 0;
	printf("%s\n", match);
	printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
		points[i], points[i+1], points[i+2], points[i+3], 
		points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2], points[i+3], 
			points[i+4], points[i+5], points[i+6], points[i+7]);
	if ((i += 8) < count) 
		printf("%1.2f %1.2f %1.2f\n",
			points[i], points[i+1], points[i+2]);
*/

	return(mapped);
}

int
bristolGetFreqMap(char *file, char *match, fTab *freqs, int count, int flags,
int sr)
{
	float points[128];
	int mapped;

	if ((mapped = bristolGetMap(file, match, points, count, flags)) > 0)
	{
		int i;

		for (i = 0; i < mapped; i++)
		{
			freqs[i].freq = points[i];
			freqs[i].step = 1024.0 * points[i] / sr;
		}

		printf("%i frequency mappings: %f %f, %f %f\n", mapped,
			points[0], points[127], freqs[0].step, freqs[127].step);
	}

	return(mapped);
}

/*
 * This could also go into the library so the engine could use the same code?
 *
 * We want to go through the midi controller mapping file for this synth and
 * search for directives for value maps. The names are taken from the midi
 * header file and we want to add a few others for preconfigured value tables.
 */
static void
invertMap(u_char map[128])
{
	int j;

	for (j = 0; j < 128; j++)
		map[j] = (u_char) (127 - map[j]);
}

static void
logMap(u_char map[128])
{
	int i;
	float scaler = 127 / logf(128.0);

	/*
	 * We are going to take a logarithmic scale such that values go from 0 to
	 * 127
	 */
    for (i = 1; i < 129; i++)
		map[i - 1] = (u_char) (logf((float) i) * scaler);
}

static void
exponentialMap(u_char map[128])
{
	int i;

	/*
	 * We are going to take exponential scale
	 */
    for (i = 0; i < 128; i++)
		map[i] = (u_char) (i * i / 127.0);
}

static void
parabolaMap(u_char map[128])
{
	int i;
	float scaler = 127.0 / (64.0 * 64.0);

	/*
	 * We are going to take a parabolic scale. Not sure if it will be use....
	 */
    for (i = 0; i < 128; i++)
		map[i] = (u_char) (((float) ((i - 64) * (i - 64))) * scaler);
}

void
bristolMidiValueMappingTable(u_char valuemap[128][128], int midimap[128], char *synth)
{
	float tmap[128];
	char name[256];
	int i, j;

	sprintf(name, "%s.mcm", synth);

	for (i = 0; i < 128; i++)
		for (j = 0; j < 128; j++)
			valuemap[i][j] = j;

	/*
	 * We first want to look for predefined curves:
	 *
	 *	linear: <c_id> <0|1>
	 *	inverseLinear: <c_id> <0|1>
	 *	log: <c_id> <0|1>
	 *	inverseLog: <c_id> <0|1>
	 *	parabolaMap, inverseParabola
	 *	exponential, inverseExponential
	 *
	 * These should also add some extra curves for velocity mappings that
	 * are also going to be required. These will build non-linear curves from
	 * 0 to 127 over the 127 available values.
	 */
	if (bristolGetMap(name, "controllerMap", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 0.0)
				midimap[i] = (int) tmap[i];
			else
				midimap[i] = i;
		}
		bzero(tmap, sizeof(tmap));
	} else {
		for (i = 0; i < 128; i++)
			midimap[i] = i;
	}

	if (bristolGetMap(name, "inverseLinear", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		/* We have some requirements to map inverse linear */
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
				invertMap(&valuemap[i][0]);
		}
		bzero(tmap, sizeof(tmap));
	}

	if (bristolGetMap(name, "log", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
				logMap(&valuemap[i][0]);
		}
		bzero(tmap, sizeof(tmap));
	}
	if (bristolGetMap(name, "inverseLog", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
			{
				logMap(&valuemap[i][0]);
				invertMap(&valuemap[i][0]);
			}
		}
		bzero(tmap, sizeof(tmap));
	}

	if (bristolGetMap(name, "exponential", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
				exponentialMap(&valuemap[i][0]);
		}
		bzero(tmap, sizeof(tmap));
	}
	if (bristolGetMap(name, "inverseExponential", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
			{
				exponentialMap(&valuemap[i][0]);
				invertMap(&valuemap[i][0]);
			}
		}
		bzero(tmap, sizeof(tmap));
	}

	if (bristolGetMap(name, "parabola", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
				parabolaMap(&valuemap[i][0]);
		}
		bzero(tmap, sizeof(tmap));
	}
	if (bristolGetMap(name, "inverseParabola", &tmap[0], 128, NO_INTERPOLATE) > 0)
	{
		for (i = 0; i < 128; i++)
		{
			if (tmap[i] >= 1.0)
			{
				parabolaMap(&valuemap[i][0]);
				invertMap(&valuemap[i][0]);
			}
		}
		bzero(tmap, sizeof(tmap));
	}

	/*
	 * Then converge the known controllers for any specific profiles configured
	 * by the user to override both the default linear curves and any other
	 * predefined curves.
	 */
	for (i = 0; i < 128; i++) {
		if (controllerName[i] == NULL)
			continue;

		if (bristolGetMap(name, controllerName[i], &tmap[0], 128, 0) > 0)
		{
			/*
			 * If something was mapped for this controller then see about
			 * converging the values
			 */
			for (j = 0; j < 128; j++)
				if ((tmap[j] >= 0) && (tmap[j] < 128))
					valuemap[i][j] = (u_char) tmap[j];
			bzero(tmap, sizeof(tmap));
		}
	}
}

