#!/usr/bin/env python3
"""
Generates random rough surface of the sample with the given 
heights statistics and lateral roughness spectrum.
"""

import numpy as np
import bornagain as ba
from bornagain import nm
from matplotlib import pyplot as plt, gridspec

def plot(h):          
    fig = plt.figure(figsize=(10, 6))
    gs = gridspec.GridSpec(3, 2)
    gs.update(left=0.1, right=0.48, wspace=0.15, hspace=0.5)

    # map
    fig.add_subplot(gs[:-1, :])
    plt.imshow(h, extent = [0, Lx, 0, Ly], cmap='inferno', aspect='auto')  
    plt.colorbar()
    
    # psd
    fig.add_subplot(gs[-1, 0])    
    factor = X_points*Y_points/np.sqrt(Lx*Ly)
    N = np.floor(X_points/2 + 1).astype(int)
    psd = abs(np.fft.rfft2(h)/factor)**2
    psd_points_x = np.array([psd[0][i] for i in range(1, N)])
    f_points_x = np.array([i/Lx for i in range(1, N)])
    plt.plot(f_points_x, psd_points_x)
    plt.yscale('log') 
    plt.xscale('log')
    plt.title('PSD')
    plt.xlabel('$\\nu, nm^{-1}$')
     
    # hist
    fig.add_subplot(gs[-1, 1])
    plt.hist(h.flatten(), range=[-4*sigma, 4*sigma], 
             bins=300, color='black')
    plt.title('Height histogram')
    plt.xlabel('h, nm')
    plt.tick_params(left = False, labelleft = False) 

    plt.show()

# roughness parameters
sigma = 1*nm
alpha = 0.5
xi = 35*nm
height_distribution = ba.ErfRoughness()

# sample size
Lx = 1000*nm
Ly = 1000*nm
X_points = 512
Y_points = 512

# create roughness model
roughness = ba.LayerRoughness(sigma, alpha, xi, height_distribution)

# generate roughness map
roughness_map = ba.RoughnessMap(X_points, Y_points, Lx, Ly, roughness)
surface = roughness_map.generate()
      
print("rms = {:.3}".format(np.std(surface)),"nm")

plot(surface)


