//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/SampleItem.h
//! @brief     Defines class SampleItem.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H

#include "Base/Type/OwningVector.h"
#include "GUI/Model/Descriptor/VectorProperty.h"
#include "GUI/Model/Material/MaterialsSet.h"
#include "GUI/Model/Sample/Item3D.h"
#include "GUI/Model/Type/NamedItem.h"
#include <QString>
#include <QXmlStreamReader>

class ItemWithMaterial;
class LayerItem;

class SampleItem : public virtual Item3D, public NamedItem {
public:
    SampleItem();
    ~SampleItem();

    SampleItem* clone() const;

    std::vector<ItemWithMaterial*> itemsWithMaterial() const;

    void addStandardMaterials();

    DoubleProperty& crossCorrLength() { return m_cross_correlation_length; }
    const DoubleProperty& crossCorrLength() const { return m_cross_correlation_length; }
    void setCrossCorLength(double d) { m_cross_correlation_length.setDVal(d); }

    VectorProperty& externalField() { return m_external_field; }
    const VectorProperty& externalField() const { return m_external_field; }
    void setExternalField(const R3& r) { m_external_field.setR3(r); }

    const std::vector<LayerItem*>& layerItems() const { return m_layers.shared(); }

    //! Creates and inserts a layer at given index.
    //!
    //! No properties etc. have been initialized; this has to be done by the caller.
    //! If index = -1, create a layer at the end of the list.
    LayerItem* createLayerItemAt(int index = -1);

    void updateTopBottom();
    void removeLayer(LayerItem* layer);
    void moveLayer(LayerItem* layer, LayerItem* aboveThisLayer);

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    MaterialsSet& materialModel() { return m_materials; }
    const MaterialsSet& materialModel() const { return m_materials; }

    void updateDefaultLayerColors();

    bool expandInfo = true;

private:
    DoubleProperty m_cross_correlation_length;
    VectorProperty m_external_field;
    OwningVector<LayerItem> m_layers;
    MaterialsSet m_materials;
};

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_SAMPLEITEM_H
