//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Descriptor/AxisProperty.cpp
//! @brief     Implements class AxisProperty.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2022
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Descriptor/AxisProperty.h"
#include "GUI/Model/Util/UtilXML.h"

namespace {
namespace Tag {

const QString Nbins("Nbins");
const QString Min("Min");
const QString Max("Max");
const QString ExpandGroupbox("ExpandGroupbox"); // obsolete since v22.0

} // namespace Tag
} // namespace


AxisProperty::AxisProperty() = default;

AxisProperty::AxisProperty(const QString& name, const QString& unit)
{
    initMin(name + "_min (" + unit + ")", name + "_min (" + unit + ")", -1.,
            RealLimits::limitless(), 3);
    initMax(name + "_max (" + unit + ")", name + "_max (" + unit + ")", +1.,
            RealLimits::limitless(), 3);
}

void AxisProperty::initMin(const QString& label, const QString& tooltip, double value,
                           const RealLimits& limits, uint decimals)
{
    m_min.init(label, tooltip, value, decimals, limits, "min");
}

void AxisProperty::initMax(const QString& label, const QString& tooltip, double value,
                           const RealLimits& limits, uint decimals)
{
    m_max.init(label, tooltip, value, decimals, limits, "max");
}

void AxisProperty::writeTo(QXmlStreamWriter* w) const
{
    XML::writeTaggedValue(w, Tag::Nbins, m_nbins);

    // min
    m_min.writeTo2(w, Tag::Min);

    // max
    m_max.writeTo2(w, Tag::Max);
}

void AxisProperty::readFrom(QXmlStreamReader* r)
{
    while (r->readNextStartElement()) {
        QString tag = r->name().toString();
        if (tag == Tag::Nbins)
            m_nbins = XML::readTaggedUInt(r, tag);
        else if (tag == Tag::Min) {
            m_min.readFrom2(r, tag);
        } else if (tag == Tag::Max) {
            m_max.readFrom2(r, tag);

        } else
            r->skipCurrentElement();
    }
}
