/***************************************************************************
*            play-list.c
*
*  mer mai 25 22:22:53 2005
*  Copyright  2005  Philippe Rouquier
*  bonfire-app@wanadoo.fr
****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef BUILD_PLAYLIST

#include <string.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib-object.h>

#include <gtk/gtkwidget.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeviewcolumn.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkselection.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkmessagedialog.h>

#include <libgnomevfs/gnome-vfs.h>

#ifdef BUILD_SEARCH
#include <beagle/beagle.h>
#endif

#include <totem-pl-parser.h>

#include "play-list.h"
#include "utils.h"
#include "metadata.h"
#include "bonfire-uri-container.h"

static void bonfire_playlist_class_init (BonfirePlaylistClass * klass);
static void bonfire_playlist_init (BonfirePlaylist * sp);
static void bonfire_playlist_iface_uri_container_init (BonfireURIContainerIFace *iface);
static void bonfire_playlist_finalize (GObject * object);
static void bonfire_playlist_destroy (GtkObject * object);

struct BonfirePlaylistPrivate {

#ifdef BUILD_SEARCH
	BeagleClient *client;
	BeagleQuery *query;
	int id;
#endif

	GtkWidget *tree;
	GtkWidget *button_add;
	GtkWidget *button_remove;
	guint activity_counter;
};

enum {
	BONFIRE_PLAYLIST_DISPLAY_COL,
	BONFIRE_PLAYLIST_NB_SONGS_COL,
	BONFIRE_PLAYLIST_LEN_COL,
	BONFIRE_PLAYLIST_GENRE_COL,
	BONFIRE_PLAYLIST_URI_COL,
	BONFIRE_PLAYLIST_NB_COL,
};

#ifdef BUILD_SEARCH
static void bonfire_playlist_beagle_hit_added_cb (BeagleQuery * query,
						  BeagleHitsAddedResponse *
						  response,
						  BonfirePlaylist *
						  playlist);
static void bonfire_playlist_beagle_hit_substracted_cb (BeagleQuery *
							query,
							BeagleHitsSubtractedResponse
							* response,
							BonfirePlaylist *
							playlist);
static void bonfire_playlist_beagle_finished_cb (BeagleQuery * query,
						 BeagleFinishedResponse *
						 response,
						 BonfirePlaylist *
						 playlist);
#endif

static void bonfire_playlist_drag_data_get_cb (GtkTreeView * tree,
					       GdkDragContext *
					       drag_context,
					       GtkSelectionData *
					       selection_data, guint info,
					       guint time,
					       BonfirePlaylist * playlist);
static void bonfire_playlist_add_cb (GtkButton * button,
				     BonfirePlaylist * playlist);
static void bonfire_playlist_remove_cb (GtkButton * button,
					BonfirePlaylist * playlist);
static void bonfire_playlist_add_uri_playlist (BonfirePlaylist * playlist,
					       const char *uri,
					       gboolean quiet);
static void bonfire_playlist_search_playlists_rhythmbox (BonfirePlaylist *
							 playlist);
static void bonfire_playlist_increase_activity_counter (BonfirePlaylist *
							playlist);
static void bonfire_playlist_decrease_activity_counter (BonfirePlaylist *
							playlist);
static void bonfire_playlist_row_activated_cb (GtkTreeView * tree,
					       GtkTreeIter * row,
					       GtkTreeViewColumn * column,
					       BonfirePlaylist * playlist);
static void bonfire_playlist_selection_changed_cb (GtkTreeSelection *
						   selection,
						   BonfirePlaylist *
						   playlist);
static char **bonfire_playlist_get_selected_uris_real (BonfirePlaylist * playlist);

static char **
bonfire_playlist_get_selected_uris (BonfireURIContainer *container);
static char *
bonfire_playlist_get_selected_uri (BonfireURIContainer *container);

enum {
	TARGET_URIS_LIST,
};

static GtkTargetEntry ntables[] = {
	{"text/uri-list", 0, TARGET_URIS_LIST}
};
static guint nb_ntables = sizeof (ntables) / sizeof (ntables[0]);

static GObjectClass *parent_class = NULL;

GType
bonfire_playlist_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfirePlaylistClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_playlist_class_init,
			NULL,
			NULL,
			sizeof (BonfirePlaylist),
			0,
			(GInstanceInitFunc) bonfire_playlist_init,
		};

		static const GInterfaceInfo uri_container_info =
		{
			(GInterfaceInitFunc) bonfire_playlist_iface_uri_container_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GTK_TYPE_VBOX,
					       "BonfirePlaylist",
					       &our_info, 0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_URI_CONTAINER,
					     &uri_container_info);
	}

	return type;
}

static void
bonfire_playlist_class_init (BonfirePlaylistClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtkobject_class = GTK_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_playlist_finalize;
	gtkobject_class->destroy = bonfire_playlist_destroy;
}

static void
bonfire_playlist_iface_uri_container_init (BonfireURIContainerIFace *iface)
{
	iface->get_selected_uri = bonfire_playlist_get_selected_uri;
	iface->get_selected_uris = bonfire_playlist_get_selected_uris;
}

static void
bonfire_playlist_init (BonfirePlaylist * obj)
{
	GtkWidget *hbox, *scroll;
	GtkTreeStore *store = NULL;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	obj->priv = g_new0 (BonfirePlaylistPrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 6);

	hbox = gtk_hbox_new (FALSE, 8);

	obj->priv->button_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_box_pack_start (GTK_BOX (hbox),
			    obj->priv->button_add,
			    FALSE,
			    FALSE, 0);
	g_signal_connect (G_OBJECT (obj->priv->button_add), "clicked",
			  G_CALLBACK (bonfire_playlist_add_cb), obj);

	obj->priv->button_remove = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
	gtk_box_pack_start (GTK_BOX (hbox), obj->priv->button_remove,
			    FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (obj->priv->button_remove), "clicked",
			  G_CALLBACK (bonfire_playlist_remove_cb), obj);

	gtk_box_pack_end (GTK_BOX (obj), hbox, FALSE, FALSE, 0);

	store = gtk_tree_store_new (BONFIRE_PLAYLIST_NB_COL,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_STRING, 
				    G_TYPE_STRING);

	obj->priv->tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW
					     (obj->priv->tree), TRUE);
	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (obj->priv->tree),
					 TRUE);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (obj->priv->tree),
				      TRUE);
	gtk_tree_view_set_expander_column (GTK_TREE_VIEW (obj->priv->tree),
					   BONFIRE_PLAYLIST_DISPLAY_COL);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Playlists"),
							   renderer, "text",
							   BONFIRE_PLAYLIST_DISPLAY_COL,
							   NULL);
	gtk_tree_view_column_set_sort_column_id (column,
						 BONFIRE_PLAYLIST_DISPLAY_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_expand (column, TRUE);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Number of songs"),
							   renderer, "text",
							   BONFIRE_PLAYLIST_NB_SONGS_COL,
							   NULL);
	gtk_tree_view_column_set_sort_column_id (column,
						 BONFIRE_PLAYLIST_NB_SONGS_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Length"),
							   renderer, "text",
							   BONFIRE_PLAYLIST_LEN_COL,
							   NULL);
	gtk_tree_view_column_set_sort_column_id (column,
						 BONFIRE_PLAYLIST_LEN_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Genre"), renderer,
							   "text",
							   BONFIRE_PLAYLIST_GENRE_COL,
							   NULL);
	gtk_tree_view_column_set_sort_column_id (column,
						 BONFIRE_PLAYLIST_GENRE_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);

	gtk_tree_view_set_search_column (GTK_TREE_VIEW (obj->priv->tree),
					 BONFIRE_PLAYLIST_DISPLAY_COL);
	gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (obj->priv->tree),
						GDK_BUTTON1_MASK, ntables,
						nb_ntables,
						GDK_ACTION_COPY |
						GDK_ACTION_MOVE);

	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_data_get",
			  G_CALLBACK (bonfire_playlist_drag_data_get_cb),
			  obj);

	g_signal_connect (G_OBJECT (obj->priv->tree), "row_activated",
			  G_CALLBACK (bonfire_playlist_row_activated_cb),
			  obj);

	g_signal_connect (G_OBJECT (gtk_tree_view_get_selection (GTK_TREE_VIEW (obj->priv->tree))),
			  "changed",
			  G_CALLBACK (bonfire_playlist_selection_changed_cb),
			  obj);

	gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (obj->priv->tree)),
				     GTK_SELECTION_MULTIPLE);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (scroll), obj->priv->tree);
	gtk_box_pack_start (GTK_BOX (obj), scroll, TRUE, TRUE, 0);
}

static void
bonfire_playlist_increase_activity_counter (BonfirePlaylist * playlist)
{
	if (!GTK_WIDGET (playlist->priv->tree)->window)
		return;

	if (playlist->priv->activity_counter == 0) {
		GdkCursor *cursor;

		cursor = gdk_cursor_new (GDK_WATCH);
		gdk_window_set_cursor (GTK_WIDGET (playlist->priv->tree)->window,
				       cursor);
		gdk_cursor_unref (cursor);
	}
	playlist->priv->activity_counter++;
}

static void
bonfire_playlist_decrease_activity_counter (BonfirePlaylist * playlist)
{
	if (playlist->priv->activity_counter > 0)
		playlist->priv->activity_counter--;

	if (!GTK_WIDGET (playlist->priv->tree)->window)
		return;

	if (playlist->priv->activity_counter == 0)
		gdk_window_set_cursor (GTK_WIDGET (playlist->priv->tree)->window,
				       NULL);
}

static void
bonfire_playlist_destroy (GtkObject * object)
{

#ifdef BUILD_SEARCH

	BonfirePlaylist *playlist = BONFIRE_PLAYLIST (object);

	if (playlist->priv->id) {
		g_source_remove (playlist->priv->id);
		playlist->priv->id = 0;
	}

	if (playlist->priv->client) {
		g_object_unref (playlist->priv->client);
		playlist->priv->client = NULL;
	}

	if (playlist->priv->query) {
		g_object_unref (playlist->priv->query);
		playlist->priv->query = NULL;
	}

#endif

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
bonfire_playlist_finalize (GObject * object)
{
	BonfirePlaylist *cobj;

	cobj = BONFIRE_PLAYLIST (object);

	g_free (cobj->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

#ifdef BUILD_SEARCH
static gboolean
bonfire_playlist_try_again (BonfirePlaylist *playlist)
{
	playlist->priv->client = beagle_client_new (NULL);
	if (!playlist->priv->client)
		return TRUE;

	playlist->priv->id = 0;
	return FALSE;
}
#endif

GtkWidget *
bonfire_playlist_new ()
{
	BonfirePlaylist *obj;

	obj = BONFIRE_PLAYLIST (g_object_new (BONFIRE_TYPE_PLAYLIST, NULL));

#ifdef BUILD_SEARCH

	obj->priv->client = beagle_client_new (NULL);
	if(obj->priv->client) {
		GError *error = NULL;

		obj->priv->query = beagle_query_new ();

		g_signal_connect (G_OBJECT (obj->priv->query), "hits-added",
				  G_CALLBACK (bonfire_playlist_beagle_hit_added_cb),
				  obj);
		g_signal_connect (G_OBJECT (obj->priv->query), "hits-subtracted",
				  G_CALLBACK (bonfire_playlist_beagle_hit_substracted_cb),
				  obj);
		g_signal_connect (G_OBJECT (obj->priv->query), "finished",
				  G_CALLBACK (bonfire_playlist_beagle_finished_cb),
				  obj);
	
		beagle_query_add_source (obj->priv->query, "Files");
	
		beagle_query_add_mime_type (obj->priv->query, "audio/x-scpls");
		beagle_query_add_mime_type (obj->priv->query, "audio/x-ms-asx");
		beagle_query_add_mime_type (obj->priv->query, "audio/x-mp3-playlist");
	
		bonfire_playlist_increase_activity_counter (obj);
		beagle_client_send_request_async (obj->priv->client,
						  BEAGLE_REQUEST (obj->priv->query),
						  &error);
		if (error) {
			g_warning ("Could not connect to beagle : %s\n",
				   error->message);
			g_error_free (error);
		}
	}
	else {
		/* we will retry in 10 seconds */
		obj->priv->id = g_timeout_add (10000,
					       (GSourceFunc) bonfire_playlist_try_again,
					       obj);
	}

#endif

	bonfire_playlist_search_playlists_rhythmbox (obj);
	return GTK_WIDGET (obj);
}

static char **
bonfire_playlist_get_selected_uris (BonfireURIContainer *container)
{
	BonfirePlaylist *playlist;

	playlist = BONFIRE_PLAYLIST (container);
	return bonfire_playlist_get_selected_uris_real (playlist);
}

static char *
bonfire_playlist_get_selected_uri (BonfireURIContainer *container)
{
	BonfirePlaylist *playlist;
	char **uris = NULL;
	char *uri;

	playlist = BONFIRE_PLAYLIST (container);
	uris = bonfire_playlist_get_selected_uris_real (playlist);

	if (uris) {
		uri = uris [0];
		g_strfreev (uris);
		return uri;
	}

	return NULL;
}

void
bonfire_playlist_unselect_all (BonfirePlaylist * playlist)
{
	GtkTreeSelection *selection;

	selection =
	    gtk_tree_view_get_selection (GTK_TREE_VIEW
					 (playlist->priv->tree));
	gtk_tree_selection_unselect_all (selection);
}

static void
bonfire_playlist_add_cb (GtkButton * button, BonfirePlaylist * playlist)
{
	GtkWidget *dialog, *toplevel;
	char *uri;
	GSList *uris, *iter;
	gint result;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (playlist));
	if (!GTK_WIDGET_TOPLEVEL (toplevel))
		return;

	dialog = gtk_file_chooser_dialog_new (_("Select a playlist"),
					      GTK_WINDOW (toplevel),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					      NULL);

	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 GTK_RESPONSE_ACCEPT);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (dialog), TRUE);
	gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (dialog), TRUE);
	gtk_file_chooser_set_current_folder_uri (GTK_FILE_CHOOSER (dialog),
						 g_get_home_dir ());

	gtk_widget_show_all (dialog);
	result = gtk_dialog_run (GTK_DIALOG (dialog));
	if (result == GTK_RESPONSE_CANCEL) {
		gtk_widget_destroy (dialog);
		return;
	}

	uris = gtk_file_chooser_get_uris (GTK_FILE_CHOOSER (dialog));
	gtk_widget_destroy (dialog);

	for (iter = uris; iter; iter = iter->next) {
		uri = iter->data;
		bonfire_playlist_add_uri_playlist (playlist, uri, FALSE);
		g_free (uri);
	}
	g_slist_free (uris);
}

static void
bonfire_playlist_remove_cb (GtkButton * button, BonfirePlaylist * playlist)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter row;
	GList *rows, *iter;
	gboolean valid;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (playlist->priv->tree));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (playlist->priv->tree));
	rows = gtk_tree_selection_get_selected_rows (selection, &model);

	if (rows == NULL)
		return;

	/* we just remove the lists removing particular songs would be a nonsense */
	/* we must reverse the list otherwise the last paths wouldn't be valid */
	for (iter = g_list_last (rows); iter; iter = iter->prev) {
		path = iter->data;
		valid = gtk_tree_model_get_iter (model, &row, path);
		gtk_tree_path_free (path);

		if (valid == FALSE)	/* if we remove the whole list it could happen that we try to remove twice a song */
			continue;

		if (gtk_tree_model_iter_has_child (model, &row)) {
			GtkTreeIter child;

			/* we remove the songs if it's a list */
			gtk_tree_model_iter_children (model, &child, &row);
			while (gtk_tree_store_remove
			       (GTK_TREE_STORE (model), &child) == TRUE);
			gtk_tree_store_remove (GTK_TREE_STORE (model),
					       &row);
		}
	}

	g_list_free (rows);
}

static char **
bonfire_playlist_get_selected_uris_real (BonfirePlaylist * playlist)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreePath *path;
	GList *rows, *iter;
	GtkTreeIter row;
	char **uris = NULL, *uri;
	GPtrArray *array;
	gboolean valid;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (playlist->priv->tree));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (playlist->priv->tree));

	rows = gtk_tree_selection_get_selected_rows (selection, &model);

	if (rows == NULL)
		return NULL;

	rows = g_list_reverse (rows);

	array = g_ptr_array_sized_new (g_list_length (rows) + 1);
	for (iter = rows; iter; iter = iter->next) {
		path = iter->data;
		valid = gtk_tree_model_get_iter (model, &row, path);
		gtk_tree_path_free (path);

		if (valid == FALSE)
			continue;
		//FIXME : we must find a way to reverse the list        
		/* check if it is a list or not */
		if (gtk_tree_model_iter_has_child (model, &row)) {
			GtkTreeIter child;

			if (gtk_tree_model_iter_children
			    (model, &child, &row) == FALSE)
				continue;

			do {
				/* first check if the row is selected to prevent to put it in the list twice */
				if (gtk_tree_selection_iter_is_selected
				    (selection, &child) == TRUE)
					continue;

				gtk_tree_model_get (model, &child,
						    BONFIRE_PLAYLIST_URI_COL,
						    &uri, -1);
				g_ptr_array_add (array, uri);
			} while (gtk_tree_model_iter_next (model, &child));

			continue;
		}

		gtk_tree_model_get (model, &row, BONFIRE_PLAYLIST_URI_COL,
				    &uri, -1);
		g_ptr_array_add (array, uri);
	}

	g_list_free (rows);

	g_ptr_array_set_size (array, array->len + 1);
	uris = (char **) array->pdata;
	g_ptr_array_free (array, FALSE);
	return uris;
}

static void
bonfire_playlist_drag_data_get_cb (GtkTreeView * tree,
				   GdkDragContext * drag_context,
				   GtkSelectionData * selection_data,
				   guint info,
				   guint time, BonfirePlaylist * playlist)
{
	char **uris;

	uris = bonfire_playlist_get_selected_uris_real (playlist);
	gtk_selection_data_set_uris (selection_data, uris);
	g_strfreev (uris);
}

struct _BonfirePlaylistParseData {
	BonfirePlaylist *playlist;
	TotemPlParserResult result;

	char *uri;
	char *title;
	guint nb_songs;
	double length;

	GSList *songs;
	gboolean quiet;
};
typedef struct _BonfirePlaylistParseData BonfirePlaylistParseData;

struct _BonfirePlaylistSong {
	gchar *uri;
	gchar *title;
	gchar *genre;
	char *type;
	double length;
};
typedef struct _BonfirePlaylistSong BonfirePlaylistSong;

static void
bonfire_playlist_start_cb (TotemPlParser * parser,
			   const char *title,
			   BonfirePlaylistParseData * data)
{
	data->title = g_strdup (title);
}

static void
bonfire_playlist_end_cb (TotemPlParser * parser,
			 const char *title,
			 BonfirePlaylistParseData * data)
{
	if (data->title == NULL)
		data->title = g_strdup (title);
}

static void
bonfire_playlist_entry_cb (TotemPlParser * parser,
			   const char *uri,
			   const char *title,
			   const char *genre,
			   BonfirePlaylistParseData * data)
{
	BonfireMetadata *metadata;
	BonfirePlaylistSong *song;
	GError *error = NULL;

	/* now let's get the length of it */
	metadata = bonfire_metadata_new (uri);
	if (!metadata)
		return;

	if (!metadata || !bonfire_metadata_get_sync (metadata, &error)) {
		if (error) {
			g_warning ("ERROR : %s\n", error->message);
			g_error_free (error);
			g_object_unref (metadata);
		}

		return;
	}

	data->nb_songs++;
	song = g_new0 (BonfirePlaylistSong, 1);
	data->songs = g_slist_prepend (data->songs, song);
	song->uri = g_strdup (uri);
	song->title = g_strdup (title);
	song->genre = g_strdup (genre);

	song->length = metadata->len;
	song->type = g_strdup (metadata->type);
	data->length += metadata->len;
	g_object_unref (metadata);
}

static gboolean
bonfire_playlist_parse_thread_end (BonfirePlaylistParseData * data)
{
	char *name;
	GtkTreeModel *model;
	GtkTreeIter row, parent;

	BonfirePlaylistSong *song_info;
	char *len_string, *total_len_string, *num_string;
	GSList *song;

	if (data->result != TOTEM_PL_PARSER_RESULT_SUCCESS) {
		char *name;
		GtkWidget *dialog;
		GtkWidget *toplevel;

		if (data->quiet)
			goto cleanup;

		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (data->playlist));

		if (!GTK_WIDGET_TOPLEVEL (toplevel)) {
			g_warning ("Error parsing playlist %s\n", data->uri);
			goto cleanup;
		}

		name = g_path_get_basename (data->uri);
		dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
						 GTK_DIALOG_DESTROY_WITH_PARENT|
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Error parsing playlist \"%s\":"),
						 name);
		g_free (name);

		gtk_window_set_title (GTK_WINDOW (dialog), _("Playlist loading error"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  _("an unknown error occured."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		goto cleanup;
	}

	/* actually add it */
	if (data->nb_songs == 0) {
		GtkWidget *dialog;
		GtkWidget *toplevel;
		char *name;

		if (data->quiet)
			goto cleanup;

		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (data->playlist));
		if (!GTK_WIDGET_TOPLEVEL (toplevel)) {
			g_warning ("No song in the playlist %s\n", data->uri);
			goto cleanup;
		}

		name = g_path_get_basename (data->uri);
		dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
						 GTK_DIALOG_DESTROY_WITH_PARENT|
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Error loading playlist \"%s\":"),
						 name);
		g_free (name);

		gtk_window_set_title (GTK_WINDOW (dialog), _("Playlist loading error"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  _("the playlist is empty."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		goto cleanup;
	}
	else if (data->nb_songs == 1)
		num_string = g_strdup_printf (_("1 song"));
	else
		num_string = g_strdup_printf (_("%i songs"), data->nb_songs);

	total_len_string = bonfire_utils_get_time_string (data->length, TRUE);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (data->playlist->priv->tree));

	name = g_path_get_basename (data->uri);
	gtk_tree_store_append (GTK_TREE_STORE (model), &parent, NULL);
	gtk_tree_store_set (GTK_TREE_STORE (model), &parent,
			    BONFIRE_PLAYLIST_DISPLAY_COL,
			    data->title ? data->title : name,
			    BONFIRE_PLAYLIST_NB_SONGS_COL, num_string,
			    BONFIRE_PLAYLIST_URI_COL, data->uri,
			    BONFIRE_PLAYLIST_LEN_COL, total_len_string,
			    -1);
	g_free (name);
	g_free (num_string);
	g_free (total_len_string);

	for (song = data->songs; song; song = song->next) {
		song_info = song->data;
		gtk_tree_store_append (GTK_TREE_STORE (model), &row,
				       &parent);

		if (song_info->length > 0)
			len_string = bonfire_utils_get_time_string ((double) song_info->length,
								    TRUE);
		else
			len_string = NULL;

		name = g_path_get_basename (song_info->uri);
		gtk_tree_store_set (GTK_TREE_STORE (model), &row,
				    BONFIRE_PLAYLIST_DISPLAY_COL,
				    song_info->title ? song_info->title : name,
				    BONFIRE_PLAYLIST_LEN_COL, len_string,
				    BONFIRE_PLAYLIST_URI_COL, song_info->uri,
				    BONFIRE_PLAYLIST_GENRE_COL, song_info->genre,
				    -1);

		g_free (song_info->title);
		g_free (song_info->genre);
		g_free (song_info->type);
		g_free (song_info->uri);
		g_free (song_info);
		g_free (name);

		if (len_string)
			g_free (len_string);
	}

	g_slist_free (data->songs);


      cleanup:

	bonfire_playlist_decrease_activity_counter (data->playlist);
	g_free (data->title);
	g_free (data->uri);
	g_free (data);
	return FALSE;
}

static gpointer
bonfire_playlist_parse_thread (BonfirePlaylistParseData * data)
{
	TotemPlParser *parser;

	parser = totem_pl_parser_new ();
	g_signal_connect (G_OBJECT (parser), "playlist-start",
			  G_CALLBACK (bonfire_playlist_start_cb), data);
	g_signal_connect (G_OBJECT (parser), "playlist-end",
			  G_CALLBACK (bonfire_playlist_end_cb), data);
	g_signal_connect (G_OBJECT (parser), "entry",
			  G_CALLBACK (bonfire_playlist_entry_cb), data);

	data->result = totem_pl_parser_parse (parser, data->uri, TRUE);
	g_object_unref (parser);

	g_idle_add ((GSourceFunc) bonfire_playlist_parse_thread_end, data);

	return NULL;
}

static void
bonfire_playlist_add_uri_playlist (BonfirePlaylist *playlist,
				   const char *uri,
				   gboolean quiet)
{
	BonfirePlaylistParseData *data;
	GError *error = NULL;
	GThread *thread;

	data = g_new0 (BonfirePlaylistParseData, 1);
	data->uri = g_strdup (uri);
	data->playlist = playlist;
	data->quiet = quiet;

	thread = g_thread_create ((GThreadFunc) bonfire_playlist_parse_thread,
				  data,
				  TRUE,
				  &error);

	if (!thread || error) {
		g_warning ("ERROR can't create thread : %s\n", error->message);
		g_error_free (error);
		g_free (data->uri);
		g_free (data);
	}
	else
		bonfire_playlist_increase_activity_counter (playlist);
}

static void
bonfire_playlist_search_playlists_rhythmbox (BonfirePlaylist * playlist)
{
/*	RBSource *source;

	manager = rb_playlist_manager_new ();
	lists = rb_playlist_manager_get_playlists (manager);
*/
}

#ifdef BUILD_SEARCH
static void
bonfire_playlist_beagle_hit_added_cb (BeagleQuery * query,
				      BeagleHitsAddedResponse * response,
				      BonfirePlaylist * playlist)
{
	GSList *list, *iter;
	BeagleHit *hit;
	const char *uri;

	list = beagle_hits_added_response_get_hits (response);
	for (iter = list; iter; iter = iter->next) {
		hit = iter->data;
		uri = beagle_hit_get_uri (hit);
		bonfire_playlist_add_uri_playlist (playlist, uri, TRUE);
	}
	bonfire_playlist_decrease_activity_counter (playlist);
}

static void
bonfire_playlist_beagle_hit_substracted_cb (BeagleQuery * query,
					    BeagleHitsSubtractedResponse *
					    response,
					    BonfirePlaylist * playlist)
{
	GSList *list, *iter;
	const char *uri;
	char *row_uri;

	GtkTreeModel *model;
	GtkTreeIter row;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (playlist->priv->tree));

	list = beagle_hits_subtracted_response_get_uris (response);
	for (iter = list; iter; iter = iter->next) {
		uri = iter->data;

		if (gtk_tree_model_get_iter_first (model, &row))
			do {
				gtk_tree_model_get (model, &row,
						    BONFIRE_PLAYLIST_URI_COL,
						    &row_uri, -1);
				if (!strcmp (row_uri, uri)) {
					gtk_tree_store_remove
					    (GTK_TREE_STORE (model), &row);
					g_free (row_uri);
					break;
				}
				g_free (row_uri);
			} while (gtk_tree_model_iter_next (model, &row));
	}
}

static void
bonfire_playlist_beagle_finished_cb (BeagleQuery * query,
				     BeagleFinishedResponse * response,
				     BonfirePlaylist * playlist)
{
	bonfire_playlist_decrease_activity_counter (playlist);
}

#endif

static void
bonfire_playlist_row_activated_cb (GtkTreeView * tree,
				   GtkTreeIter * row,
				   GtkTreeViewColumn * column,
				   BonfirePlaylist * playlist)
{
	bonfire_uri_container_uri_activated (BONFIRE_URI_CONTAINER (playlist));
}

static void
bonfire_playlist_selection_changed_cb (GtkTreeSelection * selection,
				       BonfirePlaylist * playlist)
{
	bonfire_uri_container_uri_selected (BONFIRE_URI_CONTAINER (playlist));
}

#endif
