/***************************************************************************
 *            transcode.c
 *
 *  ven jui  8 16:15:04 2005
 *  Copyright  2005  Philippe Rouquier
 *  Bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>
#include <errno.h>
#include <unistd.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gst/gst.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "bonfire-marshal.h"
#include "burn-common.h"
#include "burn-job.h"
#include "burn-imager.h"
#include "burn-transcode.h"

static void bonfire_transcode_class_init (BonfireTranscodeClass *klass);
static void bonfire_transcode_init (BonfireTranscode *sp);
static void bonfire_transcode_finalize (GObject *object);
static void bonfire_transcode_iface_init_image (BonfireImagerIFace *iface);

static BonfireBurnResult
bonfire_transcode_set_source (BonfireJob *job,
			      const BonfireTrackSource *source,
			      GError **error);
static BonfireBurnResult
bonfire_transcode_set_output (BonfireImager *imager,
			      const char *output,
			      gboolean overwrite,
			      gboolean clean,
			      GError **error);
static BonfireBurnResult
bonfire_transcode_set_output_type (BonfireImager *imager, 
				   BonfireTrackSourceType type,
				   GError **error);
static BonfireBurnResult
bonfire_transcode_get_track (BonfireImager *imager,
			     BonfireTrackSource **track,
			     GError **error);
static BonfireBurnResult
bonfire_transcode_get_track_type (BonfireImager *imager,
				  BonfireTrackSourceType *type);
static BonfireBurnResult
bonfire_transcode_get_size (BonfireImager *imager,
			    gint64 *size,
			    gboolean sectors,
			    GError **error);

static BonfireBurnResult
bonfire_transcode_get_action_string (BonfireJob *job,
				     BonfireBurnAction action,
				     char **string);
static BonfireBurnResult
bonfire_transcode_start (BonfireJob *job,
			 int fd_in,
			 int *fd_out,
			 GError **error);
static BonfireBurnResult
bonfire_transcode_stop (BonfireJob *job,
			BonfireBurnResult retval);
static BonfireBurnResult
bonfire_transcode_set_rate (BonfireJob *job,
			    gint64 rate);

static gboolean bonfire_transcode_bus_messages (GstBus *bus,
						  GstMessage *msg,
						  BonfireTranscode *transcode);
static void bonfire_transcode_new_decoded_pad_cb (GstElement *decode,
						   GstPad *pad,
						   gboolean arg2,
						   GstElement *convert);
static BonfireBurnResult
bonfire_transcode_next_song (BonfireTranscode *transcode,
			     GError **error);

typedef struct _BonfireSong BonfireSong;
struct _BonfireSong {
	BonfireSong *next;
	char *uri;
	char *dest;
	char *inf;

	gint64 duration;
	int sectors;
	int index0;
	int start;
	int index;

	/* for CD-Text */
	char *artist;
	char *title;
	char *composer;
	int isrc;
};

typedef enum {
	BONFIRE_TRANSCODE_ACTION_NONE,
	BONFIRE_TRANSCODE_ACTION_INF,
	BONFIRE_TRANSCODE_ACTION_TRANSCODING,
	BONFIRE_TRANSCODE_ACTION_GETTING_SIZE
} BonfireTranscodeAction;

struct BonfireTranscodePrivate {
	BonfireTranscodeAction action;

	GstElement *pipeline;
	GstElement *identity;
	GstElement *convert;
	GstElement *decode;
	GstElement *source;
	GstElement *sink;

	gint64 rate;

	char *output;
	int pipe_out;

	int progress_id;
	GTimer *timer;
	gint64 start_pos;
	gint64 global_pos;
	GSList *rates;

	BonfireTrackSourceType track_type;
	BonfireSong *current;
	BonfireSong *songs;
	char *album;

	int sectors_num;

	int pad_size;
	gint pad_id;

	int audio_ready:1;
	int inf_ready:1;

	int own_output:1;
	int overwrite:1;
	int clean:1;

	int on_the_fly:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_transcode_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireTranscodeClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_transcode_class_init,
			NULL,
			NULL,
			sizeof (BonfireTranscode),
			0,
			(GInstanceInitFunc) bonfire_transcode_init,
		};

		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) bonfire_transcode_iface_init_image,
			NULL,
			NULL
		};
		type = g_type_register_static (BONFIRE_TYPE_JOB,
					       "BonfireTranscode",
					       &our_info, 0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
	}

	return type;
}

static void
bonfire_transcode_iface_init_image (BonfireImagerIFace *iface)
{
	iface->get_size = bonfire_transcode_get_size;
	iface->get_track = bonfire_transcode_get_track;

	iface->set_output = bonfire_transcode_set_output;
	iface->get_track_type = bonfire_transcode_get_track_type;
	iface->set_output_type = bonfire_transcode_set_output_type;
}

static void
bonfire_transcode_class_init (BonfireTranscodeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_transcode_finalize;

	job_class->get_action_string = bonfire_transcode_get_action_string;
	job_class->set_source = bonfire_transcode_set_source;
	job_class->set_rate = bonfire_transcode_set_rate;
	job_class->start = bonfire_transcode_start;
	job_class->stop = bonfire_transcode_stop;
}

static void
bonfire_transcode_init (BonfireTranscode *obj)
{
	obj->priv = g_new0 (BonfireTranscodePrivate, 1);

	obj->priv->clean = TRUE;
	obj->priv->pipe_out = -1;
}

static void
bonfire_transcode_free_songs (BonfireTranscode *transcode)
{
	BonfireSong *song, *next;

	if (transcode->priv->album) {
		g_free (transcode->priv->album);
		transcode->priv->album = NULL;
	}

	for (song = transcode->priv->songs; song; song = next) {
		next = song->next;
	
		if (song->uri)
			g_free (song->uri);
		if (song->title)
			g_free (song->title);
		if (song->artist)
			g_free (song->artist);
		if (song->composer)
			g_free (song->composer);

		if (song->dest) {
			if (transcode->priv->clean)
				g_remove (song->dest);

			g_free (song->dest);
		}

		if (song->inf) {
			if (transcode->priv->clean)
				g_remove (song->inf);

			g_remove (song->inf);
		}

		g_free (song);
	}

	if (transcode->priv->own_output) {
		g_remove (transcode->priv->output);
		transcode->priv->own_output = 0;
	}

	transcode->priv->songs = NULL;
	transcode->priv->current = NULL;
	transcode->priv->sectors_num = 0;
	transcode->priv->global_pos = 0;

	transcode->priv->inf_ready = 0;
	transcode->priv->audio_ready = 0;
}

static void
bonfire_transcode_finalize (GObject *object)
{
	BonfireTranscode *cobj;

	cobj = BONFIRE_TRANSCODE (object);

	if (cobj->priv->pad_id) {
		g_source_remove (cobj->priv->pad_id);
		cobj->priv->pad_id = 0;
	}

	if (cobj->priv->pipe_out != -1) {
		close (cobj->priv->pipe_out);
		cobj->priv->pipe_out = -1;
	}

	if (cobj->priv->progress_id) {
		g_source_remove (cobj->priv->progress_id);
		cobj->priv->progress_id = 0;
	}

	if (cobj->priv->pipeline) {
		gst_element_set_state (cobj->priv->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (cobj->priv->pipeline));
	}

	bonfire_transcode_free_songs (cobj);

	if (cobj->priv->output) {
		g_free (cobj->priv->output);
		cobj->priv->output = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

BonfireTranscode *
bonfire_transcode_new (void)
{
	BonfireTranscode *obj;

	obj = BONFIRE_TRANSCODE (g_object_new (BONFIRE_TYPE_TRANSCODE,NULL));
	return obj;
}

static BonfireBurnResult
bonfire_transcode_set_rate (BonfireJob *job, gint64 rate)
{
	BonfireTranscode *transcode;

	transcode = BONFIRE_TRANSCODE (job);

	if (transcode->priv->rate == rate)
		return BONFIRE_BURN_OK;

	transcode->priv->rate = rate;
	if (transcode->priv->identity)
		g_object_set (transcode->priv->identity,
			      "datarate", rate,
			      NULL);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_set_source (BonfireJob *job,
			      const BonfireTrackSource *source,
			      GError **error)
{
	BonfireTranscode *transcode;
	BonfireSong *song = NULL;
	GSList *iter;
	int num = 0;

	transcode = BONFIRE_TRANSCODE (job);

	bonfire_transcode_free_songs (transcode);

	g_return_val_if_fail (source != NULL, BONFIRE_BURN_ERR);

	if (source->type != BONFIRE_TRACK_SOURCE_SONG)
		return BONFIRE_BURN_NOT_SUPPORTED;

	transcode->priv->album = g_strdup (source->contents.songs.album);

	transcode->priv->sectors_num = 0;
	transcode->priv->global_pos = 0;
	for (iter = source->contents.songs.files; iter; iter = iter->next) {
		BonfireSongFile *file;

		file = iter->data;
		if (song) {
			song->next = g_new0 (BonfireSong, 1);
			song = song->next;
		}
		else {
			song = g_new0 (BonfireSong, 1);
			transcode->priv->songs = song;
		}

		if (file->gap == FALSE)
			song->index0 = -1;
		if (file->title)
			song->title = g_strdup (file->title);
		if (file->artist)
			song->artist = g_strdup (file->artist);
		if (file->composer)
			song->composer = g_strdup (file->composer);
		if (file->isrc > 0)
			song->isrc = file->isrc;

		song->uri = g_strdup (file->uri);
		song->index = num;
		num ++;
	}

	return BONFIRE_BURN_OK;
}

static void
bonfire_transcode_rm_songs_from_disc (BonfireTranscode *transcode)
{
	BonfireSong *song;

	if (!transcode->priv->clean || !transcode->priv->songs)
		return;

	for (song = transcode->priv->songs; song; song =song->next) {
		if (song->dest) {
			if (transcode->priv->audio_ready)
				g_remove (song->dest);

			g_free (song->dest);
			song->dest = NULL;
		}
		
		if (song->inf) {
			if (transcode->priv->inf_ready)
				g_remove (song->inf);

			g_free (song->inf);
			song->inf = NULL;
		}
	}

	transcode->priv->global_pos = 0;
	transcode->priv->inf_ready = 0;
	transcode->priv->audio_ready = 0;
}

static BonfireBurnResult
bonfire_transcode_set_output (BonfireImager *imager,
			      const char *output,
			      gboolean overwrite,
			      gboolean clean,
			      GError **error)
{
	BonfireTranscode *transcode;

	transcode = BONFIRE_TRANSCODE (imager);

	if (transcode->priv->clean) {
		bonfire_transcode_rm_songs_from_disc (transcode);

		if (transcode->priv->own_output) {
			g_remove (transcode->priv->output);
			transcode->priv->own_output = 0;
		}
	}

	if (transcode->priv->output) {
		g_free (transcode->priv->output);
		transcode->priv->output = NULL;
	}

	/* we check that this is a directory */
	if (output) {
		if (!g_file_test (output, G_FILE_TEST_IS_DIR)) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s is not a directory"),
				     output);
			return BONFIRE_BURN_ERR;
		}

		transcode->priv->output = g_strdup (output);
		transcode->priv->own_output = 0;
	}
	else
		transcode->priv->own_output = 1;

	transcode->priv->overwrite = overwrite;
	transcode->priv->clean = clean;
	
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_set_output_type (BonfireImager *imager, 
				   BonfireTrackSourceType type,
				   GError **error)
{
	BonfireTranscode *transcode;

	transcode = BONFIRE_TRANSCODE (imager);

	if (type == BONFIRE_TRACK_SOURCE_DEFAULT)
		type = BONFIRE_TRACK_SOURCE_AUDIO;
	else if (type != BONFIRE_TRACK_SOURCE_INF
	      &&  type != BONFIRE_TRACK_SOURCE_AUDIO)
		return BONFIRE_BURN_NOT_SUPPORTED;

	transcode->priv->track_type = type;
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_get_track (BonfireImager *imager,
			     BonfireTrackSource **track,
			     GError **error)
{
	BonfireTranscode *transcode;
	BonfireTrackSource *retval;
	BonfireBurnResult result;
	BonfireSong *iter;

	transcode = BONFIRE_TRANSCODE (imager);

	if (transcode->priv->track_type == BONFIRE_TRACK_SOURCE_INF && !transcode->priv->inf_ready) {
		transcode->priv->global_pos = 0;
	
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_INF;
		result = bonfire_job_run (BONFIRE_JOB (imager), error);
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_NONE;
	
		if (result != BONFIRE_BURN_OK) {
			transcode->priv->track_type = BONFIRE_TRACK_SOURCE_UNKNOWN;
			return result;
		}
	}
	else if (transcode->priv->track_type == BONFIRE_TRACK_SOURCE_AUDIO) {
		transcode->priv->global_pos = 0;
	
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_TRANSCODING;
		result = bonfire_job_run (BONFIRE_JOB (imager), error);
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_NONE;
	
		if (result != BONFIRE_BURN_OK) {
			transcode->priv->track_type = BONFIRE_TRACK_SOURCE_UNKNOWN;
			return result;
		}
	}

	retval = g_new0 (BonfireTrackSource, 1);
	retval->type = transcode->priv->track_type;

	if (transcode->priv->track_type == BONFIRE_TRACK_SOURCE_AUDIO) {
		for (iter = transcode->priv->songs; iter; iter = iter->next) {
			retval->contents.audio.files = g_slist_append (retval->contents.audio.files,
								       g_strdup (iter->dest));
		}

		transcode->priv->audio_ready = 1;
		transcode->priv->inf_ready = 1;
	}
	else if (transcode->priv->track_type == BONFIRE_TRACK_SOURCE_INF) {
		for (iter = transcode->priv->songs; iter; iter = iter->next) {
			retval->contents.inf.files = g_slist_append (retval->contents.inf.files,
								     g_strdup (iter->inf));
		}

		transcode->priv->inf_ready = 1;
	}

	*track = retval;
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_get_size (BonfireImager *imager,
			    gint64 *size,
			    gboolean sectors,
			    GError **error)
{
	BonfireTranscode *transcode;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	transcode = BONFIRE_TRANSCODE (imager);

	if (!transcode->priv->songs)
		return BONFIRE_BURN_NOT_READY;

	if (!transcode->priv->sectors_num) {
		/* now we need to be able to return the global size */
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_GETTING_SIZE;
		result = bonfire_job_run (BONFIRE_JOB (transcode), error);
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	if (sectors)
		*size = transcode->priv->sectors_num;
	else
		*size = transcode->priv->sectors_num * 2352;

	return result;
}

static gboolean
bonfire_transcode_get_fraction_done (BonfireTranscode *transcode)
{
	gdouble fraction = 0.0;
	gdouble ave_rate;
	gdouble rate;
	gint64 duration = -1;
	gint64 total;
	gint64 pos = -1;
	gdouble elapsed;
	long secs;

	if (transcode->priv->pipeline) {
		gpointer element;
		GstIterator *iterator;
		GstFormat format = GST_FORMAT_TIME;

		/* this is a workaround : when asking the pipeline especially
		 * a pipeline such as filesrc ! decodebin ! fakesink we can't
		 * get the duration in time or the results are simply crazy.
		 * so we iterate through the elements in decodebin until we
		 * find an element that can tell us the position in this format */
		iterator = gst_bin_iterate_sorted (GST_BIN (transcode->priv->decode));
		while (gst_iterator_next (iterator, &element) == GST_ITERATOR_OK) {
			if (gst_element_query_position (GST_ELEMENT (element),
							 &format,
							 &pos)) {
				gst_object_unref (element);
				break;
			}
			gst_object_unref (element);
		}
		gst_iterator_free (iterator);

		if (pos == -1) {
			bonfire_job_debug_message (BONFIRE_JOB (transcode),
						   "can't get position in the stream");
			return TRUE;
		}

		if (!gst_element_query_duration (transcode->priv->pipeline,
						 &format,
						 &duration)) {
			bonfire_job_debug_message (BONFIRE_JOB (transcode),
						   "can't get duration of the stream");
			return TRUE;
		}

		bonfire_job_debug_message (BONFIRE_JOB (transcode),
					   "got position (%" G_GINT64_FORMAT ") and duration (%" G_GINT64_FORMAT ")",
					   pos,
					   duration);
	}
	else
		return FALSE;

	if (pos > -1 && duration > 0) {
		fraction = (double) pos / (double) duration;

		if (fraction > 1.0)
			fraction = 1.0;
		else if (fraction < 0.0)
			fraction = 0.0;
	}

//	fraction = (gdouble) pos / ((gdouble) GST_FORMAT_PERCENT_MAX * 100.0);
	if (!transcode->priv->sectors_num)
		return TRUE;

	if (!transcode->priv->timer) {
		transcode->priv->timer = g_timer_new ();
		transcode->priv->start_pos = pos;
		return TRUE;
	}

	/* calculate the secs remaining we are called approximately every 200 ms */
	elapsed = g_timer_elapsed (transcode->priv->timer, NULL);
	if (elapsed < 1)
		return TRUE;

	pos = transcode->priv->global_pos + fraction * transcode->priv->current->duration;
	total = (gint64) transcode->priv->sectors_num * 1000000000 / 75;

	fraction = (double) pos / (double) total;

	rate = (double) (pos - transcode->priv->start_pos) / elapsed;
	ave_rate = bonfire_burn_common_get_average_rate (&transcode->priv->rates, rate);

	if (ave_rate > 1) {
		secs = ceil ((total - pos) / ave_rate);
		bonfire_job_progress_changed (BONFIRE_JOB (transcode), fraction, secs);
	}

	return TRUE;
}

static gboolean
bonfire_transcode_create_pipeline (BonfireTranscode *transcode, GError **error)
{
	GstElement *pipeline;
	GstCaps *filtercaps;
	GstElement *convert = NULL;
	GstElement *resample = NULL;
	GstElement *source;
	GstElement *identity = NULL;
	GstElement *decode;
	GstElement *filter = NULL;
	GstElement *sink = NULL;
	BonfireSong *song;

	song = transcode->priv->current;
	if (!song)
		return FALSE;

	/* free the possible current pipeline and create a new one */
	if (transcode->priv->pipeline) {
		gst_element_set_state (transcode->priv->pipeline, GST_STATE_NULL);
		gst_object_unref (G_OBJECT (transcode->priv->pipeline));
		transcode->priv->pipeline = NULL;
		transcode->priv->sink = NULL;
		transcode->priv->source = NULL;
		transcode->priv->convert = NULL;
		transcode->priv->identity = NULL;
		transcode->priv->pipeline = NULL;
	}

	/* create three types of pipeline according to the needs:
	 * - filesrc ! decodebin ! audioconvert ! fakesink
	 * - filesrc ! decodebin ! audioresample ! audioconvert ! audio/x-raw-int,rate=44100,width=16,depth=16,endianness=4321,signed ! filesink
	 * - filesrc ! decodebin ! audioresample ! audioconvert ! audio/x-raw-int,rate=44100,width=16,depth=16,endianness=4321,signed ! fdsink
	 */
	pipeline = gst_pipeline_new (NULL);
	gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)),
			   (GstBusFunc) bonfire_transcode_bus_messages,
			   transcode);

	/* source */
	source = gst_element_make_from_uri (GST_URI_SRC,
					    song->uri,
					    NULL);
	if (source == NULL) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("source can't be created"));
		goto error;
	}
	gst_bin_add (GST_BIN (pipeline), source);
	g_object_set (source,
		      "typefind", FALSE,
		      NULL);

	/* sink */
	switch (transcode->priv->action) {
	case BONFIRE_TRANSCODE_ACTION_GETTING_SIZE:
		sink = gst_element_factory_make ("fakesink", NULL);
		break;

	case BONFIRE_TRANSCODE_ACTION_INF:
		sink = gst_element_factory_make ("fakesink", NULL);
		break;

	case BONFIRE_TRANSCODE_ACTION_TRANSCODING:
		if (!transcode->priv->on_the_fly) {
			char *path;

			path = g_strdup_printf ("%s/Track%02i.cdr",
						transcode->priv->output,
						song->index);
		
			if (!transcode->priv->overwrite && g_file_test (path, G_FILE_TEST_EXISTS)) {
				g_set_error (error,
					     BONFIRE_BURN_ERROR,
					     BONFIRE_BURN_ERROR_FILE_EXIST,
					     _("%s already exist (can't overwrite)"),
					     path);
		
				g_free (path);
				return BONFIRE_BURN_ERR;
			}

			sink = gst_element_factory_make ("filesink", NULL);
			g_object_set (sink,
				      "location", path,
				      NULL);
			song->dest = path;
		}
		else {
			sink = gst_element_factory_make ("fdsink", NULL);
			g_object_set (sink,
				      "fd", transcode->priv->pipe_out,
				      NULL);
		}
		break;

	default:
		goto error;
	}

	if (!sink) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("sink can't be created"));
		goto error;
	}
	gst_bin_add (GST_BIN (pipeline), sink);
	g_object_set (sink,
		      "sync", FALSE,
		      NULL);

	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING) {
		/* identity to control rate of data */
/*		identity = gst_element_factory_make ("identity", NULL);
		if (identity == NULL) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("identity can't be created"));
			goto error;
		}
		gst_bin_add (GST_BIN (pipeline), identity);
		if (transcode->priv->rate)
			g_object_set (identity,
				      "datarate", transcode->priv->rate,
				      "sync", TRUE,
				      NULL);

		/* audioresample */
		resample = gst_element_factory_make ("audioresample", NULL);
		if (resample == NULL) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("audioresample can't be created"));
			goto error;
		}
		gst_bin_add (GST_BIN (pipeline), resample);

		/* audioconvert */
		convert = gst_element_factory_make ("audioconvert", NULL);
		if (convert == NULL) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("audioconvert can't be created"));
			goto error;
		}
		gst_bin_add (GST_BIN (pipeline), convert);

		/* filter */
		filter = gst_element_factory_make ("capsfilter", NULL);
		if (!filter) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("filter can't be created"));
			goto error;
		}
		gst_bin_add (GST_BIN (pipeline), filter);
		filtercaps = gst_caps_new_full (gst_structure_new ("audio/x-raw-int",
								   "channels", G_TYPE_INT, 2,
								   "width", G_TYPE_INT, 16,
								   "depth", G_TYPE_INT, 16,
								   "endianness", G_TYPE_INT, 4321,
								   "rate", G_TYPE_INT, 44100,
								   "signed", G_TYPE_BOOLEAN, TRUE,
								   NULL),
						NULL);
		g_object_set (GST_OBJECT (filter), "caps", filtercaps, NULL);
		gst_caps_unref (filtercaps);
	}

	/* decode */
	decode = gst_element_factory_make ("decodebin", NULL);
	if (decode == NULL) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("decode can't be created"));
		goto error;
	}
	gst_bin_add (GST_BIN (pipeline), decode);
	transcode->priv->decode = decode;

	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING) {
		if (transcode->priv->on_the_fly)
			transcode->priv->sink = sink;

		gst_element_link_many (source, /* identity, */ decode, NULL);
		g_signal_connect (G_OBJECT (decode),
				  "new-decoded-pad",
				  G_CALLBACK (bonfire_transcode_new_decoded_pad_cb),
				  resample);
		gst_element_link_many (resample,
				       convert,
				       filter,
				       sink,
				       NULL);
	}
	else {
		gst_element_link (source, decode);
		g_signal_connect (G_OBJECT (decode),
				  "new-decoded-pad",
				  G_CALLBACK (bonfire_transcode_new_decoded_pad_cb),
				  sink);
	}

	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_GETTING_SIZE)
		bonfire_job_action_changed (BONFIRE_JOB (transcode),
					    BONFIRE_BURN_ACTION_GETTING_SIZE, 
					    TRUE);
	else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF)
		bonfire_job_action_changed (BONFIRE_JOB (transcode),
					    BONFIRE_BURN_ACTION_ANALYSING,
					    TRUE);
	else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING)
		bonfire_job_action_changed (BONFIRE_JOB (transcode),
					    BONFIRE_BURN_ACTION_TRANSCODING,
					    TRUE);

	transcode->priv->source = source;
	transcode->priv->convert = convert;
	transcode->priv->identity = identity;
	transcode->priv->pipeline = pipeline;

	gst_element_set_state (transcode->priv->pipeline, GST_STATE_PLAYING);
	return TRUE;

error:
	if (error)
		bonfire_job_debug_message (BONFIRE_JOB (transcode),
					   "can't create object : %s \n",
					   (*error)->message);

	gst_object_unref (GST_OBJECT (pipeline));
	return FALSE;
}

static gboolean
bonfire_transcode_write_inf (BonfireTranscode *transcode,
			     BonfireSong *song,
			     GError **error)
{
	int fd;
	int size;
	char *path;
	char *string;
	int b_written;
	char buffer [128];

	/* NOTE about the .inf files: they should have the exact same path
	 * but the ending suffix file is replaced by inf.
	 * example : /path/to/file.mp3 => /path/to/file.inf */
	path = g_strdup_printf ("%s/Track%02i.inf",
				transcode->priv->output,
				song->index);

	if (!transcode->priv->overwrite && g_file_test (path, G_FILE_TEST_EXISTS)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("%s already exist (can't overwrite)"),
			     path);
		g_free (path);
		return FALSE;
	}

	fd = open (path, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
	if (fd < 0)
		goto error;

	strcpy (buffer, "# created by bonfire\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	strcpy (buffer, "MCN=\t\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	if (song->isrc > 0)
		string = g_strdup_printf ("ISRC=\t%i\n", song->isrc);
	else
		string = g_strdup ("ISRC=\t\n");
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Albumperformer=\t\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	if (transcode->priv->album)
		string = g_strdup_printf ("Albumtitle=\t%s\n", transcode->priv->album);
	else
		string = strdup ("Albumtitle=\t\n");
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	if (song->artist)
		string = g_strdup_printf ("Performer=\t%s\n", song->artist);
	else
		string = strdup ("Performer=\t\n");
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	if (song->composer)
		string = g_strdup_printf ("Composer=\t%s\n", song->composer);
	else
		string = strdup ("Composer=\t\n");
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	if (song->title)
		string = g_strdup_printf ("Tracktitle=\t%s\n", song->title);
	else
		string = strdup ("Tracktitle=\t\n");
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	string = g_strdup_printf ("Tracknumber=\t%i\n", song->index);
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	string = g_strdup_printf ("Trackstart=\t%i\n", song->start);
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	string = g_strdup_printf ("Tracklength=\t%i, 0\n", song->sectors);
	size = strlen (string);
	b_written = write (fd, string, size);
	g_free (string);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Pre-emphasis=\tyes\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Channels=\t2\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Copy_permitted=\tyes\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Endianess=\tlittle\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	strcpy (buffer, "Index=\t\t0\n");
	size = strlen (buffer);
	b_written = write (fd, buffer, size);
	if (b_written != size)
		goto error;

	string = g_strdup_printf ("Index0=\t\t%i\n", song->sectors);
	size = strlen (string);
	b_written = write (fd, string, size);
	if (b_written != size)
		goto error;

	close (fd);
	song->inf = path;

	return TRUE;


error:
	g_remove (path);
	g_free (path);

	g_set_error (error,
		     BONFIRE_BURN_ERROR,
		     BONFIRE_BURN_ERROR_GENERAL,
		     _("the inf file can't be written : %s"), 
		     strerror (errno));
	return FALSE;
}

static void
bonfire_transcode_set_inf (BonfireTranscode *transcode,
			   BonfireSong *song,
			   gint64 duration)
{
	gint64 start;
	gint64 sectors;
	BonfireSong *iter;

	/* 1 sec = 75 sectors = 2352 bytes */
	sectors = duration * 75;
	if (sectors % 1000000000)
		sectors = sectors / 1000000000 + 1;
	else
		sectors /= 1000000000;

	/* if it's on the fly we add 2 sector for security since gstreamer is
	 * not really reliable when it comes to getting an accurate duration */
	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF)
		sectors += 2;

	/* index0 != -1 means a gap is required after the song */
	if (song->index0 >= 0) {
		song->index0 = sectors;
		sectors += 150;
	}

	/* if transcoding on the fly we should add some length just to make
	 * sure we won't be too short (gstreamer duration discrepancy) */
	transcode->priv->sectors_num -= song->sectors;
	transcode->priv->sectors_num += sectors;

	song->sectors = sectors;
	song->duration = duration;

	start = 0;
	for (iter = transcode->priv->songs; iter && iter != song; iter = iter->next)
		start += iter->sectors;

	song->start = start;

	bonfire_job_debug_message (BONFIRE_JOB (transcode),
				   "Song %s"
				   "\nsectors %" G_GINT64_FORMAT
				   "\ntime %" G_GINT64_FORMAT 
				   "\nTOTAL %i sectors\n",
				   song->uri,
				   sectors,
				   duration,
				   transcode->priv->sectors_num);
}

static BonfireBurnResult
bonfire_transcode_create_inf_siblings (BonfireTranscode *transcode,
				       BonfireSong *src,
				       BonfireSong *dest,
				       GError **error)
{
	/* it means the same file uri is in the selection and
	 * was already created. Simply get the values for the 
	 * inf file from the already created one and write the
	 * inf file */
	bonfire_transcode_set_inf (transcode, dest, src->duration);

	/* let's not forget the tags */
	if (!dest->artist)
		dest->artist = g_strdup (src->artist);
	if (!dest->composer)
		dest->composer = g_strdup (src->composer);
	if (!dest->title)
		dest->title = g_strdup (src->title);

	/* since we skip this song we update the position */
	transcode->priv->global_pos += src->duration;

	if (!bonfire_transcode_write_inf (transcode, dest, error))
		return BONFIRE_BURN_ERR;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_create_song_siblings (BonfireTranscode *transcode,
					BonfireSong *src,
					BonfireSong *dest,
					GError **error)
{
	char *path_dest;
	char *path_src;

	/* it means the file is already in the selection.
	 * Simply create a symlink pointing to the first
	 * file in the selection with the same uri */
	path_src = g_strdup_printf ("%s/Track%02i.cdr",
				    transcode->priv->output,
				    src->index);

	path_dest = g_strdup_printf ("%s/Track%02i.cdr",
				     transcode->priv->output,
				     dest->index);

	/* check that path_dest doesn't exist */
	if (g_file_test (path_dest, G_FILE_TEST_EXISTS)) {
		if (!transcode->priv->overwrite) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_FILE_EXIST,
				     _("%s already exist"),
				     path_dest);
			goto error;			
		}
		else
			g_remove (path_dest);
	}

	if (symlink (path_src, path_dest) == -1) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("a symlink could not be created (%s)"),
			     strerror (errno));

		goto error;
	}

	g_free (path_src);
	dest->dest = path_dest;

	/* now we generate the associated inf path */
	return bonfire_transcode_create_inf_siblings (transcode, src, dest, error);

error:
	g_free (path_src);
	g_free (path_dest);

	return BONFIRE_BURN_ERR;
}

static BonfireSong *
bonfire_transcode_find_sibling (BonfireTranscode *transcode,
				BonfireSong *song)
{
	BonfireSong *sibling;

	sibling = transcode->priv->songs;
	for (; sibling && sibling != song; sibling = sibling->next) {
		if (!strcmp (sibling->uri, song->uri))
			return sibling;
	}

	return NULL;
}

static gboolean
bonfire_transcode_check_tmpdir (BonfireTranscode *transcode, GError **error)
{
	char *folder;

	if (transcode->priv->output)
		return TRUE;

	folder = g_strdup_printf ("%s/bonfireXXXXXX", g_get_tmp_dir ());
	transcode->priv->output = mkdtemp (folder);

	if (!transcode->priv->output) {
		g_free (folder);
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_TMP_DIR,
			     _("a temporary directory could not be created"));
		return FALSE;
	}

	bonfire_job_debug_message (BONFIRE_JOB (transcode),
				   "created temporary directory %s",
				   transcode->priv->output);

	transcode->priv->own_output = 1;

	return TRUE;
}

static BonfireBurnResult
bonfire_transcode_start (BonfireJob *job,
			 int in_fd,
			 int *out_fd,
			 GError **error)
{
	BonfireTranscode *transcode;

	if (in_fd > 0)
		return BONFIRE_BURN_NOT_SUPPORTED;

	transcode = BONFIRE_TRANSCODE (job);

	if (!transcode->priv->songs)
		return BONFIRE_BURN_NOT_READY;

	transcode->priv->global_pos = 0;
	transcode->priv->current = transcode->priv->songs;

	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_GETTING_SIZE) {
		transcode->priv->on_the_fly = 0;

		if (!bonfire_transcode_create_pipeline (transcode, error))
			return BONFIRE_BURN_ERR;
	}
	else if (out_fd) {
		int pipe_out [2];
		long flags = 0;

		transcode->priv->track_type = BONFIRE_TRACK_SOURCE_AUDIO;
		transcode->priv->action = BONFIRE_TRANSCODE_ACTION_TRANSCODING;

		/* now we generate the data, piping it to cdrecord presumably */
		if (pipe (pipe_out)) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("the pipe couldn't be created (%s)"),
				     strerror (errno));
			return BONFIRE_BURN_ERR;
		}

		if (fcntl (pipe_out [0], F_GETFL, &flags) != -1) {
			flags |= O_NONBLOCK;
			if (fcntl (pipe_out [1], F_SETFL, flags) == -1)
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "couldn't set non blocking mode");
		}
		else
			bonfire_job_debug_message (BONFIRE_JOB (transcode),
						   "couldn't get pipe flags");
		flags = 0;
		if (fcntl (pipe_out [1], F_GETFL, &flags) != -1) {
			flags |= O_NONBLOCK;
			if (fcntl (pipe_out [1], F_SETFL, flags) == -1)
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "couldn't set non blocking mode");
		}
		else
			bonfire_job_debug_message (BONFIRE_JOB (transcode),
						   "couldn't get pipe flags");

		transcode->priv->pipe_out = pipe_out [1];
		transcode->priv->on_the_fly = 1;

		if (!bonfire_transcode_create_pipeline (transcode, error)) {
			close (pipe_out [0]);
			close (pipe_out [1]);
			return BONFIRE_BURN_ERR;
		}

		*out_fd = pipe_out [0];
	}
	else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING) {
		if (!bonfire_transcode_check_tmpdir (transcode, error))
			return BONFIRE_BURN_ERR;

		transcode->priv->on_the_fly = 0;
		if (!bonfire_transcode_create_pipeline (transcode, error))
			return BONFIRE_BURN_ERR;
	}
	else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF) {
		if (!bonfire_transcode_check_tmpdir (transcode, error))
			return BONFIRE_BURN_ERR;

		if (!bonfire_transcode_create_pipeline (transcode, error))
			return BONFIRE_BURN_ERR;
	}
	else {
		bonfire_job_debug_message (BONFIRE_JOB (transcode), "don't know what to do");
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	return BONFIRE_BURN_OK;
}

static void
bonfire_transcode_stop_pipeline (BonfireTranscode *transcode)
{
	if (transcode->priv->progress_id) {
		g_source_remove (transcode->priv->progress_id);
		transcode->priv->progress_id = 0;
	}

	if (transcode->priv->pipeline) {
		gst_element_set_state (transcode->priv->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (transcode->priv->pipeline));
		transcode->priv->pipeline = NULL;
		transcode->priv->sink = NULL;
		transcode->priv->source = NULL;
		transcode->priv->convert = NULL;
		transcode->priv->identity = NULL;
		transcode->priv->pipeline = NULL;
	}
}

static BonfireBurnResult
bonfire_transcode_stop (BonfireJob *job,
			BonfireBurnResult retval)
{
	BonfireTranscode *transcode;

	transcode = BONFIRE_TRANSCODE (job);

	transcode->priv->current = NULL;

	if (transcode->priv->pad_id) {
		g_source_remove (transcode->priv->pad_id);
		transcode->priv->pad_id = 0;
	}

	bonfire_transcode_stop_pipeline (transcode);

	if (transcode->priv->pipe_out != -1) {
		close (transcode->priv->pipe_out);
		transcode->priv->pipe_out = -1;
	}

	if (transcode->priv->timer) {
		g_timer_destroy (transcode->priv->timer);
		transcode->priv->timer = NULL;
	}

	if (transcode->priv->rates) {
		g_slist_free (transcode->priv->rates);
		transcode->priv->rates = NULL;
	}

	transcode->priv->action = BONFIRE_TRANSCODE_ACTION_NONE;
	return retval;
}

static gint64
bonfire_transcode_pad_real (BonfireTranscode *transcode,
			    int fd,
			    gint64 bytes2write,
			    GError **error)
{
	const int buffer_size = 512;
	char buffer [buffer_size];
	gint64 b_written;
	gint64 size;

	b_written = 0;
	bzero (buffer, sizeof (buffer));
	for (; bytes2write; bytes2write -= b_written) {
		size = bytes2write > buffer_size ? buffer_size : bytes2write;
		b_written = write (fd, buffer, (int) size);

		bonfire_job_debug_message (BONFIRE_JOB (transcode),
					   "written %" G_GINT64_FORMAT " bytes for padding",
					   b_written);

		/* we should not handle EINTR and EAGAIN as errors */
		if (b_written < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "got EINTR / EAGAIN, retrying");
	
				/* we'll try later again */
				return bytes2write;
			}
		}

		if (size != b_written) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("error padding (%s)."),
				     strerror (errno));
			return -1;
		}
	}

	return 0;
}

static gboolean
bonfire_transcode_pad_idle (BonfireTranscode *transcode)
{
	gint64 bytes2write;
	GError *error = NULL;
	BonfireBurnResult result;

	bytes2write = bonfire_transcode_pad_real (transcode,
						  transcode->priv->pipe_out,
						  transcode->priv->pad_size,
						  &error);

	if (bytes2write == -1) {
		transcode->priv->pad_id = 0;
		bonfire_job_error (BONFIRE_JOB (transcode), error);
		return FALSE;
	}

	if (bytes2write) {
		transcode->priv->pad_size = bytes2write;
		return TRUE;
	}

	/* we are finished with padding */
	transcode->priv->pad_id = 0;
	if (!transcode->priv->on_the_fly) {
		close (transcode->priv->pipe_out);
		transcode->priv->pipe_out = -1;
	}

	/* set the next song or finish */
	result = bonfire_transcode_next_song (transcode, &error);
	if (result != BONFIRE_BURN_OK) {
		bonfire_job_error (BONFIRE_JOB (transcode), error);
		return FALSE;
	}

	if (!transcode->priv->current) {
		bonfire_job_finished (BONFIRE_JOB (transcode));
		return FALSE;
	}

	return FALSE;
}

static gint64
bonfire_transcode_get_duration (BonfireTranscode *transcode)
{
	GstElement *element;
	gint64 duration = -1;
	GstFormat format = GST_FORMAT_TIME;
	
	/* this is the most reliable way to get the duration in bytes of any song */
	if (transcode->priv->convert)
		element = transcode->priv->convert;
	else
		element = transcode->priv->pipeline;

	gst_element_query_position (GST_ELEMENT (element),
				    &format,
				    &duration);

	/* That could happen I don't know why but better have something */
	if (duration == -1)
		gst_element_query_duration (GST_ELEMENT (element),
					    &format,
					    &duration);

	if (duration == -1)	
	    bonfire_job_error (BONFIRE_JOB (transcode),
			       g_error_new (BONFIRE_BURN_ERROR,
					    BONFIRE_BURN_ERROR_GENERAL,
					    _("error getting duration")));
	return duration;
}

/* we must make sure that the track size is a multiple
 * of 2352 to be burnt by cdrecord with on the fly */
static gboolean
bonfire_transcode_pad (BonfireTranscode *transcode, int fd, GError **error)
{
	gint64 duration;
	gint64 b_written;
	gint64 bytes2write;

	duration = bonfire_transcode_get_duration (transcode);
	if (duration == -1)
		return FALSE;

	/* we need to comply more or less to what we told
	 * cdrecord about the size of the track in sectors */
	b_written = duration * 75 * 2352;
	b_written = b_written % 1000000000 ? b_written / 1000000000 + 1 : b_written / 1000000000;
	bytes2write = transcode->priv->current->sectors * 2352 - b_written;

	bonfire_job_debug_message (BONFIRE_JOB (transcode),
				   "Padding\t= %" G_GINT64_FORMAT 
				   "\n\t\t\t\t%i"
				   "\n\t\t\t\t%" G_GINT64_FORMAT,
				   bytes2write,
				   transcode->priv->current->sectors,
				   b_written);

	bytes2write = bonfire_transcode_pad_real (transcode,
						  fd,
						  bytes2write,
						  error);
	if (bytes2write == -1)
		return FALSE;

	if (bytes2write) {
		/* when writing to a pipe it can happen that its buffer is full
		 * because cdrecord is not fast enough. Therefore we couldn't
		 * write/pad it and we'll have to wait for the pipe to become
		 * available again */
		transcode->priv->pipe_out = fd;
		transcode->priv->pad_size = bytes2write;
		transcode->priv->pad_id = g_timeout_add (50,
							 (GSourceFunc) bonfire_transcode_pad_idle,
							 transcode);
		return FALSE;		
	}

	return TRUE;
}

static gboolean
bonfire_transcode_pad_pipe (BonfireTranscode *transcode, GError **error)
{
	return bonfire_transcode_pad (transcode, transcode->priv->pipe_out, error);
}

static gboolean
bonfire_transcode_pad_file (BonfireTranscode *transcode, GError **error)
{
	int fd;
	gboolean result;

	fd = open (transcode->priv->current->dest, O_WRONLY | O_CREAT | O_APPEND);
	if (fd == -1) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("error opening file for padding : %s."),
			     strerror (errno));
		return FALSE;
	}

	result = bonfire_transcode_pad (transcode, fd, error);
	if (result)
		close (fd);

	return result;
}

static gboolean
bonfire_transcode_set_current_song_start (BonfireTranscode *transcode, 
					  GError **error)
{
	gint64 duration;
	GstFormat format = GST_FORMAT_TIME;

	/* now we set the length in sectors */
	if (gst_element_query_duration (GST_ELEMENT (transcode->priv->pipeline),
					&format,
					&duration) == FALSE) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("error getting duration"));
		return FALSE;
	}

	bonfire_transcode_set_inf (transcode, transcode->priv->current, duration);
	return TRUE;
}

static BonfireBurnResult
bonfire_transcode_next_song (BonfireTranscode *transcode, GError **error)
{
	BonfireSong *song;
	BonfireSong *sibling = NULL;

	/* make sure the pipeline is really stopped */
	bonfire_transcode_stop_pipeline (transcode);

	if (!transcode->priv->current->next) {
		transcode->priv->current = NULL;

		if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING
		||  transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF)
			bonfire_job_progress_changed (BONFIRE_JOB (transcode),
						      1.0,
						      -1);
		return BONFIRE_BURN_OK;
	}

	song = transcode->priv->current->next;
	transcode->priv->current = song;
	if (!song)
		return BONFIRE_BURN_OK;

	if (!transcode->priv->on_the_fly)
		sibling = bonfire_transcode_find_sibling (transcode, song);

	if (sibling) {
		BonfireBurnResult result;

		bonfire_job_debug_message (BONFIRE_JOB (transcode),
					   "found sibling for %s : skipping\n",
					   song->uri);
		if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING
		&&  !transcode->priv->on_the_fly) {
			result = bonfire_transcode_create_song_siblings (transcode,
									 sibling,
									 song,
									 error);
			if (result != BONFIRE_BURN_OK)
				return result;

			return bonfire_transcode_next_song (transcode, error);
		}
		else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF) {
			result = bonfire_transcode_create_inf_siblings (transcode,
									sibling,
									song,
									error);
			if (result != BONFIRE_BURN_OK)
				return result;

			return bonfire_transcode_next_song (transcode, error);
		}
		else if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_GETTING_SIZE) {
			bonfire_transcode_set_inf (transcode, song, sibling->duration);
			return bonfire_transcode_next_song (transcode, error);
		}	
	}

	if (!bonfire_transcode_create_pipeline (transcode, error))
		return BONFIRE_BURN_ERR;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_song_end_reached (BonfireTranscode *transcode)
{
	GError *error = NULL;
	BonfireBurnResult result;

	transcode->priv->global_pos += transcode->priv->current->duration;
	if (!transcode->priv->on_the_fly) {
		gint64 duration;

		duration = bonfire_transcode_get_duration (transcode);
		if (duration == -1)
			return FALSE;

		bonfire_transcode_set_inf (transcode,
					   transcode->priv->current,
					   duration);

		/* write the inf */
		if (!bonfire_transcode_write_inf (transcode, transcode->priv->current, &error)) {
			bonfire_job_error (BONFIRE_JOB (transcode), error);
			return FALSE;
		}
	}

	if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_TRANSCODING) {
		gboolean result;

		/* pad file so it is a multiple of 2352 (= 1 sector) */
		if (transcode->priv->pipe_out != -1)
			result = bonfire_transcode_pad_pipe (transcode, &error);
		else
			result = bonfire_transcode_pad_file (transcode, &error);
	
		if (error) {
			bonfire_job_error (BONFIRE_JOB (transcode), error);
			return FALSE;
		}
		if (!result) {
			bonfire_transcode_stop_pipeline (transcode);
			return FALSE;
		}
	}

	/* set the next song */
	result = bonfire_transcode_next_song (transcode, &error);
	if (result != BONFIRE_BURN_OK) {
		bonfire_job_error (BONFIRE_JOB (transcode), error);
		return FALSE;
	}

	if (!transcode->priv->current) {
		bonfire_job_finished (BONFIRE_JOB (transcode));
		return FALSE;
	}

	return TRUE;
}

static void
foreach_tag (const GstTagList *list,
	     const char *tag,
	     BonfireTranscode *transcode)
{
	if (!strcmp (tag, GST_TAG_TITLE)) {
		if (!transcode->priv->current->title)
			gst_tag_list_get_string (list, tag, &(transcode->priv->current->title));
	}
	else if (!strcmp (tag, GST_TAG_ARTIST)) {
		if (!transcode->priv->current->artist)
			gst_tag_list_get_string (list, tag, &(transcode->priv->current->artist));
	}
	else if (!strcmp (tag, GST_TAG_ISRC)) {
		gst_tag_list_get_int (list, tag, &(transcode->priv->current->isrc));
	}
	else if (!strcmp (tag, GST_TAG_PERFORMER)) {
		if (!transcode->priv->current->artist)
			gst_tag_list_get_string (list, tag, &(transcode->priv->current->artist));
	}
}

static gboolean
bonfire_transcode_bus_messages (GstBus *bus,
				GstMessage *msg,
				BonfireTranscode *transcode)
{
	GstTagList *tags = NULL;
	GError *error = NULL;
	GstState state;
	char *debug;

	switch (GST_MESSAGE_TYPE (msg)) {
	case GST_MESSAGE_TAG:
		/* we use the information to write an .inf file 
		 * for the time being just store the information */
		gst_message_parse_tag (msg, &tags);
		gst_tag_list_foreach (tags, (GstTagForeachFunc) foreach_tag, transcode);
		gst_tag_list_free (tags);
		return TRUE;

	case GST_MESSAGE_ERROR:
		gst_message_parse_error (msg, &error, &debug);
		bonfire_job_debug_message (BONFIRE_JOB (transcode), debug);
		g_free (debug);

	        bonfire_job_error (BONFIRE_JOB (transcode), error);
		return FALSE;

	case GST_MESSAGE_EOS:
		return bonfire_transcode_song_end_reached (transcode);

	case GST_MESSAGE_STATE_CHANGED: {
		GstStateChangeReturn result;

		result = gst_element_get_state (transcode->priv->pipeline,
						&state,
						NULL,
						1);

		if (result != GST_STATE_CHANGE_SUCCESS)
			return TRUE;

		if (state != GST_STATE_PAUSED && state != GST_STATE_PLAYING)
			return TRUE;

		if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_GETTING_SIZE) {
			GError *error = NULL;

			if (!bonfire_transcode_set_current_song_start (transcode, &error)) {
				bonfire_job_error (BONFIRE_JOB (transcode), error);
				return FALSE;
			}

			result = bonfire_transcode_next_song (transcode, &error);
			if (result != BONFIRE_BURN_OK)
				bonfire_job_error (BONFIRE_JOB (transcode), error);
			else if (!transcode->priv->current)
				bonfire_job_finished (BONFIRE_JOB (transcode));

			return FALSE;
		}

		if (state == GST_STATE_PLAYING && !transcode->priv->progress_id) {
			if (transcode->priv->action == BONFIRE_TRANSCODE_ACTION_INF)
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "start generating inf %s/Track%02i.inf for %s",
							   transcode->priv->output,
							   transcode->priv->current->index,
							   transcode->priv->current->uri);
			else if (transcode->priv->on_the_fly)
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "start piping %s",
							   transcode->priv->current->uri);
			else
				bonfire_job_debug_message (BONFIRE_JOB (transcode),
							   "start decoding %s to %s",
							   transcode->priv->current->uri,
							   transcode->priv->current->dest);

			transcode->priv->progress_id = g_timeout_add (200,
								      (GSourceFunc) bonfire_transcode_get_fraction_done,
								      transcode);
		}

		return TRUE;
	}

	default:
		return TRUE;
	}

	return TRUE;
}

static void
bonfire_transcode_new_decoded_pad_cb (GstElement *decode,
				      GstPad *pad,
				      gboolean arg2,
				      GstElement *convert)
{
	GstPad *sink;
	GstCaps *caps;
	GstStructure *structure;

	sink = gst_element_get_pad (convert, "sink");
	if (GST_PAD_IS_LINKED (sink))
		return;

	/* make sure we only have audio */
	caps = gst_pad_get_caps (pad);
	structure = gst_caps_get_structure (caps, 0);
	if (g_strrstr (gst_structure_get_name (structure), "audio"))
		gst_pad_link (pad, sink);

	gst_object_unref (sink);
	gst_caps_unref (caps);
}

static BonfireBurnResult
bonfire_transcode_get_track_type (BonfireImager *imager,
				  BonfireTrackSourceType *type)
{
	BonfireTranscode *transcode;

	g_return_val_if_fail (type != NULL, BONFIRE_BURN_ERR);

	transcode = BONFIRE_TRANSCODE (imager);

	if (transcode->priv->track_type == BONFIRE_TRACK_SOURCE_UNKNOWN)
		return BONFIRE_BURN_NOT_READY;

	*type = transcode->priv->track_type;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_transcode_get_action_string (BonfireJob *job,
				     BonfireBurnAction action,
				     char **string)
{
	BonfireTranscode *transcode;

	transcode = BONFIRE_TRANSCODE (job);

	if (action == BONFIRE_BURN_ACTION_TRANSCODING && transcode->priv->current->uri) {
		char *name;

		name = g_path_get_basename (transcode->priv->current->uri);
		*string = g_strdup_printf (_("Transcoding \"%s\""), name);
		g_free (name);
	}
	else if (action == BONFIRE_BURN_ACTION_ANALYSING && transcode->priv->current->uri) {
		char *name;

		name = g_path_get_basename (transcode->priv->current->uri);
		*string = g_strdup_printf (_("Analysing \"%s\""), name);
		g_free (name);
	}
	else {
		const char *tmp;

		tmp = bonfire_burn_action_to_string (action);
		*string = g_strdup (tmp);
	}

	return BONFIRE_BURN_OK;
}
