/***************************************************************************
 *            readcd.c
 *
 *  dim jan 22 18:06:10 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-common.h"
#include "burn-readcd.h"
#include "burn-imager.h"
#include "burn-process.h"
#include "burn-job.h"

static void bonfire_readcd_class_init (BonfireReadcdClass *klass);
static void bonfire_readcd_init (BonfireReadcd *sp);
static void bonfire_readcd_finalize (GObject *object);
static void bonfire_readcd_iface_init_image (BonfireImagerIFace *iface);

static BonfireBurnResult
bonfire_readcd_read_stderr (BonfireProcess *process, const char *line);
static BonfireBurnResult
bonfire_readcd_set_argv (BonfireProcess *process,
			 GPtrArray *argv,
			 gboolean has_master,
			 GError **error);
static BonfireBurnResult
bonfire_readcd_post (BonfireProcess *process, BonfireBurnResult retval);

static BonfireBurnResult
bonfire_readcd_set_source (BonfireJob *job,
			   const BonfireTrackSource *source,
			   GError **error);
static BonfireBurnResult
bonfire_readcd_set_output (BonfireImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error);
static BonfireBurnResult
bonfire_readcd_set_output_type (BonfireImager *imager,
				BonfireTrackSourceType type,
				GError **error);
static BonfireBurnResult
bonfire_readcd_get_track (BonfireImager *imager,
			  BonfireTrackSource **track,
			  GError **error);
static BonfireBurnResult
bonfire_readcd_get_size (BonfireImager *imager,
			 gint64 *size,
			 gboolean sectors,
			 GError **error);
static BonfireBurnResult
bonfire_readcd_get_speed (BonfireImager *imager,
			  int *speed);
static BonfireBurnResult
bonfire_readcd_get_track_type (BonfireImager *imager,
			       BonfireTrackSourceType *track_type);

typedef enum {
	BONFIRE_READCD_ACTION_IMAGING,
	BONFIRE_READCD_ACTION_GETTING_SIZE,
	BONFIRE_READCD_ACTION_NONE
} BonfireReadcdAction;

struct BonfireReadcdPrivate {
	BonfireReadcdAction action;
	GTimer *timer;

	BonfireTrackSourceType track_type;
	NautilusBurnDrive *drive;

	GSList *rates;
	gint64 sectors_num;
	gint64 prev_sect;
	gint64 speed;
	int start;

	char *toc;
	char *output;

	int overwrite:1;
	int clean:1;

	int track_ready:1;
	int is_DVD:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_readcd_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireReadcdClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_readcd_class_init,
			NULL,
			NULL,
			sizeof (BonfireReadcd),
			0,
			(GInstanceInitFunc)bonfire_readcd_init,
		};

		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) bonfire_readcd_iface_init_image,
			NULL,
			NULL
		};

		type = g_type_register_static (BONFIRE_TYPE_PROCESS,
					       "BonfireReadcd",
					       &our_info,
					       0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
	}

	return type;
}

static void
bonfire_readcd_class_init (BonfireReadcdClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_readcd_finalize;

	job_class->set_source = bonfire_readcd_set_source;

	process_class->stderr_func = bonfire_readcd_read_stderr;
	process_class->set_argv = bonfire_readcd_set_argv;
	process_class->post = bonfire_readcd_post;
}

static void
bonfire_readcd_iface_init_image (BonfireImagerIFace *iface)
{
	iface->set_output_type = bonfire_readcd_set_output_type;
	iface->set_output = bonfire_readcd_set_output;

	iface->get_track_type = bonfire_readcd_get_track_type;
	iface->get_track = bonfire_readcd_get_track;
	iface->get_speed = bonfire_readcd_get_speed;
	iface->get_size = bonfire_readcd_get_size;
}

static void
bonfire_readcd_init (BonfireReadcd *obj)
{
	obj->priv = g_new0(BonfireReadcdPrivate, 1);
}

static void
bonfire_readcd_finalize (GObject *object)
{
	BonfireReadcd *cobj;
	cobj = BONFIRE_READCD(object);
	
	if (cobj->priv->timer) {
		g_timer_destroy (cobj->priv->timer);
		cobj->priv->timer = NULL;
	}

	if (cobj->priv->toc) {
		if (cobj->priv->clean)
			g_remove (cobj->priv->toc);

		g_free (cobj->priv->toc);
		cobj->priv->toc= NULL;
	}

	if (cobj->priv->output) {
		if (cobj->priv->clean)
			g_remove (cobj->priv->output);

		g_free (cobj->priv->output);
		cobj->priv->output= NULL;
	}

	if (cobj->priv->drive) {
		nautilus_burn_drive_unref (cobj->priv->drive);
		cobj->priv->drive = NULL;
	}

	if (cobj->priv->rates) {
		g_slist_free (cobj->priv->rates);
		cobj->priv->rates = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

static BonfireBurnResult
bonfire_readcd_get_track (BonfireImager *imager,
			  BonfireTrackSource **track,
			  GError **error)
{
	BonfireReadcd *readcd;
	BonfireTrackSource *retval;

	readcd = BONFIRE_READCD (imager);

	if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_UNKNOWN)
		return BONFIRE_BURN_NOT_READY;

	if (!readcd->priv->track_ready) {
		BonfireBurnResult result;
		NautilusBurnMediaType media;

		media = nautilus_burn_drive_get_media_type (readcd->priv->drive);
		if (media > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			readcd->priv->is_DVD = TRUE;
		else
			readcd->priv->is_DVD = FALSE;

		readcd->priv->prev_sect = 0;
		readcd->priv->action = BONFIRE_READCD_ACTION_IMAGING;
		result = bonfire_job_run (BONFIRE_JOB (readcd), error);
		readcd->priv->action = BONFIRE_READCD_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;

		readcd->priv->track_ready = 1;
	}

	/* see if we are ready */
	retval = g_new0 (BonfireTrackSource, 1);

	retval->type = readcd->priv->track_type;
	if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_ISO) {
		retval->contents.iso.image = g_strdup (readcd->priv->output);
	}
	else if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_RAW) {
		retval->contents.raw.toc = g_strdup (readcd->priv->toc);
		retval->contents.raw.image = g_strdup (readcd->priv->output);
	}

	*track = retval;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_get_size (BonfireImager *imager,
			 gint64 *size,
			 gboolean sectors,
			 GError **error)
{
	BonfireReadcd *readcd;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	readcd = BONFIRE_READCD (imager);

	if (!readcd->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	if (readcd->priv->sectors_num == 0) {
		if (bonfire_job_is_running (BONFIRE_JOB (imager)))
			return BONFIRE_BURN_RUNNING;

		readcd->priv->action = BONFIRE_READCD_ACTION_GETTING_SIZE;
		result = bonfire_job_run (BONFIRE_JOB (readcd), error);
		readcd->priv->action = BONFIRE_READCD_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	if (sectors)
		*size = readcd->priv->sectors_num;
	else if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_ISO)
		*size = readcd->priv->sectors_num * 2048;
	else	/* when used with -clone option */
		*size = readcd->priv->sectors_num * 2448;

	return result;
}

static BonfireBurnResult
bonfire_readcd_get_speed (BonfireImager *imager,
			  int *speed)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (imager);

	if (!readcd->priv->speed)
		return BONFIRE_BURN_NOT_READY;

	/* FIXME: why not do a bonfire_job_get_rate and it would be up to burn.c
	 * to determine if it's a CD or a DVD */
	/* we adjust the speed wether it is a CD or a DVD */
	if (readcd->priv->is_DVD)
		*speed = (gint) readcd->priv->speed / DVD_SPEED;
	else
		*speed = (gint) readcd->priv->speed / CDR_SPEED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_get_track_type (BonfireImager *imager,
			       BonfireTrackSourceType *track_type)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (imager);

	*track_type = readcd->priv->track_type;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_set_source (BonfireJob *job,
			   const BonfireTrackSource *source,
			   GError **error)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (job);

	if (readcd->priv->drive) {
		nautilus_burn_drive_unref (readcd->priv->drive);
		readcd->priv->drive = NULL;
	}

	if (readcd->priv->clean && readcd->priv->track_ready) {
		if (readcd->priv->output)
			g_remove (readcd->priv->output);

		if (readcd->priv->toc)
			g_remove (readcd->priv->toc);
	}

	readcd->priv->sectors_num = 0;
	readcd->priv->track_ready = 0;

	if (source->type != BONFIRE_TRACK_SOURCE_DISC)
		return BONFIRE_BURN_NOT_SUPPORTED;

	readcd->priv->drive = source->contents.drive.disc;
	nautilus_burn_drive_ref (readcd->priv->drive);

	return BONFIRE_BURN_OK;	
}

static BonfireBurnResult
bonfire_readcd_set_output_type (BonfireImager *imager,
				BonfireTrackSourceType type,
				GError **error)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (imager);

	if (readcd->priv->track_type == type)
		return BONFIRE_BURN_OK;

	if (readcd->priv->clean && readcd->priv->track_ready) {
		if (readcd->priv->output)
			g_remove (readcd->priv->output);

		if (readcd->priv->toc)
			g_remove (readcd->priv->toc);
	}

	readcd->priv->sectors_num = 0;
	readcd->priv->track_ready = 0;

	readcd->priv->track_type = BONFIRE_TRACK_SOURCE_UNKNOWN;
	if (type == BONFIRE_TRACK_SOURCE_DEFAULT)
		type = BONFIRE_TRACK_SOURCE_RAW;
	else if (type != BONFIRE_TRACK_SOURCE_RAW
	      &&  type != BONFIRE_TRACK_SOURCE_ISO
	      &&  type != BONFIRE_TRACK_SOURCE_ISO_JOLIET)
		return BONFIRE_BURN_NOT_SUPPORTED;

	readcd->priv->track_type = type;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_set_output (BonfireImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (imager);

	if (readcd->priv->output) {
		if (readcd->priv->clean && readcd->priv->track_ready)
			g_remove (readcd->priv->output);

		g_free (readcd->priv->output);
		readcd->priv->output = NULL;
	}

	if (readcd->priv->toc) {
		if (readcd->priv->clean && readcd->priv->track_ready)
			g_remove (readcd->priv->toc);

		g_free (readcd->priv->toc);
		readcd->priv->toc = NULL;
	}
	readcd->priv->track_ready = 0;

	if (output)
		readcd->priv->output = g_strdup (output);

	readcd->priv->overwrite = overwrite;
	readcd->priv->clean = clean;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_read_stderr (BonfireProcess *process, const char *line)
{
	BonfireReadcd *readcd;
	char *pos;

	readcd = BONFIRE_READCD (process);
	if ((pos = strstr (line, "addr:"))) {
		int sector;
		double fraction;
		long remaining = -1;

		pos += strlen ("addr:");
		sector = (int) atoi (pos);

		/* readcd reports very often when copying a DVD */
		if (readcd->priv->is_DVD
		&&  readcd->priv->prev_sect + 256 > sector)
			return BONFIRE_BURN_OK;

		readcd->priv->prev_sect = sector;
		fraction = (double) sector / readcd->priv->sectors_num;

		if (!readcd->priv->timer && sector > 10) {
			/* let the speed regulate before keeping track */
			readcd->priv->timer = g_timer_new ();
			readcd->priv->start = sector;
		}

		if (readcd->priv->sectors_num && readcd->priv->timer) {
			gdouble elapsed;
			double rate;

			elapsed = g_timer_elapsed (readcd->priv->timer, NULL);
			rate = (sector - readcd->priv->start) / elapsed;

			if (rate > 0.0) {
				double ave_rate;

				ave_rate = bonfire_burn_common_get_average_rate (&readcd->priv->rates, rate);
				/* here the ave_rate = sectors / second
				 * 1 sec = 75 sectors
				 * speed 1 => 75 / second for a CD */
				readcd->priv->speed = ave_rate * 2352;

				remaining = (long) ((gdouble) readcd->priv->sectors_num /
							        rate - (gdouble) elapsed);
			}
		}

		if (remaining >= 0)
			bonfire_job_progress_changed (BONFIRE_JOB (readcd),
						      fraction,
						      remaining);
		else
			bonfire_job_progress_changed (BONFIRE_JOB (readcd),
						      fraction,
						      -1);
	}
	else if ((pos = strstr (line, "Capacity:"))) {
		if (readcd->priv->action == BONFIRE_READCD_ACTION_GETTING_SIZE) {
			pos += strlen ("Capacity:");
			readcd->priv->sectors_num = atoi (pos);
			bonfire_job_finished (BONFIRE_JOB (readcd));
		}
		else {
			readcd->priv->timer = g_timer_new ();
			bonfire_job_action_changed (BONFIRE_JOB (readcd),
						    BONFIRE_BURN_ACTION_DRIVE_COPY,
						    FALSE);
		}
	}
	else if (strstr (line, "Device not ready.")) {
		bonfire_job_error (BONFIRE_JOB (readcd),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("the drive is not ready")));
	}
	else if (strstr (line, "Device or resource busy")) {
		bonfire_job_error (BONFIRE_JOB (readcd),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("you don't seem to have the required permissions to access the drive")));
	}
	else if (strstr (line, "Cannot open SCSI driver.")) {
		bonfire_job_error (BONFIRE_JOB (readcd),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("you don't seem to have the required permissions to access the drive")));		
	}
	else if (strstr (line, "Cannot send SCSI cmd via ioctl")) {
		bonfire_job_error (BONFIRE_JOB (readcd),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_SCSI_IOCTL,
						_("you don't seem to have the required permissions to access the drive")));
	}
	else if (strstr (line, "Time total:")) {
		bonfire_job_finished (BONFIRE_JOB (process));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_set_argv (BonfireProcess *process,
			 GPtrArray *argv,
			 gboolean has_master,
			 GError **error)
{
	BonfireBurnResult result = FALSE;
	NautilusBurnMediaType type;
	BonfireReadcd *readcd;
	char *outfile_arg;
	char *dev_str;

	readcd = BONFIRE_READCD (process);
	bonfire_job_set_run_slave (BONFIRE_JOB (readcd), FALSE);

	if (!readcd->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	g_ptr_array_add (argv, g_strdup ("readcd"));

	if (readcd->priv->drive->cdrecord_id)
		dev_str = g_strdup_printf ("dev=%s", readcd->priv->drive->cdrecord_id);
	else if (readcd->priv->drive->device)
		dev_str = g_strdup_printf ("dev=%s", readcd->priv->drive->device);
	else
		return BONFIRE_BURN_ERR;

	g_ptr_array_add (argv, dev_str);

	if (readcd->priv->action == BONFIRE_READCD_ACTION_GETTING_SIZE) {
		g_ptr_array_add (argv, g_strdup ("-sectors=0-0"));
		bonfire_job_action_changed (BONFIRE_JOB (readcd),
					    BONFIRE_BURN_ACTION_GETTING_SIZE,
					    FALSE);
		return BONFIRE_BURN_OK;
	}

	g_ptr_array_add (argv, g_strdup ("-nocorr"));

	type = nautilus_burn_drive_get_media_type (readcd->priv->drive);
	if (type > NAUTILUS_BURN_MEDIA_TYPE_CDRW 
	&&  readcd->priv->track_type != BONFIRE_TRACK_SOURCE_ISO) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("raw images cannot be created with DVDs"));
		return BONFIRE_BURN_ERR;
	}

	if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_RAW) {
		/* NOTE: with this option the sector size is 2448 
		 * because it is raw96 (2352+96) otherwise it is 2048  */
		g_ptr_array_add (argv, g_strdup ("-clone"));
	}
	else /* This is for *.iso images to ease the process */
		g_ptr_array_add (argv, g_strdup ("-noerror"));

	if (!has_master) {
		if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_ISO) {
			result = bonfire_burn_common_check_output (&readcd->priv->output,
								   readcd->priv->overwrite,
								   NULL,
								   error);
		}
		else if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_RAW) {	
			result = bonfire_burn_common_check_output (&readcd->priv->output,
								   readcd->priv->overwrite,
								   &readcd->priv->toc,
								   error);
		}

		if (result != BONFIRE_BURN_OK)
			return result;

		outfile_arg = g_strdup_printf ("-f=%s", readcd->priv->output);
		g_ptr_array_add (argv, outfile_arg);
	}
	else if (readcd->priv->track_type == BONFIRE_TRACK_SOURCE_ISO) {
		outfile_arg = g_strdup ("-f=-");
		g_ptr_array_add (argv, outfile_arg);
	}
	else 	/* unfortunately raw images can't be piped out */
		return BONFIRE_BURN_NOT_SUPPORTED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_readcd_post (BonfireProcess *process, BonfireBurnResult retval)
{
	BonfireReadcd *readcd;

	readcd = BONFIRE_READCD (process);

	if (readcd->priv->timer) {
		g_timer_destroy (readcd->priv->timer);
		readcd->priv->timer = NULL;
	}

	if (readcd->priv->rates) {
		g_slist_free (readcd->priv->rates);
		readcd->priv->rates = NULL;
	}

	readcd->priv->speed = 0;
	readcd->priv->start = 0;

	return BONFIRE_BURN_OK;
}

BonfireReadcd *
bonfire_read_cd_new (void)
{
	BonfireReadcd *obj;

	obj = BONFIRE_READCD (g_object_new (BONFIRE_TYPE_READCD, NULL));

	return obj;
}
