/***************************************************************************
 *            bonfire-layout.c
 *
 *  mer mai 24 15:14:42 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtkuimanager.h>
#include <gtk/gtktoggleaction.h>
#include <gtk/gtkradioaction.h>
#include <gtk/gtkaction.h>

#include <gtk/gtkstock.h>

#include <gtk/gtkbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkvpaned.h>
#include <gtk/gtkimage.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkhseparator.h>

#include <gconf/gconf-client.h>

#include "bonfire-layout.h"


static void bonfire_layout_class_init (BonfireLayoutClass *klass);
static void bonfire_layout_init (BonfireLayout *sp);
static void bonfire_layout_finalize (GObject *object);

static void
bonfire_layout_preview_toggled_cb (GtkToggleAction *action,
				   BonfireLayout *layout);
static void
bonfire_layout_radio_toggled_cb (GtkRadioAction *action,
				 GtkRadioAction *current,
				 BonfireLayout *layout);

static void
bonfire_layout_show (GtkWidget *widget);
static void
bonfire_layout_hide (GtkWidget *widget);

typedef struct {
	gchar *id;
	GtkWidget *widget;
	BonfireLayoutType types;
} BonfireLayoutItem;

struct BonfireLayoutPrivate {
	GtkActionGroup *action_group;
	GtkRadioActionEntry *entries;
	gint nb_entries;

	gint accel;

	BonfireLayoutType type;
	GSList *items;

	GConfClient *client;
	gint radio_notify;
	gint preview_notify;

	GtkWidget *notebook;
	GtkWidget *main_box;
	GtkWidget *preview_pane;
};


static GObjectClass *parent_class = NULL;

#define BONFIRE_LAYOUT_PROJECT_ID	"Project"
#define BONFIRE_LAYOUT_PROJECT_NAME	N_("Project")
#define BONFIRE_LAYOUT_PROJECT_ICON	GTK_STOCK_CDROM /* it depends on project type */
#define BONFIRE_LAYOUT_PREVIEW_ID	"Viewer"
#define BONFIRE_LAYOUT_PREVIEW_NAME	N_("Preview")
#define BONFIRE_LAYOUT_PREVIEW_MENU	N_("Preview")
#define BONFIRE_LAYOUT_PREVIEW_TOOLTIP	N_("Display video, audio and image preview")
#define BONFIRE_LAYOUT_PREVIEW_ICON	GTK_STOCK_FILE

/* GCONF keys */
#define BONFIRE_KEY_DISPLAY_DIR		"/apps/bonfire/display/"
#define BONFIRE_KEY_SHOW_PREVIEW	BONFIRE_KEY_DISPLAY_DIR "preview"
#define BONFIRE_KEY_LAYOUT_AUDIO	BONFIRE_KEY_DISPLAY_DIR "audio_pane"
#define BONFIRE_KEY_LAYOUT_DATA		BONFIRE_KEY_DISPLAY_DIR "data_pane"

GType
bonfire_layout_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireLayoutClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_layout_class_init,
			NULL,
			NULL,
			sizeof (BonfireLayout),
			0,
			(GInstanceInitFunc)bonfire_layout_init,
		};

#ifdef BUILD_GDL
		type = g_type_register_static (GDL_TYPE_DOCK, 
					       "BonfireLayout",
					       &our_info,
					       0);
#else
		type = g_type_register_static (GTK_TYPE_HPANED, 
					       "BonfireLayout",
					       &our_info,
					       0);
#endif
	}

	return type;
}

static void
bonfire_layout_class_init (BonfireLayoutClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *gtk_widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_layout_finalize;

	gtk_widget_class->hide = bonfire_layout_hide;
	gtk_widget_class->show = bonfire_layout_show;
}

static void
bonfire_layout_init (BonfireLayout *obj)
{
	obj->priv = g_new0 (BonfireLayoutPrivate, 1);

	obj->priv->action_group = gtk_action_group_new ("BonfireLayoutActions");
	gtk_action_group_set_translation_domain (obj->priv->action_group, GETTEXT_PACKAGE);

	/* init GConf */
	obj->priv->client = gconf_client_get_default ();

	/* set up containers */
	obj->priv->main_box = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->main_box), 6);
	gtk_widget_show (obj->priv->main_box);
	gtk_paned_pack2 (GTK_PANED (obj), obj->priv->main_box, TRUE, FALSE);

	obj->priv->notebook = gtk_notebook_new ();
	gtk_widget_show (obj->priv->notebook);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (obj->priv->notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (obj->priv->notebook), FALSE);
	gtk_box_pack_start (GTK_BOX (obj->priv->main_box),
			    obj->priv->notebook,
			    TRUE,
			    TRUE,
			    0);
}

static void
bonfire_layout_finalize (GObject *object)
{
	BonfireLayout *cobj;
	GSList *iter;

	cobj = BONFIRE_LAYOUT(object);

	for (iter = cobj->priv->items; iter; iter = iter->next) {
		BonfireLayoutItem *item;

		item = iter->data;
		g_free (item->id);
		g_free (item);
	}

	g_slist_free (cobj->priv->items);
	cobj->priv->items = NULL;

	/* close GConf */
	gconf_client_notify_remove (cobj->priv->client, cobj->priv->preview_notify);
	gconf_client_notify_remove (cobj->priv->client, cobj->priv->radio_notify);
	g_object_unref (cobj->priv->client);

	g_free(cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_layout_new ()
{
	BonfireLayout *obj;
	
	obj = BONFIRE_LAYOUT (g_object_new (BONFIRE_TYPE_LAYOUT, NULL));

	return GTK_WIDGET (obj);
}

static void
bonfire_layout_show (GtkWidget *widget)
{
	BonfireLayout *layout;

	layout = BONFIRE_LAYOUT (widget);

	gtk_action_group_set_visible (layout->priv->action_group, TRUE);
	gtk_action_group_set_sensitive (layout->priv->action_group, TRUE);
	gtk_widget_set_sensitive (widget, TRUE);

	if (GTK_WIDGET_CLASS (parent_class)->show)
		GTK_WIDGET_CLASS (parent_class)->show (widget);
}

static void
bonfire_layout_hide (GtkWidget *widget)
{
	BonfireLayout *layout;

	layout = BONFIRE_LAYOUT (widget);

	gtk_action_group_set_visible (layout->priv->action_group, FALSE);
	gtk_action_group_set_sensitive (layout->priv->action_group, FALSE);
	gtk_widget_set_sensitive (widget, FALSE);

	if (GTK_WIDGET_CLASS (parent_class)->hide)
		GTK_WIDGET_CLASS (parent_class)->hide (widget);
}

void
bonfire_layout_add_project (BonfireLayout *layout,
			    GtkWidget *project)
{
	gtk_container_set_border_width (GTK_CONTAINER (project), 6);
	gtk_paned_pack1 (GTK_PANED (layout), project, TRUE, FALSE);
}

static GtkWidget *
_make_pane (GtkWidget *widget,
	    const char *icon,
	    const char *text,
	    gboolean fill)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *image;

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox);

	gtk_box_pack_end (GTK_BOX (vbox), widget, fill, fill, 0);

	hbox = gtk_hbox_new (FALSE, 6);
	gtk_widget_show (hbox);
	gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);

	image = gtk_image_new_from_stock (icon, GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_widget_show (image);
	gtk_misc_set_alignment (GTK_MISC (image), 0.0, 0.0);
	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);

	label = gtk_label_new (text);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

	return vbox;
}

static void
bonfire_layout_preview_toggled_cb (GtkToggleAction *action, BonfireLayout *layout)
{
	gboolean active;

	active = gtk_toggle_action_get_active (action);
	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);
	
	/* we set the correct value in GConf */
	gconf_client_set_bool (layout->priv->client,
			       BONFIRE_KEY_SHOW_PREVIEW,
			       active,
			       NULL);
}

static void
bonfire_layout_preview_changed_cb (GConfClient *client,
				   guint cxn,
				   GConfEntry *entry,
				   gpointer data)
{
	BonfireLayout *layout;
	GtkAction *action;
	GConfValue *value;
	gboolean active;

	layout = BONFIRE_LAYOUT (data);

	value = gconf_entry_get_value (entry);
	if (value->type != GCONF_VALUE_BOOL)
		return;

	active = gconf_value_get_bool (value);
	action = gtk_action_group_get_action (layout->priv->action_group,
					      BONFIRE_LAYOUT_PREVIEW_ID);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), active);

 	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);
}

void
bonfire_layout_add_preview (BonfireLayout *layout,
			    GtkWidget *preview)
{
	gboolean active;
	GtkAction *action;
	GError *error = NULL;
	GtkToggleActionEntry entry;

	layout->priv->preview_pane = preview;
	gtk_box_pack_end (GTK_BOX (layout->priv->main_box),
			  layout->priv->preview_pane,
			  FALSE,
			  FALSE,
			  0);

	/* add menu entry in display */
	entry.name = BONFIRE_LAYOUT_PREVIEW_ID;
	entry.stock_id = BONFIRE_LAYOUT_PREVIEW_ICON;
	entry.label = _(BONFIRE_LAYOUT_PREVIEW_MENU);
	entry.accelerator = g_strdup ("F11");
	entry.tooltip = _(BONFIRE_LAYOUT_PREVIEW_TOOLTIP);
	entry.is_active = FALSE;
	entry.callback = G_CALLBACK (bonfire_layout_preview_toggled_cb);
	gtk_action_group_add_toggle_actions (layout->priv->action_group,
					     &entry,
					     1,
					     layout);

	/* initializes the display */
	active = gconf_client_get_bool (layout->priv->client,
					BONFIRE_KEY_SHOW_PREVIEW,
					&error);
	if (error) {
		g_warning ("Can't access GConf key %s. This is probably harmless (first launch of bonfire).\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	action = gtk_action_group_get_action (layout->priv->action_group, BONFIRE_LAYOUT_PREVIEW_ID);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), active);

	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);

	if (!layout->priv->preview_notify)
		layout->priv->preview_notify = gconf_client_notify_add (layout->priv->client,
									BONFIRE_KEY_SHOW_PREVIEW,
									bonfire_layout_preview_changed_cb,
									layout,
									NULL,
									&error);
	if (error) {
		g_warning ("Could set notify for GConf key %s.\n", error->message);
		g_error_free (error);
		error = NULL;
	}
}

/**************************** for the source panes *****************************/
static void
bonfire_layout_pane_changed (BonfireLayout *layout, const char *id)
{
	GSList *iter;
	BonfireLayoutItem *item;

	for (iter = layout->priv->items; iter; iter = iter->next) {
		item = iter->data;
		if (!strcmp (id, item->id))
			gtk_widget_show (item->widget);
		else
			gtk_widget_hide (item->widget);
	}
}

static void
bonfire_layout_radio_toggled_cb (GtkRadioAction *action,
				 GtkRadioAction *current,
				 BonfireLayout *layout)
{
	GError *error = NULL;
	const char *id;

	id = gtk_action_get_name (GTK_ACTION (current));
	bonfire_layout_pane_changed (layout, id);

	/* update gconf value */
	if (layout->priv->type == BONFIRE_LAYOUT_AUDIO)
		gconf_client_set_string (layout->priv->client,
					 BONFIRE_KEY_LAYOUT_AUDIO,
					 id,
					 &error);
	else if (layout->priv->type == BONFIRE_LAYOUT_DATA)
		gconf_client_set_string (layout->priv->client,
					 BONFIRE_KEY_LAYOUT_DATA,
					 id,
					 &error);

	if (error) {
		g_warning ("Can't set GConf key %s. \n", error->message);
		g_error_free (error);
		error = NULL;
	}
}

static void
bonfire_layout_displayed_item_changed_cb (GConfClient *client,
					  guint cxn,
					  GConfEntry *entry,
					  gpointer data)
{
	BonfireLayout *layout;
	GConfValue *value;
	const char *id;

	layout = BONFIRE_LAYOUT (data);

	value = gconf_entry_get_value (entry);
	if (value->type != GCONF_VALUE_STRING)
		return;

	id = gconf_value_get_string (value);
	bonfire_layout_pane_changed (layout, id);
}

void
bonfire_layout_add_source (BonfireLayout *layout,
			   GtkWidget *source,
			   const char *id,
			   const char *name,
			   const char *menu,
			   const char *tooltip,
			   const char *icon,
			   BonfireLayoutType types)
{
	GtkWidget *pane;
	char *accelerator;
	BonfireLayoutItem *item;

	pane = _make_pane (source, icon, name, TRUE);
	gtk_notebook_append_page (GTK_NOTEBOOK (layout->priv->notebook),
				  pane,
				  NULL);

	/* add menu radio entry in display */
	accelerator = g_strdup_printf ("F%i", (layout->priv->accel ++) + 7);

	if (!layout->priv->entries)
		layout->priv->entries = g_new0 (GtkRadioActionEntry, 3);

	layout->priv->entries [layout->priv->nb_entries].name = g_strdup (id);
	layout->priv->entries [layout->priv->nb_entries].stock_id = g_strdup (icon);
	layout->priv->entries [layout->priv->nb_entries].label = g_strdup (menu);
	layout->priv->entries [layout->priv->nb_entries].accelerator = accelerator;
	layout->priv->entries [layout->priv->nb_entries].tooltip = g_strdup (tooltip);
	layout->priv->entries [layout->priv->nb_entries].value = layout->priv->nb_entries;

	layout->priv->nb_entries ++;

	/* add it to the items list */
	item = g_new0 (BonfireLayoutItem, 1);
	item->id = g_strdup (id);
	item->widget = pane;
	item->types = types;

	layout->priv->items = g_slist_append (layout->priv->items, item);
}

void
bonfire_layout_load (BonfireLayout *layout, BonfireLayoutType type)
{
	GSList *iter;
	char *layout_id = NULL;
	GError *error = NULL;

	/* remove GCONF notification if any */
	if (layout->priv->radio_notify)
		gconf_client_notify_remove (layout->priv->client,
					    layout->priv->radio_notify);

	if (type == BONFIRE_LAYOUT_NONE) {
		gtk_widget_hide (GTK_WIDGET (layout));
		return;
	}
	else
		gtk_widget_show (GTK_WIDGET (layout));

	/* takes care of other panes */
	if (type == BONFIRE_LAYOUT_AUDIO)
		layout_id = gconf_client_get_string (layout->priv->client,
						     BONFIRE_KEY_LAYOUT_AUDIO,
						     &error);
	else if (type == BONFIRE_LAYOUT_DATA)
		layout_id = gconf_client_get_string (layout->priv->client,
						     BONFIRE_KEY_LAYOUT_DATA,
						     &error);

	if (error) {
		g_warning ("Can't access GConf key %s. This is probably harmless (first launch of bonfire).\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	/* add new notify for the new */
	if (type == BONFIRE_LAYOUT_AUDIO)
		layout->priv->radio_notify = gconf_client_notify_add (layout->priv->client,
								      BONFIRE_KEY_LAYOUT_AUDIO,
								      bonfire_layout_displayed_item_changed_cb,
								      layout,
								      NULL,
								      &error);
	else if (type == BONFIRE_LAYOUT_DATA)
		layout->priv->radio_notify = gconf_client_notify_add (layout->priv->client,
								      BONFIRE_KEY_LAYOUT_DATA,
								      bonfire_layout_displayed_item_changed_cb,
								      layout,
								      NULL,
								      &error);

	if (error) {
		g_warning ("Could not set notify for GConf key %s.\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	layout->priv->type = type;
	for (iter = layout->priv->items; iter; iter = iter->next) {
		GtkAction *action;
		BonfireLayoutItem *item = NULL;

		item = iter->data;
		action = gtk_action_group_get_action (layout->priv->action_group, item->id);
		if (!(item->types & type)) {
			gtk_widget_hide (item->widget);
			gtk_action_set_visible (action, FALSE);
			continue;
		}

		gtk_action_set_visible (action, TRUE);
		if (layout_id) {
			if (!strcmp (layout_id, item->id)) {
				/* this is it! we found the pane to display */
				gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), TRUE);
				gtk_widget_show (item->widget);
			}
			else
				gtk_widget_hide (item->widget);
		}
	}

	g_free (layout_id);
}

void
bonfire_layout_register_menu (BonfireLayout *layout,
			      GtkUIManager *manager)
{
	GSList *iter;
	GString *description;
	GError *error = NULL;
	BonfireLayoutItem *item;

	gtk_action_group_add_radio_actions (layout->priv->action_group,
					    layout->priv->entries,
					    layout->priv->nb_entries,
					    0,
					    G_CALLBACK (bonfire_layout_radio_toggled_cb),
					    layout);
	/* FIXME: */
	g_free (layout->priv->entries);

	gtk_ui_manager_insert_action_group (manager,
					    layout->priv->action_group,
					    0);

	/* build the description of the menu */
	description = g_string_new ("<ui>"
				    "<menubar name='menubar' >"
				    "<menu action='EditMenu'>"
				    "</menu>"
				    "<menu action='ViewMenu'>"
				    "<placeholder name='ViewPlaceholder'/>");
		
	for (iter = layout->priv->items; iter; iter = iter->next) {
		item = iter->data;
		g_string_append_printf (description,
					"<menuitem action='%s'/>",
					item->id);
	}

	g_string_append_printf (description, "<separator/>");
	g_string_append_printf (description,
				"<menuitem action='%s'/>",
				BONFIRE_LAYOUT_PREVIEW_ID);

	g_string_append (description,
			 "<separator/>"
			 "</menu>"
			 "</menubar>"
			 "</ui>");

	if (!gtk_ui_manager_add_ui_from_string (manager, description->str, -1, &error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	g_string_free (description, TRUE);
}

