/***************************************************************************
 *            audio-disc.c
 *
 *  dim nov 27 15:34:32 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <time.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>

#include <gdk/gdkkeysyms.h>

#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreeviewcolumn.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkcelllayout.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkuimanager.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkstock.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-file-info.h>

#include "burn-basics.h"
#include "disc.h"
#include "async-job-manager.h"
#include "audio-disc.h"
#include "metadata.h"
#include "utils.h"
#include "song-properties.h"

#ifdef BUILD_INOTIFY
#include "inotify.h"
#include "inotify-syscalls.h"
#endif

typedef void (*GetFileInfoAsyncResultFunc) (BonfireAudioDisc *disc,
					    const char *uri,
					    GnomeVFSResult result,
					    const GnomeVFSFileInfo *info,
					    const BonfireMetadata *metadata,
					    gpointer user_data);

struct _GetFileInfoAsyncData {
	BonfireAudioDisc *disc;
	char *uri;
	gboolean metadata;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
	GnomeVFSFileInfoOptions flags;
	GetFileInfoAsyncResultFunc callback_func;
	gpointer user_data;
};
typedef struct _GetFileInfoAsyncData GetFileInfoAsyncData;

static void bonfire_audio_disc_class_init (BonfireAudioDiscClass *klass);
static void bonfire_audio_disc_init (BonfireAudioDisc *sp);
static void bonfire_audio_disc_finalize (GObject *object);
static void bonfire_audio_disc_iface_disc_init (BonfireDiscIface *iface);

static void bonfire_audio_disc_get_property (GObject * object,
					     guint prop_id,
					     GValue * value,
					     GParamSpec * pspec);
static void bonfire_audio_disc_set_property (GObject * object,
					     guint prop_id,
					     const GValue * value,
					     GParamSpec * spec);

static BonfireDiscResult
bonfire_audio_disc_get_status (BonfireDisc *disc);

static BonfireDiscResult
bonfire_audio_disc_get_track (BonfireDisc *disc,
			      BonfireDiscTrack *track,
			      gboolean src_format);
static BonfireDiscResult
bonfire_audio_disc_load_track (BonfireDisc *disc,
			       BonfireDiscTrack *track);

static BonfireDiscResult
bonfire_audio_disc_add_uri (BonfireDisc *disc,
			    const char *uri);
static void
bonfire_audio_disc_delete_selected (BonfireDisc *disc);
static void
bonfire_audio_disc_clear (BonfireDisc *disc);
static void
bonfire_audio_disc_reset (BonfireDisc *disc);

static gboolean
bonfire_audio_disc_button_released_cb (GtkTreeView *tree,
				      GdkEventButton *event,
				      BonfireAudioDisc *disc);
static gboolean
bonfire_audio_disc_key_released_cb (GtkTreeView *tree,
				    GdkEventKey *event,
				    BonfireAudioDisc *disc);
static void
bonfire_audio_disc_display_edited_cb (GtkCellRendererText *cellrenderertext,
				      gchar *path_string,
				      gchar *text,
				      BonfireAudioDisc *disc);
static void
bonfire_audio_disc_display_editing_started_cb (GtkCellRenderer *renderer,
					       GtkCellEditable *editable,
					       gchar *path,
					       BonfireAudioDisc *disc);
static void
bonfire_audio_disc_display_editing_canceled_cb (GtkCellRenderer *renderer,
						 BonfireAudioDisc *disc);
static void
bonfire_audio_disc_drag_data_received_cb (GtkTreeView *tree,
					  GdkDragContext *drag_context,
					  gint x,
					  gint y,
					  GtkSelectionData *selection_data,
					  guint info,
					  guint time,
					  BonfireAudioDisc *disc);
static void
bonfire_audio_disc_row_deleted_cb (GtkTreeModel *model,
				   GtkTreePath *arg1,
				   BonfireAudioDisc *disc);
static void
bonfire_audio_disc_row_inserted_cb (GtkTreeModel *model,
				    GtkTreePath *arg1,
				    GtkTreeIter *arg2,
				    BonfireAudioDisc *disc);
static void
bonfire_audio_disc_edit_information_cb (GtkAction *action,
					BonfireAudioDisc *disc);
static void
bonfire_audio_disc_open_activated_cb (GtkAction *action,
				      BonfireAudioDisc *disc);
static void
bonfire_audio_disc_delete_activated_cb (GtkAction *action,
					BonfireDisc *disc);
static void
bonfire_audio_disc_paste_activated_cb (GtkAction *action,
				       BonfireAudioDisc *disc);
static void
bonfire_audio_disc_decrease_activity_counter (BonfireAudioDisc *disc);

static char *
bonfire_audio_disc_get_selected_uri (BonfireDisc *disc);

#ifdef BUILD_INOTIFY
static gboolean
bonfire_audio_disc_inotify_monitor_cb (GIOChannel *channel,
				       GIOCondition condition,
				       BonfireAudioDisc *disc);
static void
bonfire_audio_disc_start_monitoring (BonfireAudioDisc *disc,
				     const char *uri);
static void
bonfire_audio_disc_cancel_monitoring (BonfireAudioDisc *disc,
				      const char *uri);
#endif

struct _BonfireAudioDiscPrivate {
	GtkWidget *notebook;
	GtkWidget *tree;

	GtkUIManager *manager;

	GtkTreePath *selected_path;

	BonfireAsyncJobManager *jobs;
	int file_type;
	int dir_type;

	GSList *processing_meta;
	GSList *unprocessed_meta;
	gint meta_process_id;

#ifdef BUILD_INOTIFY
	int notify_id;
	GIOChannel *notify;
	GHashTable *monitored;
#endif

	double size;

	int activity_counter;

	int editing:1;
	int reject_files:1;
};

enum {
	TRACK_NUM_COL,
	ICON_COL,
	NAME_COL,
	SIZE_COL,
	ARTIST_COL,
	URI_COL,
	DSIZE_COL,
	COMPOSER_COL,
	ISRC_COL,
	NB_COL
};

static GtkActionEntry entries[] = {
	{"ContextualMenu", NULL, N_("Menu")},
	{"Open", GTK_STOCK_OPEN, N_("Open"), NULL, NULL,
	 G_CALLBACK (bonfire_audio_disc_open_activated_cb)},
	{"Edit", GTK_STOCK_PROPERTIES, N_("Edit information"), NULL, NULL,
	 G_CALLBACK (bonfire_audio_disc_edit_information_cb)},
	{"Delete", GTK_STOCK_REMOVE, N_("Remove"), NULL, NULL,
	 G_CALLBACK (bonfire_audio_disc_delete_activated_cb)},
	{"Paste", GTK_STOCK_PASTE, N_("Paste"), NULL, NULL,
	 G_CALLBACK (bonfire_audio_disc_paste_activated_cb)},
};

static const char *description = {
	"<ui>"
	"<popup action='ContextMenu'>"
		"<menuitem action='Open'/>"
		"<menuitem action='Delete'/>"
		"<menuitem action='Edit'/>"
		"<separator/>"
		"<menuitem action='Paste'/>"
	"</popup>"
	"</ui>"
};

enum {
	TREE_MODEL_ROW = 150,
	TARGET_URIS_LIST,
};

static GtkTargetEntry ntables_cd [] = {
	{"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, TREE_MODEL_ROW},
	{"text/uri-list", 0, TARGET_URIS_LIST}
};
static guint nb_targets_cd = sizeof (ntables_cd) / sizeof (ntables_cd[0]);

static GtkTargetEntry ntables_source [] = {
	{"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, TREE_MODEL_ROW},
};
static guint nb_targets_source = sizeof (ntables_source) / sizeof (ntables_source[0]);

static GObjectClass *parent_class = NULL;

enum {
	PROP_NONE,
	PROP_REJECT_FILE,
};

struct _SearchAudioContentData {
	GSList *found;
	char *uri;

	int cancel:1;
};
typedef struct _SearchAudioContentData SearchAudioContentData;

#define MAX_CONCURENT_META 4
#define BONFIRE_AUDIO_DISC_UNPROCESSED_METADATA(disc, metadata)	\
{	\
	if (!disc->priv->meta_process_id	\
	&&  g_slist_length (disc->priv->processing_meta) <= MAX_CONCURENT_META)	\
		disc->priv->meta_process_id = g_idle_add ((GSourceFunc) bonfire_audio_disc_process_metadata, disc);	\
	disc->priv->unprocessed_meta = g_slist_append (disc->priv->unprocessed_meta, metadata);	\
	bonfire_audio_disc_increase_activity_counter (disc);	\
}

#define BONFIRE_AUDIO_DISC_METADATA_PROCESSED(disc, metadata)	\
{	\
	disc->priv->processing_meta = g_slist_remove (disc->priv->processing_meta, metadata);	\
	bonfire_audio_disc_decrease_activity_counter (disc);	\
	g_object_unref (metadata);	\
	if (!disc->priv->meta_process_id	\
	&&  g_slist_length (disc->priv->processing_meta) <= MAX_CONCURENT_META)	\
		disc->priv->meta_process_id = g_idle_add ((GSourceFunc) bonfire_audio_disc_process_metadata, disc);	\
}

#define BONFIRE_TIME_TO_SIZE(len) ((double) ((len * 1024 * 1024 * 7 / ((double) 1000 * 1000 * 1000 * 48)) + (double) 1024 * 1024))
#define COL_KEY "column_key"

GType
bonfire_audio_disc_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireAudioDiscClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_audio_disc_class_init,
			NULL,
			NULL,
			sizeof (BonfireAudioDisc),
			0,
			(GInstanceInitFunc) bonfire_audio_disc_init,
		};

		static const GInterfaceInfo disc_info =
		{
			(GInterfaceInitFunc) bonfire_audio_disc_iface_disc_init,
			NULL,
			NULL
		};

		type = g_type_register_static(GTK_TYPE_VBOX, 
					      "BonfireAudioDisc",
					      &our_info,
					      0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_DISC,
					     &disc_info);
	}

	return type;
}

static void
bonfire_audio_disc_class_init (BonfireAudioDiscClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_audio_disc_finalize;
	object_class->set_property = bonfire_audio_disc_set_property;
	object_class->get_property = bonfire_audio_disc_get_property;

	g_object_class_install_property (object_class,
					 PROP_REJECT_FILE,
					 g_param_spec_boolean
					 ("reject-file",
					  "Whether it accepts files",
					  "Whether it accepts files",
					  FALSE,
					  G_PARAM_READWRITE));
}

static void
bonfire_audio_disc_iface_disc_init (BonfireDiscIface *iface)
{
	iface->add_uri = bonfire_audio_disc_add_uri;
	iface->delete_selected = bonfire_audio_disc_delete_selected;
	iface->clear = bonfire_audio_disc_clear;
	iface->reset = bonfire_audio_disc_reset;
	iface->get_track = bonfire_audio_disc_get_track;
	iface->load_track = bonfire_audio_disc_load_track;
	iface->get_status = bonfire_audio_disc_get_status;
	iface->get_selected_uri = bonfire_audio_disc_get_selected_uri;
}

static void bonfire_audio_disc_get_property (GObject * object,
					     guint prop_id,
					     GValue * value,
					     GParamSpec * pspec)
{
	BonfireAudioDisc *disc;

	disc = BONFIRE_AUDIO_DISC (object);

	switch (prop_id) {
	case PROP_REJECT_FILE:
		g_value_set_boolean (value, disc->priv->reject_files);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void bonfire_audio_disc_set_property (GObject * object,
					     guint prop_id,
					     const GValue * value,
					     GParamSpec * pspec)
{
	BonfireAudioDisc *disc;

	disc = BONFIRE_AUDIO_DISC (object);

	switch (prop_id) {
	case PROP_REJECT_FILE:
		disc->priv->reject_files = g_value_get_boolean (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
bonfire_audio_disc_build_context_menu (BonfireAudioDisc *disc)
{
	GtkActionGroup *action_group;
	GError *error = NULL;

	action_group = gtk_action_group_new ("MenuAction");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group, entries,
				      G_N_ELEMENTS (entries),
				      disc);

	disc->priv->manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (disc->priv->manager,
					    action_group,
					    0);

	if (!gtk_ui_manager_add_ui_from_string (disc->priv->manager,
						description,
						-1,
						&error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}
}

static void
bonfire_audio_disc_init (BonfireAudioDisc *obj)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeModel *model;
	GtkWidget *scroll;

	obj->priv = g_new0 (BonfireAudioDiscPrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 8);

	/* notebook to display information about how to use the tree */
	obj->priv->notebook = bonfire_utils_get_use_info_notebook ();
	gtk_box_pack_start (GTK_BOX (obj), obj->priv->notebook, TRUE, TRUE, 0);

	/* Tree */
	obj->priv->tree = gtk_tree_view_new ();
	g_signal_connect (G_OBJECT (obj->priv->tree),
			  "button-release-event",
			  G_CALLBACK (bonfire_audio_disc_button_released_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree),
			  "key-release-event",
			  G_CALLBACK (bonfire_audio_disc_key_released_cb),
			  obj);

	gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (obj->priv->tree)),
				     GTK_SELECTION_MULTIPLE);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (obj->priv->tree), TRUE);

	model = (GtkTreeModel*) gtk_list_store_new (NB_COL,
						     G_TYPE_INT,
						     GDK_TYPE_PIXBUF,
						     G_TYPE_STRING,
						     G_TYPE_STRING,
						     G_TYPE_STRING,
						     G_TYPE_STRING,
						     G_TYPE_DOUBLE, 
						     G_TYPE_STRING,
						     G_TYPE_INT);

	g_signal_connect (G_OBJECT (model),
			  "row_deleted",
			  G_CALLBACK (bonfire_audio_disc_row_deleted_cb),
			  obj);
	g_signal_connect (G_OBJECT (model),
			  "row_inserted",
			  G_CALLBACK (bonfire_audio_disc_row_inserted_cb),
			  obj);

	gtk_tree_view_set_model (GTK_TREE_VIEW (obj->priv->tree),
				 GTK_TREE_MODEL (model));
	g_object_unref (model);

	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_resizable (column, FALSE);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Num."),
							   renderer,
							   "text", TRACK_NUM_COL,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);

	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 200);

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_add_attribute (column, renderer, "pixbuf",
					    ICON_COL);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set_data (G_OBJECT (renderer), COL_KEY, GINT_TO_POINTER (NAME_COL));
	g_object_set (G_OBJECT (renderer),
		      "editable", TRUE, /* disable this? it doesn't play well with DND and double click */
		      "mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		      "ellipsize-set", TRUE,
		      "ellipsize", PANGO_ELLIPSIZE_END,
		      NULL);

	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (bonfire_audio_disc_display_edited_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-started",
			  G_CALLBACK (bonfire_audio_disc_display_editing_started_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-canceled",
			  G_CALLBACK (bonfire_audio_disc_display_editing_canceled_cb), obj);

	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "text", NAME_COL);
	gtk_tree_view_column_set_title (column, _("Title"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);

	gtk_tree_view_set_expander_column (GTK_TREE_VIEW (obj->priv->tree),
					   column);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set_data (G_OBJECT (renderer), COL_KEY, GINT_TO_POINTER (ARTIST_COL));
	g_object_set (G_OBJECT (renderer),
/*		      "editable", TRUE, disable this for the time being it doesn't play well with DND and double click */
		      "mode", GTK_CELL_RENDERER_MODE_EDITABLE,
		      "ellipsize-set", TRUE,
		      "ellipsize", PANGO_ELLIPSIZE_END,
		      NULL);

	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (bonfire_audio_disc_display_edited_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-started",
			  G_CALLBACK (bonfire_audio_disc_display_editing_started_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-canceled",
			  G_CALLBACK (bonfire_audio_disc_display_editing_canceled_cb), obj);
	column = gtk_tree_view_column_new_with_attributes (_("Artist"),
							   renderer,
							   "text", ARTIST_COL,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 200);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Length"),
							   renderer,
							   "text", SIZE_COL,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_resizable (column, FALSE);


	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scroll), obj->priv->tree);

	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook),
				  scroll,
				  NULL);

	/* dnd */
	gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (obj->priv->tree),
					      ntables_cd,
					      nb_targets_cd,
					      GDK_ACTION_COPY|
					      GDK_ACTION_MOVE);

	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_data_received",
			  G_CALLBACK (bonfire_audio_disc_drag_data_received_cb),
			  obj);

	gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (obj->priv->tree),
						GDK_BUTTON1_MASK,
						ntables_source,
						nb_targets_source,
						GDK_ACTION_COPY |
						GDK_ACTION_MOVE);

	/* create menus */
	bonfire_audio_disc_build_context_menu (obj);

#ifdef BUILD_INOTIFY
	int fd;

	obj->priv->monitored = g_hash_table_new_full (g_direct_hash,
						      g_direct_equal,
						      NULL,
						      (GDestroyNotify) g_free);

	fd = inotify_init ();
	if (fd != -1) {
		obj->priv->notify = g_io_channel_unix_new (fd);

		g_io_channel_set_encoding (obj->priv->notify, NULL, NULL);
		g_io_channel_set_close_on_unref (obj->priv->notify, TRUE);
		obj->priv->notify_id = g_io_add_watch (obj->priv->notify,
						       G_IO_IN | G_IO_HUP | G_IO_PRI,
						       (GIOFunc) bonfire_audio_disc_inotify_monitor_cb,
						       obj);
		g_io_channel_unref (obj->priv->notify);
	}
	else
		g_warning ("Failed to open /dev/inotify: %s\n",
			   strerror (errno));
#endif
}

static void
bonfire_audio_disc_reset_real (BonfireAudioDisc *disc)
{
	if (disc->priv->selected_path) {
		gtk_tree_path_free (disc->priv->selected_path);
		disc->priv->selected_path = NULL;
	}

	if (disc->priv->jobs)
		bonfire_async_job_manager_cancel_by_object (disc->priv->jobs,
							    G_OBJECT (disc));

	if (disc->priv->processing_meta) {
		g_slist_foreach (disc->priv->processing_meta,
 				 (GFunc) bonfire_metadata_cancel,
				 NULL);
		g_slist_foreach (disc->priv->processing_meta,
				 (GFunc) g_object_unref,
				 NULL);
		g_slist_free (disc->priv->processing_meta);
		disc->priv->processing_meta = NULL;
	}

	if (disc->priv->unprocessed_meta) {
		g_slist_foreach (disc->priv->unprocessed_meta,
				 (GFunc) bonfire_metadata_cancel,
				 NULL);
		g_slist_foreach (disc->priv->unprocessed_meta,
				 (GFunc) g_object_unref,
				 NULL);
		g_slist_free (disc->priv->unprocessed_meta);
		disc->priv->unprocessed_meta = NULL;
	}

	if (disc->priv->meta_process_id) {
		g_source_remove (disc->priv->meta_process_id);
		disc->priv->meta_process_id = 0;
	}

#ifdef BUILD_INOTIFY
	g_hash_table_destroy (disc->priv->monitored);
	disc->priv->monitored = g_hash_table_new_full (g_direct_hash,
						       g_direct_equal,
						       NULL,
						       (GDestroyNotify) g_free);
#endif

	disc->priv->size = 0.0;

	disc->priv->activity_counter = 1;
	bonfire_audio_disc_decrease_activity_counter (disc);
}

static void
bonfire_audio_disc_finalize (GObject *object)
{
	BonfireAudioDisc *cobj;
	cobj = BONFIRE_AUDIO_DISC(object);
	
	bonfire_audio_disc_reset_real (cobj);

#ifdef BUILD_INOTIFY
	if (cobj->priv->notify_id)
		g_source_remove (cobj->priv->notify_id);

	g_hash_table_destroy (cobj->priv->monitored);
#endif

	if (cobj->priv->jobs) {
		bonfire_async_job_manager_unregister_type (cobj->priv->jobs,
							   cobj->priv->dir_type);
		bonfire_async_job_manager_unregister_type (cobj->priv->jobs,
							   cobj->priv->file_type);
		g_object_unref (cobj->priv->jobs);
		cobj->priv->jobs = NULL;
	}

	g_object_unref (cobj->priv->manager);
	cobj->priv->manager = NULL;

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_audio_disc_new ()
{
	BonfireAudioDisc *obj;
	
	obj = BONFIRE_AUDIO_DISC (g_object_new(BONFIRE_TYPE_AUDIO_DISC, NULL));
	
	return GTK_WIDGET (obj);
}

/******************************** activity *************************************/
static BonfireDiscResult
bonfire_audio_disc_get_status (BonfireDisc *disc)
{
	if (BONFIRE_AUDIO_DISC (disc)->priv->activity_counter)
		return BONFIRE_DISC_NOT_READY;

	return BONFIRE_DISC_OK;
}

static void
bonfire_audio_disc_increase_activity_counter (BonfireAudioDisc *disc)
{
	GdkCursor *cursor;

	if (disc->priv->activity_counter == 0 && GTK_WIDGET (disc)->window) {
		cursor = gdk_cursor_new (GDK_WATCH);
		gdk_window_set_cursor (GTK_WIDGET (disc)->window, cursor);
		gdk_cursor_unref (cursor);
	}

	disc->priv->activity_counter++;
}

static void
bonfire_audio_disc_decrease_activity_counter (BonfireAudioDisc *disc)
{

	if (disc->priv->activity_counter == 1 && GTK_WIDGET (disc)->window)
		gdk_window_set_cursor (GTK_WIDGET (disc)->window, NULL);

	disc->priv->activity_counter--;
}

/******************************** utility functions ****************************/
static void
bonfire_audio_disc_re_index_track_num (BonfireAudioDisc *disc,
				       GtkTreeModel *model,
				       GtkTreePath *path)
{
	GtkTreeIter iter;
	int *indices;
	int num;

	if (!path) {
		path = gtk_tree_path_new_from_indices (0, -1);
		num = 1;
	}
	else {
		indices = gtk_tree_path_get_indices (path);

		if (!indices)
			return;

		num = indices [0] + 1;
	}

	if (!gtk_tree_model_get_iter (model, &iter, path)) {
		gtk_tree_path_free (path);
		return;
	}

	gtk_tree_path_free (path);

	gtk_list_store_set (GTK_LIST_STORE (model),
			    &iter,
			    TRACK_NUM_COL, num, -1);

	while (gtk_tree_model_iter_next (model, &iter)) {
		num ++;
		gtk_list_store_set (GTK_LIST_STORE (model),
				    &iter,
				    TRACK_NUM_COL, num, -1);
	}
}

static void
bonfire_audio_disc_row_deleted_cb (GtkTreeModel *model,
				   GtkTreePath *path,
				   BonfireAudioDisc *disc)
{
	int num;

	num = gtk_tree_model_iter_n_children (model, NULL);
	bonfire_disc_contents_changed (BONFIRE_DISC (disc), num);
	bonfire_audio_disc_re_index_track_num (disc, model, NULL);
}

static void
bonfire_audio_disc_row_inserted_cb (GtkTreeModel *model,
				    GtkTreePath *path,
				    GtkTreeIter *iter,
				    BonfireAudioDisc *disc)
{
	int num;

	num = gtk_tree_model_iter_n_children (model, NULL);
	bonfire_disc_contents_changed (BONFIRE_DISC (disc), num);
	bonfire_audio_disc_re_index_track_num (disc, model, NULL);
}

static void
bonfire_audio_disc_size_changed (BonfireAudioDisc *disc)
{
	char *string;

	if (disc->priv->size > 0.0)
		string = bonfire_utils_get_time_string_from_size (disc->priv->size,
								  TRUE,
								  FALSE);
	else
		string = NULL;

	bonfire_disc_size_changed (BONFIRE_DISC (disc),
				   disc->priv->size,
				   string);
	g_free (string);
}

static void
bonfire_audio_disc_set_row_from_metadata (BonfireAudioDisc *disc,
					  GtkTreeModel *model,
					  GtkTreeIter *iter,
					  const BonfireMetadata *metadata)
{
	GdkPixbuf *icon_pix;
	char *size;

	if (metadata->type)
		icon_pix = bonfire_utils_get_icon_for_mime (metadata->type, 16);
	else
		icon_pix = NULL;

	size = bonfire_utils_get_time_string (metadata->len, TRUE);
	gtk_list_store_set (GTK_LIST_STORE (model), iter,
			    ICON_COL, icon_pix,
			    ARTIST_COL, metadata->artist,
			    SIZE_COL, size,
			    DSIZE_COL, BONFIRE_TIME_TO_SIZE (metadata->len),
			    COMPOSER_COL, metadata->composer,
			    ISRC_COL, metadata->isrc, 
			    -1);

	if (metadata->title)
		gtk_list_store_set (GTK_LIST_STORE (model), iter,
				    NAME_COL,
				    metadata->title, -1),

	g_free (size);
	if (icon_pix)
		g_object_unref (icon_pix);
}

/*********************** get file info / metadata asynchronously ***************/
static gboolean
bonfire_audio_disc_process_metadata (BonfireAudioDisc *disc)
{
	BonfireMetadata *metadata;

	/* to avoid too much overhead we only fetch
	 * MAX_CONCURENT_META metadata files at a time */
	if (!disc->priv->unprocessed_meta
	||  g_slist_length (disc->priv->processing_meta) > MAX_CONCURENT_META) {
		disc->priv->meta_process_id = 0;
		return FALSE;
	}

	metadata = disc->priv->unprocessed_meta->data;
	disc->priv->unprocessed_meta = g_slist_remove (disc->priv->unprocessed_meta, metadata);
	disc->priv->processing_meta = g_slist_prepend (disc->priv->processing_meta, metadata);

	bonfire_metadata_get_async (metadata);
	return TRUE;
}

static void
bonfire_audio_disc_get_file_info_async_destroy (gpointer data)
{
	GetFileInfoAsyncData *callback_data = data;

	if (callback_data->info) {
		gnome_vfs_file_info_clear (callback_data->info);
		gnome_vfs_file_info_unref (callback_data->info);
	}

	g_free (callback_data->uri);
	g_free (callback_data);
}

static void
bonfire_audio_disc_get_file_info_closure_destroy (gpointer data, GClosure *closure)
{
	bonfire_audio_disc_get_file_info_async_destroy (data);
}

static void
bonfire_audio_disc_metadata_completed_cb (BonfireMetadata *metadata,
					  GError *error,
					  GetFileInfoAsyncData *callback_data)
{
	BonfireAudioDisc *disc;

	/* NOTE: no need to destroy callback_data as its
	 * destruction is handled by g_signal_connect_data */
	disc = callback_data->disc;

	if (error) {
		g_warning ("ERROR %s\n", error->message);
		BONFIRE_AUDIO_DISC_METADATA_PROCESSED (disc, metadata);
		metadata = NULL;
	}

	callback_data->callback_func (disc,
				      callback_data->uri,
				      callback_data->result,
				      callback_data->info,
				      metadata,
				      callback_data->user_data);

	if (metadata)
		BONFIRE_AUDIO_DISC_METADATA_PROCESSED (disc, metadata);
}

static gboolean
bonfire_audio_disc_get_file_info_async_result (GObject *obj, gpointer data)
{
	BonfireMetadata *metadata = NULL;
	GetFileInfoAsyncData *callback_data = data;
	BonfireAudioDisc *disc = BONFIRE_AUDIO_DISC (obj);

	bonfire_audio_disc_decrease_activity_counter (callback_data->disc);

	/* this can happen if URI is malformed */
	if (!callback_data->info)
		goto end;

	if (callback_data->info->type != GNOME_VFS_FILE_TYPE_REGULAR
	||  !callback_data->metadata)
		goto end;

	metadata = bonfire_metadata_new (callback_data->uri);
	if (!metadata)
		goto end;

	g_signal_connect_data (G_OBJECT (metadata),
			       "completed",
			       G_CALLBACK (bonfire_audio_disc_metadata_completed_cb),
			       callback_data,
			       bonfire_audio_disc_get_file_info_closure_destroy,
			       0);

	BONFIRE_AUDIO_DISC_UNPROCESSED_METADATA (disc, metadata);
	return FALSE;


end:
	callback_data->callback_func (callback_data->disc,
				      callback_data->uri,
				      callback_data->result,
				      callback_data->info,
				      NULL,
				      callback_data->user_data);
	return TRUE;
}

static gboolean
bonfire_audio_disc_get_file_info_async (GObject *obj, gpointer data)
{
	GnomeVFSFileInfo *info;
	GetFileInfoAsyncData *callback_data = data;

	info = gnome_vfs_file_info_new ();
	callback_data->result = gnome_vfs_get_file_info (callback_data->uri,
							 info,
							 callback_data->flags|
							 GNOME_VFS_FILE_INFO_FOLLOW_LINKS);

	if (callback_data->result != GNOME_VFS_OK) {
		gnome_vfs_file_info_clear (info);
		gnome_vfs_file_info_unref (info);
	}
	else
		callback_data->info = info;

	return TRUE;
}

static BonfireDiscResult 
bonfire_audio_disc_get_info_async (BonfireAudioDisc *disc,
				   const char *uri,
				   GnomeVFSFileInfoOptions flags,
				   gboolean metadata,
				   GetFileInfoAsyncResultFunc func,
				   gpointer user_data)
{
	GetFileInfoAsyncData *callback_data;

	callback_data = g_new0 (GetFileInfoAsyncData, 1);
	callback_data->disc = disc;
	callback_data->flags = flags;
	callback_data->metadata = metadata;
	callback_data->callback_func = func;
	callback_data->user_data = user_data;
	callback_data->uri = gnome_vfs_escape_host_and_path_string (uri);

	if (!disc->priv->jobs)
		disc->priv->jobs = bonfire_async_job_manager_get_default ();

	if (!disc->priv->file_type) {
		disc->priv->file_type = bonfire_async_job_manager_register_type (disc->priv->jobs,
										 G_OBJECT (disc),
										 bonfire_audio_disc_get_file_info_async,
										 bonfire_audio_disc_get_file_info_async_result,
										 bonfire_audio_disc_get_file_info_async_destroy,
										 NULL);
	}

	/* NOTE : if an error occurs the callback_data will be freed by async_job_manager */
	if (!bonfire_async_job_manager_queue (disc->priv->jobs,
					      disc->priv->file_type,
					      callback_data))
		return BONFIRE_DISC_ERROR_THREAD;

	bonfire_audio_disc_increase_activity_counter (disc);
	return BONFIRE_DISC_OK;
}

static void
bonfire_audio_disc_add_file (BonfireAudioDisc *disc,
			     const char *uri,
			     double size)
{
#ifdef BUILD_INOTIFY
	bonfire_audio_disc_start_monitoring (disc, uri);
#endif

	disc->priv->size += BONFIRE_TIME_TO_SIZE (size);
	bonfire_audio_disc_size_changed (disc);
}

static void
bonfire_audio_disc_visit_dir_destroy (gpointer data)
{
	SearchAudioContentData *callback_data = data;

	g_slist_foreach (callback_data->found, (GFunc) g_free, NULL);
	g_slist_free (callback_data->found);
	g_free (callback_data->uri);
	g_free (callback_data);
}

static void
bonfire_audio_disc_visit_dir_cancel (gpointer data)
{
	SearchAudioContentData *callback_data = data;

	callback_data->cancel = 1;
}

static void
bonfire_audio_disc_audio_file_processed (BonfireMetadata *metadata,
					 const GError *error,
					 BonfireAudioDisc *disc)
{
	char *name;
	GtkTreeIter iter;
	GtkTreeModel *model;

	if (error) {
		g_warning ("ERROR %s\n", error->message);
		BONFIRE_AUDIO_DISC_METADATA_PROCESSED (disc, metadata);
		return;
	}

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	gtk_list_store_append (GTK_LIST_STORE (model), &iter);

	name = g_path_get_basename (metadata->uri);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			    NAME_COL, name,
			    URI_COL, metadata->uri,
			    -1);
	g_free (name);

	bonfire_audio_disc_set_row_from_metadata (disc,
						  model,
						  &iter,
						  metadata);

	bonfire_audio_disc_add_file (disc,
				     metadata->uri,
				     metadata->len);

	BONFIRE_AUDIO_DISC_METADATA_PROCESSED (disc, metadata);
}

static gboolean
bonfire_audio_disc_visit_dir_thread_result (GObject *obj, gpointer data)
{
	GSList *iter;
	char *escaped_uri;
	BonfireMetadata *metadata;
	SearchAudioContentData *callback_data = data;
	BonfireAudioDisc *disc = BONFIRE_AUDIO_DISC (obj);

	for (iter = callback_data->found; iter; iter = iter->next) {
		escaped_uri = iter->data;
		metadata = bonfire_metadata_new (escaped_uri);
		if (!metadata)
			continue;
	
		g_signal_connect (G_OBJECT (metadata),
				  "completed",
				  G_CALLBACK (bonfire_audio_disc_audio_file_processed),
				  disc);

		BONFIRE_AUDIO_DISC_UNPROCESSED_METADATA (disc, metadata);
	}

	bonfire_audio_disc_decrease_activity_counter (disc);
	return TRUE;
}

static gboolean
bonfire_audio_disc_visit_dir_thread_cb (const gchar *rel_path,
					GnomeVFSFileInfo *info,
					gboolean recursing_will_loop,
					SearchAudioContentData *callback_data,
					gboolean *recurse)
{
	char *uri = NULL;
	char *escaped_uri;

	if (callback_data->cancel)
		return FALSE;

	*recurse = FALSE;

	/* check the rights, file type, ... */
	if (getuid () == info->uid && !(info->permissions & GNOME_VFS_PERM_USER_READ))
		return TRUE;
	else if (getgid () == info->gid && !(info->permissions & GNOME_VFS_PERM_GROUP_READ))
		return TRUE;
	else if (!(info->permissions & GNOME_VFS_PERM_OTHER_READ))
		return TRUE;

	if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		if (!recursing_will_loop)
			*recurse = TRUE;

		return TRUE;
	}
	else if (info->type != GNOME_VFS_FILE_TYPE_REGULAR)
		return TRUE;

	if (strncmp (info->mime_type, "audio/", 6)
	&&  strcmp (info->mime_type, "application/ogg")
	&&  strcmp (info->mime_type, "application/octet-stream"))
		return TRUE;

	uri = g_strconcat (callback_data->uri, "/", rel_path, NULL);
	escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
	g_free (uri);

	callback_data->found = g_slist_prepend (callback_data->found, escaped_uri);
	return TRUE;
}

static gboolean
bonfire_audio_disc_visit_dir_thread (GObject *obj, gpointer data)
{
	char *escaped_uri;
	GnomeVFSResult result;
	SearchAudioContentData *callback_data = data;

	escaped_uri = gnome_vfs_escape_host_and_path_string (callback_data->uri);
	result = gnome_vfs_directory_visit (escaped_uri,
					    GNOME_VFS_FILE_INFO_DEFAULT |
					    GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					    GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
					    GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
					    GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
					    GNOME_VFS_DIRECTORY_VISIT_DEFAULT |
					    GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK,
					    (GnomeVFSDirectoryVisitFunc) bonfire_audio_disc_visit_dir_thread_cb,
					    callback_data);

	if (result != GNOME_VFS_OK) {
		g_warning ("ERROR opening directory : %s\n",
			   gnome_vfs_result_to_string (result));
	}

	g_free (escaped_uri);
	return TRUE;
}

static BonfireDiscResult
bonfire_audio_disc_visit_dir_async (BonfireAudioDisc *disc,
				    const char *uri)
{
	SearchAudioContentData *callback_data;

	callback_data = g_new0 (SearchAudioContentData, 1);
	callback_data->uri = g_strdup (uri);

	if (!disc->priv->jobs)
		disc->priv->jobs = bonfire_async_job_manager_get_default ();

	if (!disc->priv->dir_type) {
		disc->priv->dir_type = bonfire_async_job_manager_register_type (disc->priv->jobs,
										G_OBJECT (disc),
										bonfire_audio_disc_visit_dir_thread,
										bonfire_audio_disc_visit_dir_thread_result,
										bonfire_audio_disc_visit_dir_destroy,
										bonfire_audio_disc_visit_dir_cancel);
	}

	if (!bonfire_async_job_manager_queue (disc->priv->jobs,
					      disc->priv->dir_type,
					      callback_data))
		return BONFIRE_DISC_ERROR_THREAD;

	bonfire_audio_disc_increase_activity_counter (disc);
	return BONFIRE_DISC_OK;
}

/**************************** New Row ******************************************/
static BonfireDiscResult
bonfire_audio_disc_add_dir (BonfireAudioDisc *disc,
			    const char *uri)
{
	gint answer;
	GtkWidget *dialog;
	GtkWidget *toplevel;
	BonfireDiscResult result;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_NONE,
					 _("Directories can't be added to an audio disc:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Directory search"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("Do you want to search for audio discs inside?"));

	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				_("Search directory"), GTK_RESPONSE_OK,
				NULL);

	gtk_widget_show_all (dialog);
	answer = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	if (answer != GTK_RESPONSE_OK)
		return BONFIRE_DISC_OK;

	result = bonfire_audio_disc_visit_dir_async (disc, uri);
	return result;
}

static void
bonfire_audio_disc_unreadable_dialog (BonfireAudioDisc *disc,
				      const char *uri,
				      GnomeVFSResult result)
{
	GtkWidget *dialog, *toplevel;
	char *name;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	if (toplevel == NULL) {
		g_warning ("Can't open file %s : %s\n",
			   uri,
			   gnome_vfs_result_to_string (result));

		return;
	}

	name = g_filename_display_basename (uri);
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT|
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 _("File \"%s\" can't be opened."),
					 name);
	g_free (name);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Unreadable file"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  gnome_vfs_result_to_string (result));

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
bonfire_audio_disc_file_type_error_dialog (BonfireAudioDisc *disc,
					   const char *uri)
{
	GtkWidget *dialog, *toplevel;
	char *name;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	if (toplevel == NULL) {
		g_warning ("Content widget error : can't handle \"%s\".\n", uri);
		return ;
	}

	name = g_path_get_basename (uri);
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT|
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 _("\"%s\" can't be handled by gstreamer:"),
					 name);
	g_free (name);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Unhandled song"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("Make sure the appropriate codec is installed."));

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
bonfire_audio_disc_new_row_cb (BonfireAudioDisc *disc,
			       const char *escaped_uri,
			       GnomeVFSResult result,
			       const GnomeVFSFileInfo *info,
			       const BonfireMetadata *metadata,
			       GtkTreeRowReference *ref)
{
	char *uri;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *treepath;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	treepath = gtk_tree_row_reference_get_path (ref);
	gtk_tree_row_reference_free (ref);
	if (!treepath)
		return;

	gtk_tree_model_get_iter (model, &iter, treepath);
	gtk_tree_path_free (treepath);

	uri = gnome_vfs_unescape_string_for_display (escaped_uri);
	if (result != GNOME_VFS_OK) {
		bonfire_audio_disc_unreadable_dialog (disc,
						      uri,
						      result);
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		goto cleanup;
	}

	if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		bonfire_audio_disc_add_dir (disc, uri);
		goto cleanup;
	}

	if (info->type != GNOME_VFS_FILE_TYPE_REGULAR || !metadata) {
		gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
		bonfire_audio_disc_file_type_error_dialog (disc, uri);
		goto cleanup;
	}

	if (GNOME_VFS_FILE_INFO_SYMLINK (info)) {
		g_free (uri);
		uri = g_strconcat ("file://", info->symlink_name, NULL);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter,
				    URI_COL, uri, -1);
	}

	bonfire_audio_disc_set_row_from_metadata (disc,
						  model,
						  &iter,
						  metadata);
	bonfire_audio_disc_add_file (disc,
				     uri,
				     metadata->len);

cleanup :
	g_free (uri);
}

static BonfireDiscResult
bonfire_audio_disc_add_uri_real (BonfireAudioDisc *disc,
				 const char *uri,
				 int pos)
{
	BonfireDiscResult success;
	GtkTreeRowReference *ref;
	GtkTreePath *treepath;
	GtkTreeModel *store;
	char *unescaped_uri;
	GtkTreeIter iter;
	char *name;

	if (disc->priv->reject_files)
		return BONFIRE_DISC_NOT_READY;

	store = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	if (pos > -1)
		gtk_list_store_insert (GTK_LIST_STORE (store), &iter, pos);
	else
		gtk_list_store_append (GTK_LIST_STORE (store), &iter);

	unescaped_uri = gnome_vfs_unescape_string_for_display (uri);
	name = g_path_get_basename (unescaped_uri);
	gtk_list_store_set (GTK_LIST_STORE (store), &iter,
			    NAME_COL, name,
			    URI_COL, unescaped_uri,
			    ARTIST_COL, _("loading"),
			    SIZE_COL, _("loading"),
			    DSIZE_COL, -1.0,
			    -1);
	g_free (name);

	treepath = gtk_tree_model_get_path (store, &iter);
	ref = gtk_tree_row_reference_new (store, treepath);
	gtk_tree_path_free (treepath);

	/* get info async for the file */
	success = bonfire_audio_disc_get_info_async (disc,
						     unescaped_uri,
						     GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						     GNOME_VFS_FILE_INFO_FOLLOW_LINKS,
						     TRUE,
						     (GetFileInfoAsyncResultFunc) bonfire_audio_disc_new_row_cb,
						     ref);
	g_free (unescaped_uri);

	if (success != BONFIRE_DISC_OK) {
		gtk_tree_row_reference_free (ref);
		return success;
	}

	return BONFIRE_DISC_OK;
}

static BonfireDiscResult
bonfire_audio_disc_add_uri (BonfireDisc *disc,
			    const char *uri)
{
	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_AUDIO_DISC (disc)->priv->notebook), 1);
	return bonfire_audio_disc_add_uri_real (BONFIRE_AUDIO_DISC (disc),
						uri,
						-1);
}

/******************************** Row removing *********************************/
static void
bonfire_audio_disc_remove (BonfireAudioDisc *disc,
			   GtkTreePath *treepath)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	double size;
	char *uri;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	gtk_tree_model_get_iter (model, &iter, treepath);
	gtk_tree_model_get (model, &iter,
			    URI_COL, &uri,
			    DSIZE_COL, &size, -1);
	gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

	if (size != -1.0) {
		disc->priv->size -= size;
		bonfire_audio_disc_size_changed (disc);
	}

#ifdef BUILD_INOTIFY
	bonfire_audio_disc_cancel_monitoring (disc, uri);
#endif
	g_free (uri);
}

static void
bonfire_audio_disc_delete_selected (BonfireDisc *disc)
{
	GtkTreeSelection *selection;
	BonfireAudioDisc *audio;
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GList *list;

	audio = BONFIRE_AUDIO_DISC (disc);
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (audio->priv->tree));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (audio->priv->tree));

	/* we must start by the end for the treepaths to point to valid rows */
	list = gtk_tree_selection_get_selected_rows (selection, &model);
	list = g_list_reverse (list);
	for (; list; list = g_list_remove (list, treepath)) {
		treepath = list->data;
		bonfire_audio_disc_remove (audio, treepath);
		gtk_tree_path_free (treepath);
	}
}

static void
bonfire_audio_disc_clear (BonfireDisc *disc)
{
	BonfireAudioDisc *audio;
	GtkTreeModel *model;

	audio = BONFIRE_AUDIO_DISC (disc);
	bonfire_audio_disc_reset_real (audio);

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (audio->priv->tree));
	gtk_list_store_clear (GTK_LIST_STORE (model));

	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_AUDIO_DISC (disc)->priv->notebook), 0);
	bonfire_disc_size_changed (disc, 0.0, NULL);
}

static void
bonfire_audio_disc_reset (BonfireDisc *disc)
{
	bonfire_audio_disc_clear (disc);
}

/********************************* create track ********************************/
static BonfireDiscResult
bonfire_audio_disc_get_track (BonfireDisc *disc,
			      BonfireDiscTrack *track,
			      gboolean src_format)
{
	char *uri;
	char *title;
	char *artist;
	GtkTreeIter iter;
	GtkTreeModel *model;
	BonfireSongFile *song;
	BonfireAudioDisc *audio;

	audio = BONFIRE_AUDIO_DISC (disc);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (audio->priv->tree));
	if (!gtk_tree_model_get_iter_first (model, &iter))
		return BONFIRE_DISC_ERROR_EMPTY_SELECTION;

	if (src_format)
		track->type = BONFIRE_DISC_TRACK_SOURCE;
	else
		track->type = BONFIRE_DISC_TRACK_AUDIO;

	do {
		if (src_format) {
			BonfireTrackSource *src;

			gtk_tree_model_get (model, &iter,
					    URI_COL, &uri,
					    NAME_COL, &title,
					    ARTIST_COL, &artist,
					    -1);

			song = g_new0 (BonfireSongFile, 1);
			song->uri = uri;
			song->title = title;
			song->artist = artist;
			song->gap = FALSE;

			if (!track->contents.src) {
				track->contents.src = g_new0 (BonfireTrackSource, 1);
				track->contents.src->type = BONFIRE_TRACK_SOURCE_SONG;
			}

			src = track->contents.src;
			src->contents.songs.files = g_slist_append (src->contents.songs.files, song);
		}
		else {
			gtk_tree_model_get (model, &iter,
					    URI_COL, &uri,
					    -1);
			track->contents.uris = g_slist_append (track->contents.uris, uri);
		}

	} while (gtk_tree_model_iter_next (model, &iter));

	return BONFIRE_DISC_OK;
}

/********************************* load track **********************************/
static BonfireDiscResult
bonfire_audio_disc_load_track (BonfireDisc *disc,
			       BonfireDiscTrack *track)
{
	GSList *iter;
	char *uri;

	g_return_val_if_fail (track->type == BONFIRE_DISC_TRACK_AUDIO, FALSE);

	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_AUDIO_DISC (disc)->priv->notebook), 1);

	if (track->contents.uris == NULL)
		return BONFIRE_DISC_ERROR_EMPTY_SELECTION;

	for (iter = track->contents.uris; iter; iter = iter->next) {
		char *escaped_uri;

		uri = iter->data;
		escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
		bonfire_audio_disc_add_uri_real (BONFIRE_AUDIO_DISC (disc),
						 escaped_uri,
						 -1);
		g_free (escaped_uri);
	}

	return BONFIRE_DISC_OK;
}

/*********************************  DND ****************************************/
static int
bonfire_audio_disc_get_dest_path (BonfireAudioDisc *disc,
				  gint x,
				  gint y)
{
	GtkTreeViewDropPosition pos;
	GtkTreePath *treepath;
	GtkTreeModel *model;
	int retval;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW (disc->priv->tree),
					   x,
					   y,
					   &treepath,
					   &pos);

	if (treepath) {
		int *indices;

		if (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER
		||  pos == GTK_TREE_VIEW_DROP_AFTER)
			gtk_tree_path_next (treepath);

		indices = gtk_tree_path_get_indices (treepath);
		retval = indices [0];
	}
	else
		retval = gtk_tree_model_iter_n_children (model, NULL);

	gtk_tree_path_free (treepath);

	return retval;
}

static void
bonfire_audio_disc_drag_data_received_cb (GtkTreeView *tree,
					  GdkDragContext *drag_context,
					  gint x,
					  gint y,
					  GtkSelectionData *selection_data,
					  guint info,
					  guint time,
					  BonfireAudioDisc *disc)
{
	BonfireDiscResult success;
	gboolean result = TRUE;
	char **uri, **uris;
	int pos;

	if (info == TREE_MODEL_ROW)
		return;

	if (selection_data->length <= 0
	||  selection_data->format != 8) {
		gtk_drag_finish (drag_context, FALSE, FALSE, time);
		return;
	}

	/* we get URIS */
	pos = bonfire_audio_disc_get_dest_path (disc, x, y);
	uris = gtk_selection_data_get_uris (selection_data);
	for (uri = uris; *uri != NULL; uri++) {
		success = bonfire_audio_disc_add_uri_real (disc, *uri, pos);
		if (success == BONFIRE_DISC_OK)
			pos ++;

		result = result ? (success == BONFIRE_DISC_OK) : FALSE;
		g_free (*uri);
	}
	g_free (uris);

	gtk_drag_finish (drag_context,
			 result,
			 (drag_context->action == GDK_ACTION_MOVE),
			 time);
}

/********************************** Cell Editing *******************************/
static void
bonfire_audio_disc_display_edited_cb (GtkCellRendererText *renderer,
				      gchar *path_string,
				      gchar *text,
				      BonfireAudioDisc *disc)
{
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeIter row;
	gint col_num;

	col_num = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (renderer), COL_KEY));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	treepath = gtk_tree_path_new_from_string (path_string);
	gtk_tree_model_get_iter (model, &row, treepath);
	gtk_list_store_set (GTK_LIST_STORE (model), &row,
			    col_num, text, -1);
	disc->priv->editing = 0;
}

static void
bonfire_audio_disc_display_editing_started_cb (GtkCellRenderer *renderer,
					       GtkCellEditable *editable,
					       gchar *path,
					       BonfireAudioDisc *disc)
{
	disc->priv->editing = 1;
}

static void
bonfire_audio_disc_display_editing_canceled_cb (GtkCellRenderer *renderer,
						BonfireAudioDisc *disc)
{
	disc->priv->editing = 0;
}

/********************************** Menus **************************************/
static void
bonfire_audio_disc_open_file (BonfireAudioDisc *disc)
{
	char *uri;
	gboolean success;
	GtkTreeIter iter;
	GList *item, *list;
	GSList *uris = NULL;
	GtkTreeModel *model;
	GtkTreePath *treepath;
	GtkTreeSelection *selection;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	list = gtk_tree_selection_get_selected_rows (selection, &model);

	for (item = list; item; item = item->next) {
		treepath = item->data;
		success = gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_path_free (treepath);

		if (!success)
			continue;

		gtk_tree_model_get (model, &iter,
				    URI_COL, &uri, -1);

		uris = g_slist_prepend (uris, uri);
	}
	g_list_free (list);

	bonfire_utils_launch_app (GTK_WIDGET (disc), uris);
	g_slist_foreach (uris, (GFunc) g_free, NULL);
	g_slist_free (uris);
}

static void
bonfire_audio_disc_edit_song_properties (BonfireAudioDisc *disc,
					 GList *list)
{
	int isrc;
	GList *item;
	char *title;
	char *artist;
	int track_num;
	char *composer;
	GtkWidget *props;
	gboolean success;
	GtkTreeIter iter;
	GtkWidget *toplevel;
	GtkTreeModel *model;
	GtkTreePath *treepath;
	GtkResponseType result;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	props = bonfire_song_props_new ();

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	gtk_window_set_transient_for (GTK_WINDOW (props),
				      GTK_WINDOW (toplevel));
	gtk_window_set_modal (GTK_WINDOW (props), TRUE);
	gtk_window_set_position (GTK_WINDOW (props),
				 GTK_WIN_POS_CENTER_ON_PARENT);

	for (item = list; item; item = item->next) {
		treepath = item->data;
		success = gtk_tree_model_get_iter (model, &iter, treepath);

		if (!success)
			continue;

		gtk_tree_model_get (model, &iter,
				    TRACK_NUM_COL, &track_num,
				    NAME_COL, &title,
				    ARTIST_COL, &artist,
				    COMPOSER_COL, &composer,
				    ISRC_COL, &isrc, 
				    -1);

		gtk_widget_show_all (GTK_WIDGET (props));

		bonfire_song_props_set_properties (BONFIRE_SONG_PROPS (props),
						   track_num,
						   artist,
						   title,
						   composer,
						   isrc);

		if (artist)
			g_free (artist);
		if (title)
			g_free (title);
		if (composer)
			g_free (composer);

		result = gtk_dialog_run (GTK_DIALOG (props));
		gtk_widget_hide (GTK_WIDGET (props));

		if (result != GTK_RESPONSE_ACCEPT)
			break;

		bonfire_song_props_get_properties (BONFIRE_SONG_PROPS (props),
						   &artist,
						   &title,
						   &composer,
						   &isrc);

		gtk_list_store_set (GTK_LIST_STORE (model), &iter,
				    NAME_COL, title,
				    ARTIST_COL, artist,
				    COMPOSER_COL, composer,
				    ISRC_COL, isrc, 
				    -1);

		g_free (artist);
		g_free (title);
		g_free (composer);
	}

	gtk_widget_destroy (props);
}

static void
bonfire_audio_disc_open_activated_cb (GtkAction *action,
				      BonfireAudioDisc *disc)
{
	bonfire_audio_disc_open_file (disc);
}

static void
bonfire_audio_disc_edit_information_cb (GtkAction *action,
					BonfireAudioDisc *disc)
{
	GList *list;
	GtkTreeModel *model;
	GtkTreeSelection *selection;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	list = gtk_tree_selection_get_selected_rows (selection, &model);

	bonfire_audio_disc_edit_song_properties (disc, list);

	g_list_foreach (list, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (list);
}

static void
bonfire_audio_disc_delete_activated_cb (GtkAction *action,
					BonfireDisc *disc)
{
	bonfire_audio_disc_delete_selected (disc);
}

static void
bonfire_audio_disc_clipboard_text_cb (GtkClipboard *clipboard,
				      const char *text,
				      BonfireAudioDisc *disc)
{
	char **array;
	char **item;
	char *uri;

	array = g_strsplit_set (text, "\n\r", 0);
	item = array;
	while (*item) {
		if (**item != '\0') {
			uri = gnome_vfs_make_uri_canonical (*item);
			bonfire_audio_disc_add_uri_real (disc,
							 uri,
							 -1);
			g_free (uri);
		}

		item++;
	}
}

static void
bonfire_audio_disc_clipboard_targets_cb (GtkClipboard *clipboard,
					 GdkAtom *atoms,
					 gint n_atoms,
					 BonfireAudioDisc *disc)
{
	GdkAtom *iter;
	char *target;

	iter = atoms;
	while (n_atoms) {
		target = gdk_atom_name (*iter);

		if (!strcmp (target, "x-special/gnome-copied-files")
		||  !strcmp (target, "UTF8_STRING")) {
			gtk_clipboard_request_text (clipboard,
						    (GtkClipboardTextReceivedFunc) bonfire_audio_disc_clipboard_text_cb,
						    disc);
			g_free (target);
			return;
		}

		g_free (target);
		iter++;
		n_atoms--;
	}
}

static void
bonfire_audio_disc_paste_activated_cb (GtkAction *action,
				       BonfireAudioDisc *disc)
{
	GtkClipboard *clipboard;

	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_clipboard_request_targets (clipboard,
				       (GtkClipboardTargetsReceivedFunc) bonfire_audio_disc_clipboard_targets_cb,
				       disc);
}

static gboolean
bonfire_audio_disc_button_released_cb (GtkTreeView *tree,
				      GdkEventButton *event,
				      BonfireAudioDisc *disc)
{
	if (event->button == 3) {
		GtkTreeSelection *selection;
		int nb_selected;

		selection = gtk_tree_view_get_selection (tree);
		nb_selected = gtk_tree_selection_count_selected_rows (selection);

		bonfire_utils_show_menu (nb_selected, disc->priv->manager, event);
		return TRUE;
	}
	else if (event->button == 1) {
		GList *list;
		gboolean result;
		GtkTreePath *treepath = NULL;

		result = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (disc->priv->tree),
							event->x,
							event->y,
							&treepath,
							NULL,
							NULL,
							NULL);
		if (!result || !treepath)
			return FALSE;

		if (disc->priv->selected_path)
			gtk_tree_path_free (disc->priv->selected_path);

		disc->priv->selected_path = treepath;
		bonfire_disc_selection_changed (BONFIRE_DISC (disc));

		list = g_list_prepend (NULL, treepath);
		if (event->type == GDK_2BUTTON_PRESS)
			bonfire_audio_disc_edit_song_properties (disc, list);

		g_list_free (list);
	}

	return FALSE;
}

/********************************** key press event ****************************/
static gboolean
bonfire_audio_disc_key_released_cb (GtkTreeView *tree,
				    GdkEventKey *event,
				    BonfireAudioDisc *disc)
{
	if (disc->priv->editing)
		return FALSE;

	if (event->keyval == GDK_KP_Delete || event->keyval == GDK_Delete) {
		bonfire_audio_disc_delete_selected (BONFIRE_DISC (disc));
	}

	return FALSE;
}

/**********************************               ******************************/
static char *
bonfire_audio_disc_get_selected_uri (BonfireDisc *disc)
{
	char *uri;
	GtkTreeIter iter;
	GtkTreeModel *model;
	BonfireAudioDisc *audio;

	audio = BONFIRE_AUDIO_DISC (disc);
	if (!audio->priv->selected_path)
		return NULL;

	/* we are asked for just one uri so return the first one */
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (audio->priv->tree));
	if (!gtk_tree_model_get_iter (model, &iter, audio->priv->selected_path)) {
		gtk_tree_path_free (audio->priv->selected_path);
		audio->priv->selected_path = NULL;
		return NULL;
	}
	else
		gtk_tree_model_get (model, &iter, URI_COL, &uri, -1);
	

	return uri;
}

/********************************** Monitoring *********************************/
#ifdef BUILD_INOTIFY

static void
bonfire_audio_disc_start_monitoring (BonfireAudioDisc *disc,
				     const char *uri)
{
	if (disc->priv->notify && !strncmp (uri, "file://", 7)) {
		int dev_fd;
		char *path;
		__u32 mask;
		int wd;

		path = gnome_vfs_get_local_path_from_uri (uri);
		dev_fd = g_io_channel_unix_get_fd (disc->priv->notify);
		mask = IN_MODIFY
		    | IN_ATTRIB
		    | IN_MOVED_FROM
		    | IN_DELETE | IN_DELETE_SELF | IN_MOVE_SELF;

		wd = inotify_add_watch (dev_fd, path, mask);
		if (wd != -1)
			g_hash_table_insert (disc->priv->monitored,
					     GINT_TO_POINTER (wd),
					     g_strdup (uri));
		else
			g_warning ("ERROR creating watch for local file %s : %s\n",
				   path,
				   strerror (errno));

		g_free (path);
	}
}

struct _BonfireMonitoredRemoveData {
	BonfireAudioDisc *disc;
	char *uri;
};
typedef struct _BonfireMonitoredRemoveData BonfireMonitoredRemoveData;

static gboolean
_foreach_monitored_remove_uri_cb (int wd,
				  const char *uri,
				  BonfireMonitoredRemoveData *data)
{
	int dev_fd;

	if (strcmp (data->uri, uri))
		return FALSE;

	dev_fd = g_io_channel_unix_get_fd (data->disc->priv->notify);
	inotify_rm_watch (dev_fd, wd);

	return TRUE;
}

static void
bonfire_audio_disc_cancel_monitoring (BonfireAudioDisc *disc,
				      const char *uri)
{
	BonfireMonitoredRemoveData callback_data;

	if (!disc->priv->notify)
		return;

	callback_data.uri = (char *) uri;
	callback_data.disc = disc;
	g_hash_table_foreach_remove (disc->priv->monitored,
				     (GHRFunc) _foreach_monitored_remove_uri_cb,
				     &callback_data);
}

static void
bonfire_audio_disc_inotify_remove (BonfireAudioDisc *disc,
				   const char *uri)
{
	double size;
	char *row_uri;
	GtkTreeIter iter;
	GtkWidget *dialog;
	GtkWidget *toplevel;
	GtkTreeModel *model;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	if (!gtk_tree_model_get_iter_first (model, &iter))
		return;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_CLOSE,
					 _("File \"%s\" was removed from the file system:"),
					 uri);

	gtk_window_set_title (GTK_WINDOW (dialog), _("File deletion noticed"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("it will be removed from the project."));

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	do {
		gtk_tree_model_get (model, &iter,
				    URI_COL, &row_uri,
				    DSIZE_COL, &size, -1);

		if (!strcmp (uri, row_uri)) {
			g_free (row_uri);

			if (size != -1.0) {
				disc->priv->size -= size;
				bonfire_audio_disc_size_changed (disc);
			}

			if (gtk_list_store_remove (GTK_LIST_STORE (model), &iter))
				continue;
			else
				break;
		}

		g_free (row_uri);
	} while (gtk_tree_model_iter_next (model, &iter));

	bonfire_audio_disc_size_changed (disc);
	bonfire_audio_disc_cancel_monitoring (disc, uri);
}

static void
bonfire_audio_disc_inotify_attributes_changed_cb (BonfireAudioDisc *disc,
						  const char *escaped_uri,
						  GnomeVFSResult result,
						  const GnomeVFSFileInfo *info,
						  const BonfireMetadata *metadata,
						  gpointer null_data)
{
	gboolean readable;
	char *uri;

	uri = gnome_vfs_unescape_string_for_display (escaped_uri);

	if (result != GNOME_VFS_OK)
		readable = FALSE;
	else if (!GNOME_VFS_FILE_INFO_LOCAL (info))
		readable = TRUE;
	else if (getuid () == info->uid
	     && (info->permissions & GNOME_VFS_PERM_USER_READ))
		readable = TRUE;
	else if (getgid () == info->gid
	     && (info->permissions & GNOME_VFS_PERM_GROUP_READ))
		readable = TRUE;
	else if (info->permissions & GNOME_VFS_PERM_OTHER_READ)
		readable = TRUE;
	else
		readable = FALSE;

	if (!readable)
		bonfire_audio_disc_inotify_remove (disc, uri);

	g_free (uri);
}

static gboolean
bonfire_audio_disc_inotify_attributes_changed (BonfireAudioDisc *disc,
					       const char *uri)
{
	BonfireDiscResult success;

	success = bonfire_audio_disc_get_info_async (disc,
						     uri,
						     GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS|
						     GNOME_VFS_FILE_INFO_FOLLOW_LINKS,
						     FALSE,
						     bonfire_audio_disc_inotify_attributes_changed_cb,
						     NULL);

	if (success != BONFIRE_DISC_OK)
		return FALSE;

	return TRUE;
}

static void
bonfire_audio_disc_inotify_modify (BonfireAudioDisc *disc,
				   const char *uri,
				   BonfireMetadata *metadata)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	char *row_uri;
	double size;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (disc->priv->tree));
	if (gtk_tree_model_get_iter_first (model, &iter))
		return;

	do {
		gtk_tree_model_get (model, &iter,
				    URI_COL, &row_uri,
				    SIZE_COL, &size,
				    -1);

		if (!strcmp (uri, row_uri)) {
			disc->priv->size -= size;
			disc->priv->size += BONFIRE_TIME_TO_SIZE (metadata->len);

			bonfire_audio_disc_set_row_from_metadata (disc,
								  model,
								  &iter,
								  metadata);
		}

		g_free (row_uri);
	} while (gtk_tree_model_iter_next (model, &iter));

	bonfire_audio_disc_size_changed (disc);
}

static gboolean
bonfire_audio_disc_inotify_monitor_cb (GIOChannel *channel,
				       GIOCondition condition,
				       BonfireAudioDisc *disc)
{
	struct inotify_event event;
	GError *err = NULL;
	GIOStatus status;
	char *monitored;
	guint size;
	char *name;

	while (condition & G_IO_IN) {
		monitored = NULL;

		status = g_io_channel_read_chars (channel,
						  (char *) &event,
						  sizeof (struct inotify_event),
						  &size,
						  &err);

		if (status == G_IO_STATUS_EOF)
			return TRUE;

		if (event.len) {
			name = g_new (char, event.len + 1);
			name[event.len] = '\0';
			status = g_io_channel_read_chars (channel,
							  name,
							  event.len,
							  &size,
							  &err);
			if (status != G_IO_STATUS_NORMAL) {
				g_warning ("Error reading inotify: %s\n",
					   err ? "Unknown error" : err->message);
				g_error_free (err);
				return TRUE;
			}
		}
		else
			name = NULL;

		/* look for ignored signal usually following deletion */
		if (event.mask & IN_IGNORED) {
			g_hash_table_remove (disc->priv->monitored,
					     GINT_TO_POINTER (event.wd));
			if (name)
				g_free (name);
			condition = g_io_channel_get_buffer_condition (channel);
			continue;
		}

		monitored = g_hash_table_lookup (disc->priv->monitored,
						 GINT_TO_POINTER (event.wd));
		if (!monitored) {
			condition = g_io_channel_get_buffer_condition (channel);
			continue;
		}

		if (event.mask & (IN_DELETE_SELF | IN_DELETE | IN_MOVED_FROM | IN_MOVE_SELF))
			bonfire_audio_disc_inotify_remove (disc, monitored);
		else if (event.mask & IN_ATTRIB)
			bonfire_audio_disc_inotify_attributes_changed (disc,
								       monitored);
		else if (event.mask & IN_MODIFY) {
			BonfireMetadata *metadata;
			GError *error = NULL;
			char *escaped_uri;

			escaped_uri = gnome_vfs_escape_host_and_path_string (monitored);
			metadata = bonfire_metadata_new (escaped_uri);
			g_free (escaped_uri);

			if (!metadata
			||  !bonfire_metadata_get_sync (metadata, &error)) {
				if (error) {
					g_warning ("ERROR : %s\n", error->message);
					g_error_free (error);
					g_object_unref (metadata);
				}

				bonfire_audio_disc_inotify_remove (disc, monitored);
				continue;
			}

			bonfire_audio_disc_inotify_modify (disc,
							   monitored,
							   metadata);
			g_object_unref (metadata);
		}

		condition = g_io_channel_get_buffer_condition (channel);
	}

	return TRUE;
}
#endif
