//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>
#include <glibmm.h>

#include "podcast-types.hh"
#include "podcast-utils.hh"
#include "podcast-libxml2-sax.hh"

#include "uri++.hh"
#include "util.hh"

using namespace Bmp::PodcastBackend;
using namespace std;
using namespace Glib;

namespace
{
  enum ElementId
  {
    E_RSS,
    E_CHANNEL,
    E_TITLE,
    E_LINK,
    E_DESCRIPTION,
    E_LANGUAGE,
    E_TTL,
    E_COPYRIGHT,
    E_PUB_DATE,
    E_CATEGORY,
    E_ITEM,
    E_ENCLOSURE,
    E_GUID,
    E_IMAGE,
    E_URL,
  };
}

#include "parser/libxml2-sax-base.hh"
using Bmp::XPath;

namespace
{
  Element_tag elements[] =
  {
      { "rss" },
      { "channel" },
      { "title" },
      { "link" },
      { "description" },
      { "language" },
      { "ttl" },
      { "copyright" },
      { "pubDate" },
      { "category" },
      { "item" },
      { "enclosure" },
      { "guid" },
      { "image" },
      { "url" },
  };
  
  const char * A_VERSION        = "version";
  const char * A_URL            = "url";
  const char * A_LENGTH         = "length";
  const char * A_IS_PERMA_LINK  = "isPermaLink";
  const char * A_TYPE           = "type";
  const char * A_HREF           = "href";

  struct PodcastParserContext : public ParseContextBase
  {
    PodcastItem     m_item;
    bool            m_block_parse;

    Podcast                     & m_cast;
    PodcastOverlayItemsMap const& m_overlays;

    PodcastParserContext (PodcastOverlayItemsMap const& overlays, Podcast & cast)
      : ParseContextBase(),
        m_cast     (cast),
        m_overlays    (overlays),
        m_block_parse (false) {}
  };

  typedef std::map < ustring, ustring > ElementAttributes; 

#define DEFAULT_REFS \
  PodcastParserContext & context (static_cast<PodcastParserContext&>(_context));  \
  Podcast & cast (context.m_cast);   \
  PodcastItem & item (context.m_item);  \
  if (context.m_block_parse)  \
  {                           \
    return;                   \
  }

  //////////////////////////////////////////////////////////////////////////////

  namespace Handlers
  {
    HANDLER(rss)
    {
      DEFAULT_REFS
      
      if (props.find ("version") != props.end())
      {
        ustring version = props.find ("version")->second;
        if (version != "" && (version != "2.0"))
        {
          g_warning ("%s: XML Is not RSS 2.0", G_STRFUNC);
          context.m_block_parse = true;
          return;
        }
      }
    }

    namespace Item
    {
      HANDLER(item)
      {
        DEFAULT_REFS
        context.m_item = PodcastItem();
      }

      HANDLER(enclosure)
      {
        DEFAULT_REFS

        if (props.find (A_URL) != props.end())
            context.m_item.enclosure_url = props.find(A_URL)->second;

        if (props.find (A_LENGTH) != props.end())
            context.m_item.enclosure_length = strtol (props.find(A_LENGTH)->second.c_str(), 0, 10);

        if (props.find (A_TYPE) != props.end())
            context.m_item.enclosure_type = props.find(A_TYPE)->second;
      }

      HANDLER(guid)
      {
        DEFAULT_REFS

        if (props.find (A_IS_PERMA_LINK) != props.end())
            context.m_item.uid_permalink = bool (props.find(A_IS_PERMA_LINK)->second == "true");
      }
    }

    HANDLER(image)
    {
      DEFAULT_REFS

      if (props.find (A_HREF) != props.end() && context.m_cast.image_url.empty())
      {
        context.m_cast.image_url = props.find (A_HREF)->second;
      }
    }
  }

  Handler_tag handlers[] = 
  {
    { XPath(elements[E_RSS]), 
      &Handlers::rss             },

    { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_IMAGE]), 
      &Handlers::image           },

    { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]), 
      &Handlers::Item::item      },

    { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_ENCLOSURE]),
      &Handlers::Item::enclosure },

    { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_GUID]),
      &Handlers::Item::guid      },
  };
  //---------
  
  namespace HandlersEnd
  {
    namespace Item
    {
      HANDLER_END(item)
      {
        DEFAULT_REFS
        
        if (context.m_item.enclosure_url.empty())
        {
          g_warning ("%s: Enclosure URL is empty", G_STRLOC);
          return;
        }

        try{
          Bmp::URI u (context.m_item.enclosure_url, true);
          }
        catch (...)
          {
            g_warning ("%s: Couldn't parse enclosure URL: '%s'", G_STRLOC, context.m_item.enclosure_url.c_str());
            return;
          }

        // Some feeds lack a GUID, so use the title (and hope it's unique).
        if (context.m_item.uid_value.empty())
        {
          context.m_item.uid_value = context.m_item.title;
        }

        PodcastOverlayItemsMap::const_iterator i (context.m_overlays.find (context.m_item.uid_value));
             
        if (i != context.m_overlays.end())
        {
          context.m_item.overlay (i->second);
        }
 
        std::string item_filename = cast_item_file (context.m_cast, context.m_item);
        if (!context.m_item.downloaded && file_test (item_filename, FILE_TEST_EXISTS))
        {
          if (file_test (item_filename, FILE_TEST_EXISTS))
          {
            context.m_item.filename = item_filename; 
            context.m_item.downloaded = true;
          }
        }
        context.m_cast.items.insert (std::make_pair (context.m_item.uid_value, context.m_item));
      }
    }
  }
  
  HandlerEnd_tag handlers_end[] =
  {
      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]), 
        &HandlersEnd::Item::item },
  };
  //---------
  
  namespace HandlersPCDATA
  {
    HANDLER_PCDATA(title)
    {
      DEFAULT_REFS
      cast.title += text;
    }

    HANDLER_PCDATA(link)
    {
      DEFAULT_REFS
      cast.link += text;
    }

    HANDLER_PCDATA(description)
    {
      DEFAULT_REFS
      cast.description += text;
    }

    HANDLER_PCDATA(language)
    {
      DEFAULT_REFS
      cast.language += text;
    }

    HANDLER_PCDATA(ttl)
    {
      DEFAULT_REFS
      cast.ttl = g_ascii_strtoull (text.c_str(), NULL, 10);
    }

    HANDLER_PCDATA(copyright)
    {
      DEFAULT_REFS
      cast.copyright += text;
    }

    HANDLER_PCDATA(pubDate)
    {
      DEFAULT_REFS
      cast.pub_date += text;
    }

    HANDLER_PCDATA(category)
    {
      DEFAULT_REFS
      cast.category += text;
    }

    //// ITEM

    namespace Item
    {
      HANDLER_PCDATA(title)
      {
        DEFAULT_REFS
        item.title += text;
      }

      HANDLER_PCDATA(link)
      {
        DEFAULT_REFS
        item.link += text;
      }

      HANDLER_PCDATA(description)
      {
        DEFAULT_REFS
        item.description += text;
      }

      HANDLER_PCDATA(guid)
      {
        DEFAULT_REFS
        item.uid_value += text;
      }

      HANDLER_PCDATA(copyright)
      {
        DEFAULT_REFS
        item.copyright += text;
      }

      HANDLER_PCDATA(pubDate)
      {
        DEFAULT_REFS

        item.pub_date += text;

        time_t item_time = Bmp::Util::parseRFC822Date (text.c_str());
        item.pub_date_unix = item_time; 
        if (cast.most_recent_item < item_time)
        {
          cast.most_recent_item = item_time;
        }
      }

      HANDLER_PCDATA(category)
      {
        DEFAULT_REFS
        item.category += text;
      }
    }

    HANDLER_PCDATA(image_url)
    {
      DEFAULT_REFS
      cast.image_url = text;
    }

    HANDLER_PCDATA(image_title)
    {
      DEFAULT_REFS
      cast.image_title += text;
    }

    HANDLER_PCDATA(image_link)
    {
      DEFAULT_REFS
      cast.image_link += text;
    }

    HANDLER_PCDATA(image_description)
    {
      DEFAULT_REFS
      cast.image_description += text;
    }
  }

  HandlerPCDATA_tag handlers_pcdata[] = 
  {
      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_TITLE]),
        &HandlersPCDATA::title       },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_LINK]),
        &HandlersPCDATA::link        },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_DESCRIPTION]),
        &HandlersPCDATA::description },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_LANGUAGE]),
        &HandlersPCDATA::language    },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_TTL]),
        &HandlersPCDATA::ttl         },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_COPYRIGHT]),
        &HandlersPCDATA::copyright   },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_PUB_DATE]),
        &HandlersPCDATA::pubDate     },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_CATEGORY]),
        &HandlersPCDATA::category    },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_TITLE]),
        &HandlersPCDATA::Item::title       },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_LINK]),
        &HandlersPCDATA::Item::link        },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_DESCRIPTION]),
        &HandlersPCDATA::Item::description },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_GUID]),
        &HandlersPCDATA::Item::guid },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_COPYRIGHT]),
        &HandlersPCDATA::Item::copyright   },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_PUB_DATE]),
        &HandlersPCDATA::Item::pubDate     },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_ITEM]) / XPath(elements[E_CATEGORY]),
        &HandlersPCDATA::Item::category    },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_IMAGE]) / XPath(elements[E_TITLE]),
        &HandlersPCDATA::image_title       },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_IMAGE]) / XPath(elements[E_LINK]),
        &HandlersPCDATA::image_link        },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_IMAGE]) / XPath(elements[E_DESCRIPTION]),
        &HandlersPCDATA::image_description },

      { XPath(elements[E_RSS]) / XPath(elements[E_CHANNEL]) / XPath(elements[E_IMAGE]) / XPath(elements[E_URL]),
        &HandlersPCDATA::image_url         },
  };
  //---------
}

namespace Bmp
{
  namespace PodcastBackend
  {
    int podcast_rss2_parse (PodcastOverlayItemsMap const& overlays, Podcast & cast, std::string const& data)
    {
      PodcastParserContext context (overlays, cast);

      for (unsigned int n = 0; n < G_N_ELEMENTS(handlers); ++n)
       {
         context.mHandlers.insert (std::make_pair (handlers[n].elementId, handlers[n].handler));
       }
       
       // handler/end 
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_end); ++n)
       {
         context.mHandlersEnd.insert (std::make_pair (handlers_end[n].elementId, handlers_end[n].handler));
       }
       
       // handler/pcdata
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_pcdata); ++n)
       {
         context.mHandlersPCDATA.insert (std::make_pair (handlers_pcdata[n].elementId, handlers_pcdata[n].handler));
       }
       
       // name <-> id map 
       for (unsigned int n = 0; n < G_N_ELEMENTS(elements); ++n)
       {
         context.mNameIdMap.insert (std::make_pair (elements[n], ElementId (n)));
       }

      return SaxParserBase::xml_base_parse(data, context);
    }
  }
}
