//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>

#include <boost/algorithm/string.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/format.hpp>

#include "main.hh"
#include "paths.hh"

#include "util_file.hh"
#include "util.hh"

#include "vfs.hh"

namespace
{
  inline bool
  is_module (std::string const &basename)
  {
    return Bmp::Util::str_has_suffix_nocase (basename.c_str (), G_MODULE_SUFFIX);
  }

  enum
  {
    LIB_BASENAME,
    LIB_PLUGNAME,
    LIB_SUFFIX
  };
}

namespace Bmp
{
    namespace VFS
    {
      bool
      VFS::load_plugins_transport (std::string const& path,
                                   std::string const& type)
      {
        using namespace boost::algorithm;

        std::string basename (Glib::path_get_basename (path));
        std::string pathname (Glib::path_get_dirname  (path));

        if (!is_module (basename)) return false;

        std::vector<std::string> subs;
        split (subs, basename, is_any_of ("_."));
        std::string dir   = Glib::build_filename (BMP_VFS_PLUGIN_DIR, type);
        std::string name  = type + std::string("_") + subs[LIB_PLUGNAME];
        std::string mpath = Glib::Module::build_path (dir, name);

        Glib::Module module (mpath, Glib::MODULE_BIND_LAZY);

        if (module)
        {
          module.make_resident ();

          PluginTransportBase* (*plugin_create) ();
          if (!module.get_symbol ("plugin_create", (void*&)plugin_create))
          {
            g_critical ("Unable to acquire plugin_create symbol from library %s", mpath.c_str ());
          }
          else
          {
            PluginTransportBase* plugin = plugin_create ();
            transport_plugins.insert (std::make_pair (subs[LIB_PLUGNAME], plugin));
            g_message ("Transport plugin '%s' loaded.", subs[LIB_PLUGNAME].c_str ()); 
          }
        }
        else
        {
          g_critical ("Failed to load Transport plugin '%s': %s", mpath.c_str (), module.get_last_error().c_str() );
        }

        return false;
      }

      bool
      VFS::load_plugins_container (std::string const& path,
                                   std::string const& type)
      {
        using namespace boost::algorithm;

        std::string basename (Glib::path_get_basename (path));
        std::string pathname (Glib::path_get_dirname  (path));

        if (!is_module (basename)) return false;

        std::vector<std::string> subs;
        split (subs, basename, is_any_of ("_."));
        std::string dir   = Glib::build_filename (BMP_VFS_PLUGIN_DIR, type);
        std::string name  = type + std::string("_") + subs[LIB_PLUGNAME];
        std::string mpath = Glib::Module::build_path (dir, name);

        Glib::Module module (mpath, Glib::MODULE_BIND_LAZY);
      
        if (module)
        {
          module.make_resident ();

          PluginContainerBase* (*plugin_create) ();
          if (!module.get_symbol ("plugin_create", (void*&)plugin_create))
          {
            g_critical ("Unable to acquire plugin_create symbol from library %s", mpath.c_str ());
          }
          else
          {
            PluginContainerBase* plugin = plugin_create ();
            container_plugins.insert (std::make_pair (subs[LIB_PLUGNAME], plugin));
            ExportData exportdata = plugin->get_export_data ();
            g_message ("Container plugin '%s' loaded", exportdata.description.c_str());
          }
        }
        else
        {
          g_critical ("Failed to load Container plugin '%s': %s", mpath.c_str (), module.get_last_error().c_str() );
        }

        return false;
      }

      VFS::VFS ()
      {
          Util::dir_for_each_entry (BMP_TRANSPORT_PLUGIN_DIR,
                                    sigc::bind (sigc::mem_fun (*this, &Bmp::VFS::VFS::load_plugins_transport), "transport"));

          Util::dir_for_each_entry (BMP_CONTAINER_PLUGIN_DIR,
                                    sigc::bind (sigc::mem_fun (*this, &Bmp::VFS::VFS::load_plugins_container), "container"));
      }

      VFS::~VFS ()
      {
      }

      void
      VFS::get_containers (ExportDataList& list)
      {
        for (ContainerPlugins::iterator i = container_plugins.begin(); i != container_plugins.end(); ++i)
        {
          PluginContainerBase *container = (*i).second; 
          ExportData exportdata = container->get_export_data (); 

          if (!exportdata.extension.empty())
          {
            list.push_back (exportdata);
          }
        }
      }

      bool
      VFS::has_container (Glib::ustring const& uri)
      {
        for (ContainerPlugins::iterator i = container_plugins.begin(); i != container_plugins.end(); ++i)
        {
          PluginContainerBase *container = (*i).second; 
          ExportData edata = container->get_export_data();
          if (container->can_process (uri)) return true;
        }
        return false;
      }

      void 
      VFS::read  (Handle & handle,
                  ProcessingFlags flags)
      {
        PluginTransportBase *transport = 0;
        Glib::ustring uri = handle.get_uri ();
    
        if (flags != TRANSPORT)
          throw InvalidFlagsError();

        if (flags & TRANSPORT)
          {
            for (TransportPlugins::iterator i = transport_plugins.begin () ; i != transport_plugins.end() ; ++i)
              {
                transport = i->second;     
                if (transport->can_process (uri)) break;
              }

            if (!transport)
              throw ProcessingError ((boost::format ("No transport found for URI %s") % std::string(uri)).str());
            else
              if (!transport->handle_read (handle))
                throw TransportError ((boost::format ("Transport was unable to read from URI %s") % std::string(uri)).str());
        }
      }

      void 
      VFS::read  (Handle & handle,
                  VUri & list,
                  ProcessingFlags flags)
      {
        PluginContainerBase *container = 0;
        PluginTransportBase *transport = 0;
        Glib::ustring uri = handle.get_uri ();
    
        if (flags == 0)
          throw InvalidFlagsError(); 

        if (flags & TRANSPORT)
        {
          for (TransportPlugins::iterator i = transport_plugins.begin () ; i != transport_plugins.end () ; ++i)
            {
              transport = i->second;     
              if (transport->can_process (uri)) break;
            }

          if (!transport)
            throw ProcessingError ((boost::format ("No transport found for URI %s") % std::string(uri)).str());
          else
            if (!transport->handle_read (handle))
              throw TransportError ((boost::format ("Transport was unable to read from URI %s") % std::string(uri)).str());
        }

        if (flags & CONTAINER)
        {
          for (ContainerPlugins::iterator i = container_plugins.begin () ; i != container_plugins.end () ; ++i)
                                         
            {
              container = i->second;     
              ExportData exportdata = container->get_export_data ();
              if (container->can_process (uri)) break;
            }

          if (!container)
            throw ProcessingError ((boost::format ("No container found for URI %s") % std::string(uri)).str());
          else
            if (!container->handle_read (handle, list))
              throw ContainerError ((boost::format ("Container was unable to process URI %s") % std::string(uri)).str());
        }
      }

      void
      VFS::write (VUri const& list,
                  Glib::ustring const& uri,
                  std::string const& container_name)
      {
        Handle handle (uri); 
        PluginContainerBase *container = 0;
        PluginTransportBase *transport = 0;

        ContainerPlugins::iterator iter = container_plugins.find (container_name);
        container = (*iter).second;
      
        if (!container->handle_write (handle, list))
          {
            throw TransportError ((boost::format ("Container was unable to write list for URI %s") % std::string(uri)).str());
          }

        for (TransportPlugins::iterator iter  = transport_plugins.begin ();
                                        iter != transport_plugins.end (); ++iter)
          {
            transport = (*iter).second;     
            if (transport->can_process (uri)) break;
          }

        if (!transport)
          throw ProcessingError ((boost::format ("No transport found for URI %s") % std::string(uri)).str());
        else
          if (!transport->handle_write (handle))
            throw TransportError ((boost::format ("Transport was unable to process URI %s") % std::string(uri)).str());
      } 
  } // !VFS:: 
} // !Bmp::
