//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

// Musicbrainz
#include <musicbrainz/musicbrainz.h>

#include "uri++.hh"
#include "util.hh"
#include "util_file.hh"
#include "mbxml.hh"
#include "network.hh"

#include "audio.hh"

#include "debug.hh"
#include "stock.hh"
#include "paths.hh"
#include "main.hh"

#include "x_library.hh"
#include "x_play.hh"
#include "x_vfs.hh"
#include "x_lastfm.hh"
#include "x_mcsbind.hh"

#include "ui_toolbox.hh"
#include "ui-part-lastfm.hh"

#include "ui-banner-image.hh"

namespace Bmp
{
  namespace UiPart
  {

    guint
    LASTFM::add_ui ()
    {
      return 0;
    };

    LASTFM::~LASTFM ()
    {
    }

    LASTFM::LASTFM (Glib::RefPtr<Gnome::Glade::Xml> const&  xml,
                    Glib::RefPtr<Gtk::UIManager>            ui_manager)

        : PlaybackSource (PlaybackSource::NONE, PlaybackSource::F_PHONY_NEXT),
          Base (xml, ui_manager),
          m_streamlist_created (false),
          m_init (true)
         
    {
      using namespace Glib;
      using namespace Gtk;
      using namespace Gdk;

      if (!Network::check_connected())
        {
          m_ref_xml->get_widget ("vbox-lastfm")->set_sensitive (false);
          return;
        }

      m_audio_empty =
        Gdk::Pixbuf::create_from_file
              (Glib::build_filename (BMP_IMAGE_DIR, BMP_COVER_IMAGE_EMPTY))->scale_simple
                                    (160, 160, Gdk::INTERP_HYPER); 

      // Images
      m_ref_xml->get_widget ("last-fm-queue-size", last_fm_queue_size);
      m_ref_xml->get_widget ("last-fm-submit-enable", last_fm_enable);
      m_ref_xml->get_widget ("last-fm-netstatus", last_fm_netstatus);

      m_ref_xml->get_widget ("lastfm-coverimage", lastfm_coverimage);
      m_ref_xml->get_widget ("lastfm-connection-state", lastfm_connection_state);

      // Labels
      m_ref_xml->get_widget ("lastfm-radio-artist", lastfm_radio_artist);
      m_ref_xml->get_widget ("lastfm-radio-album", lastfm_radio_album);
      m_ref_xml->get_widget ("lastfm-radio-track", lastfm_radio_track);
      m_ref_xml->get_widget ("lastfm-current-station", lastfm_radio_station);
      m_ref_xml->get_widget ("lastfm-queue-submit-error", lastfm_queue_submit_error);
      m_ref_xml->get_widget ("lastfm-connection-message", lastfm_connection_message);

      m_ref_xml->get_widget ("lastfm-love", b_love);
      m_ref_xml->get_widget ("lastfm-skip", b_skip);
      m_ref_xml->get_widget ("lastfm-ban",  b_ban);

      m_ref_xml->get_widget ("lastfm-station", e_station);

      e_station->signal_activate().connect
          (sigc::mem_fun (this, &LASTFM::tune));

      m_ref_xml->get_widget ("hbox-lastfm-submit-error", hbox_lastfm_submit_error);

      m_ref_xml->get_widget_derived ("lastfm-stations", m_treeview_stations);

      // We construct a helper EntryCompletion for the lastfm:// URL entry
      CompletionRecord record;
      Glib::RefPtr<Gtk::ListStore> store = Gtk::ListStore::create (record);
      Gtk::TreeModel::iterator m_iter;

      m_iter = store->append ();
      (*m_iter)[record.text] = "lastfm://";
      m_iter = store->append ();
      (*m_iter)[record.text] = "lastfm://user/";
      m_iter = store->append ();
      (*m_iter)[record.text] = "lastfm://globaltags/";

      Glib::RefPtr<Gtk::EntryCompletion> completion (Gtk::EntryCompletion::create());
      completion->set_model (store); 
      completion->set_text_column (0);
      completion->set_popup_completion (false);
      completion->set_inline_completion (true);
      completion->set_minimum_key_length (3);
      completion->property_popup_single_match() = false;
      e_station->set_completion (completion);
      
      dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("station-list-update"))->
        signal_clicked().connect
        (sigc::mem_fun (this, &Bmp::UiPart::LASTFM::station_list_update));

      m_treeview_stations->station_activated().connect
          (sigc::mem_fun (this, &LASTFM::set_uri));

      b_love->signal_clicked().connect
          (sigc::mem_fun (this, &LASTFM::love));

      b_skip->signal_clicked().connect
          (sigc::mem_fun (this, &LASTFM::skip));

      b_ban->signal_clicked().connect
          (sigc::mem_fun (this, &LASTFM::ban));

      b_love->set_sensitive (false);
      b_skip->set_sensitive (false);
      b_ban->set_sensitive  (false);

      Glib::RefPtr<Gdk::Pixbuf> left, right, slice;
      left  = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM, "header-logo-left.png"));
      right = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM, "header-logo-right.png"));
      slice = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM, "header-logo-slice.png"));

      Glib::RefPtr<Gdk::Pixbuf> p;

      p = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM,
         "lastfm-station.png"));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("lastfm-i-station"))->set (p);

      p = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM,
         "lastfm-artist.png"));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("lastfm-i-artist"))->set (p);

      p = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM,
         "lastfm-album.png"));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("lastfm-i-album"))->set (p);

      p = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM,
         "lastfm-track.png"));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("lastfm-i-track"))->set (p);

      p = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_LASTFM,
         "audioscrobbler.png"));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("image-audioscrobbler"))->set (p);

      Bmp::UI::BannerImage *banner = Gtk::manage (new Bmp::UI::BannerImage (left, right, slice));
      banner->set_size_request (-1, 60);

      dynamic_cast<Gtk::Alignment *>(m_ref_xml->get_widget ("alignment-lastfm-logo")) 
        ->add (*banner);
      dynamic_cast<Gtk::Alignment *>(m_ref_xml->get_widget ("alignment-lastfm-logo"))
        ->show_all ();

      m_ref_xml->get_widget ("b_lastfm_reconnect", b_lastfm_reconnect);

      mcs->subscribe ("Last.FM", "lastfm", "username",
          sigc::mem_fun (this, &LASTFM::mcs_lastfm_credentials_changed));
      mcs->subscribe ("Last.FM", "lastfm", "password",
          sigc::mem_fun (this, &LASTFM::mcs_lastfm_credentials_changed));

      mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>
          (m_ref_xml->get_widget ("last-fm-username")) , "lastfm", "username");
      mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>
          (m_ref_xml->get_widget ("last-fm-password")) , "lastfm", "password");

      mcs->subscribe ("Last.FM", "lastfm", "queue-enable",
              sigc::mem_fun (this, &LASTFM::mcs_lastfm_queue_enable_changed));
      mcs->subscribe ("Last.FM", "lastfm", "submit-enable",
              sigc::mem_fun (this, &LASTFM::mcs_lastfm_submit_enable_changed));

      mcs_bind->bind_toggle_button (dynamic_cast<Gtk::CheckButton *>
              (m_ref_xml->get_widget("last-fm-queue-enable")), "lastfm", "queue-enable");
      mcs_bind->bind_toggle_button (dynamic_cast<Gtk::CheckButton *>
              (m_ref_xml->get_widget("last-fm-submit-enable")), "lastfm", "submit-enable");

      lastfm_connection_state->set
        (Gtk::StockID (BMP_STOCK_APPROVE_PARTIAL), Gtk::ICON_SIZE_SMALL_TOOLBAR);

      const char *last_fm_m_network[] =
      {
        "lastfm-net-idle.png",
        "lastfm-net-rx.png",
        "lastfm-net-tx.png"
      };

      for (unsigned int n = 0; n < 3; n++)
        {
          m_network[n] = Gdk::Pixbuf::create_from_file
            (Glib::build_filename (BMP_IMAGE_DIR_LASTFM, last_fm_m_network[n]));
        }

      lastfm_scrobbler->signal_queue_size ().connect
          (sigc::mem_fun (this, &LASTFM::on_lastfm_queue_size));

      lastfm_scrobbler->signal_submit_start ().connect
          (sigc::mem_fun (this, &LASTFM::on_lastfm_submit_start));

      lastfm_scrobbler->signal_submit_end ().connect
          (sigc::mem_fun (this, &LASTFM::on_lastfm_submit_end));

      lastfm_scrobbler->signal_submit_error ().connect
          (sigc::mem_fun (this, &LASTFM::on_lastfm_submit_error));


      last_fm_netstatus->set (m_network[LAST_FM_IDLE]);

      ::play->signal_http_status().connect (sigc::mem_fun (this, &LASTFM::play_http_status));
      ::play->signal_lastfm_sync().connect (sigc::mem_fun (this, &LASTFM::play_lastfm_sync));

      b_lastfm_reconnect->signal_clicked().connect
          (sigc::mem_fun (this, &LASTFM::radio_connect));

      radio_connect ();

      m_ref_xml->get_widget ("last-fm-rtp", last_fm_rtp);
      mcs_bind->bind_toggle_button (last_fm_rtp, "lastfm", "record-to-profile");

      // We connect the checkbutton here as it needs the handshake
      // to be done  for the mcs syncback

      mcs->subscribe ("Last.FM", "lastfm", "record-to-profile",
          (sigc::mem_fun (this, &LASTFM::mcs_lastfm_record_to_profile_changed)));

      m_init = false;
    }

    /////////////// Queue Stuff 

    void
    LASTFM::on_lastfm_queue_size (unsigned int queue_size)
    {
      last_fm_queue_size->set_text ((boost::format ("%u") % queue_size).str ());
    }

    bool
    LASTFM::lastfm_set_idle_image ()
    {
      last_fm_netstatus->set (m_network[LAST_FM_IDLE]);
      return false;
    }

    void
    LASTFM::on_lastfm_submit_end ()
    {
      last_fm_netstatus->set (m_network[LAST_FM_RX]);
      Glib::signal_timeout().connect
        (sigc::mem_fun (this, &LASTFM::lastfm_set_idle_image), 300);
    }

    void
    LASTFM::on_lastfm_submit_start ()
    {
      last_fm_netstatus->set (m_network[LAST_FM_TX]);
    }

    void
    LASTFM::on_lastfm_submit_error (Glib::ustring message)
    {
      hbox_lastfm_submit_error->show_all();
      lastfm_queue_submit_error->set_markup ("<small>" + message + "</small>");
      mcs->key_set<bool>("lastfm", "submit-enable", false);
    }

    void
    LASTFM::mcs_lastfm_credentials_changed (MCS_CB_DEFAULT_SIGNATURE)
    {
      if (!m_init)
        {
          mcs->key_set<bool>("lastfm", "submit-enable", false);
          b_lastfm_reconnect->set_sensitive (true);
        }
    }

    void
    LASTFM::mcs_lastfm_submit_enable_changed (MCS_CB_DEFAULT_SIGNATURE)
    {
      bool active (boost::get<bool>(value));

      last_fm_enable->set_active (active);
      last_fm_netstatus->set_sensitive (active);

      if (active)
        {
          hbox_lastfm_submit_error->hide();
          lastfm_queue_submit_error->set_text ("");
        }
    }

    void
    LASTFM::mcs_lastfm_queue_enable_changed (MCS_CB_DEFAULT_SIGNATURE)
    {
      Gtk::Widget *widget = 0;
      bool         active;

      if (Network::check_connected ())
      {
        const char* widget_names[] =
        {
            "last-fm-submit-enable",
            "last-fm-netstatus",
        };

        active = mcs->key_get<bool>("lastfm", "queue-enable");
        for (unsigned int n = 0; n < G_N_ELEMENTS(widget_names); n++)
        {
            widget = m_ref_xml->get_widget (widget_names[n]);
            widget->set_sensitive (active);
        }

        last_fm_netstatus->set (m_network[LAST_FM_IDLE]);
        lastfm_scrobbler->emit_queue_size ();
      }
      else
      {
        const char* widget_names[] =
          {
            "last-fm-netstatus"
          };

        active = mcs->key_get<bool>("lastfm", "queue-enable");
        for (unsigned int n = 0; n < G_N_ELEMENTS(widget_names); n++)
          {
            widget = m_ref_xml->get_widget (widget_names[n]);
            widget->set_sensitive (active);
          }

        last_fm_netstatus->set (m_network[LAST_FM_IDLE]);
        lastfm_scrobbler->emit_queue_size ();
      }
    }

#define CATCH_DEFAULT_ERRORS \
        catch (LastFM::TimeOutError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "TimeOutError: %s", cxe.what());                \
            set_error ((boost::format (_("Timeout: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::ConnectionError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ConnectionError: %s", cxe.what());             \
            set_error ((boost::format (_("Connection Error: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::ResourceNotFoundError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ResourceNotFoundError: %s", cxe.what());         \
            set_error ((boost::format (_("Resource Not Found: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::AuthenticationError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "AuthenticationError: %s", cxe.what());         \
            set_error ((boost::format (_("Authentication Error: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::ResponseError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ResponseError: %s", cxe.what());         \
            set_error ((boost::format (_("Invalid HTTP Response: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::RequestError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ResponseError: %s", cxe.what());         \
            set_error ((boost::format (_("Response Error: %s")) % cxe.what()).str());                         \
            return; \
          } \
        catch (LastFM::LastFMNotConnectedError& cxe)  \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ConnectionError: %s", cxe.what()); \
            set_error ((boost::format (_("Not Handshaked/Session Expired: %s")) % cxe.what()).str()); \
            return; \
          } \
        catch (LastFM::LastFMBadauthError& cxe)  \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ConnectionError: %s", cxe.what()); \
            set_error ((boost::format (_("Couldn't create Session: %s")) % cxe.what()).str()); \
            return; \
          } \
        catch (LastFM::NetworkError& cxe) \
          { \
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "NetworkError: %s", cxe.what());                \
            set_error ((boost::format (_("Unknown Network Error: %s")) % cxe.what()).str());                         \
            return; \
          }

    void
    LASTFM::radio_connect ()
    {
        try {
          lastfm_radio->connect ();
          }

        CATCH_DEFAULT_ERRORS

        clear_error ();

        if (lastfm_radio->connected())
          {
            lastfm_radio->session (m_session);
            m_treeview_stations->set_user
                (mcs->key_get<std::string>("lastfm","username"), m_session.subscriber);
            m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
            s_caps_.emit (m_caps);
          }
    } 

    void
    LASTFM::clear_error ()
    {
      lastfm_connection_state->set
        (Gtk::StockID (BMP_STOCK_OK), Gtk::ICON_SIZE_SMALL_TOOLBAR);
      set_connection_message (_("OK!"));
      b_lastfm_reconnect->set_sensitive (false);
      e_station->set_sensitive (true);
    }

    void
    LASTFM::set_error (Glib::ustring const& message, bool disable_session)
    {
      lastfm_connection_state->set
        (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
      set_connection_message (message);
      if (disable_session)
      {
        b_lastfm_reconnect->set_sensitive (true);
        e_station->set_sensitive (false);
      }
    }

    Glib::ustring
    LASTFM::get_uri ()
    {
      return m_session.streamurl; 
    }

    bool
    LASTFM::go_next ()
    {
      skip ();
      return false;
    }

    bool
    LASTFM::go_prev ()
    {
      return false;
    }

    void
    LASTFM::stop ()
    {
      ::play->property_lastfm_mode() = false;

      m_artist.reset ();
      m_album.reset ();
      m_track.reset ();

      b_love->set_sensitive (false);
      b_skip->set_sensitive (false);
      b_ban->set_sensitive (false);

      lastfm_radio_artist->set_text ("");
      lastfm_radio_album->set_text  ("");
      lastfm_radio_track->set_text  ("");
      lastfm_radio_station->set_text ("");

      lastfm_coverimage->clear ();

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      s_caps_.emit (m_caps);

      m_metadata.streaming = false;
    }

    void
    LASTFM::play ()
    {
      ::play->property_lastfm_mode() = true;
      s_caps_.emit (m_caps);
    }

    void
    LASTFM::restore_context ()
    {
    }

    GHashTable*
    LASTFM::get_metadata ()
    {
      GHashTable *table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
      GValue     *value = 0;

      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.artist.c_str()); 
      g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_ARTIST).c_str()), value);

      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.track.c_str()); 
      g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_TITLE).c_str()), value);

      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.album.c_str()); 
      g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_ALBUM).c_str()), value);

      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, (m_metadata.station + " (Last.fm)").c_str());  
      g_hash_table_insert (table, g_strdup(MetadatumId(Library::DATUM_GENRE).c_str()), value);

      return table;
    }

    bool
    LASTFM::set_metadata_timeout_func ()
    {
      lastfm_radio->metadata (m_metadata);

      if (mcs->key_get<bool>("lastfm","record-to-profile") != m_metadata.rtp) 
      {
        lastfm_radio->command
          (mcs->key_get<bool>("lastfm","record-to-profile") ? LastFM::Radio::RTP   
                                                            : LastFM::Radio::NORTP);
      }
    
      send_title_set_metadata ();

      if (m_metadata.streaming)
        {
          m_caps = Caps (m_caps | PlaybackSource::CAN_GO_NEXT);
          s_caps_.emit (m_caps);
        }
      else
        {
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
          s_caps_.emit (m_caps);
          s_stop_request_.emit();
        }

      return false;
    }

    void
    LASTFM::play_lastfm_sync ()
    {
      Glib::signal_idle().connect
        (sigc::mem_fun (this, &Bmp::UiPart::LASTFM::set_metadata_timeout_func));
    }

    void
    LASTFM::set_connection_message (Glib::ustring const& message)
    {
      lastfm_connection_message->set_text (message);
    }

    void
    LASTFM::play_http_status (int status)
    {
      switch (status)
      {
        case 200: break;
        case 401: 
          {
            lastfm_connection_state->set
              (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
            set_connection_message
              (_("Session expired. Please reconnect."));
            stop ();
            ::play->request_status (PLAYSTATUS_STOPPED);
            b_lastfm_reconnect->set_sensitive (true);
            break;
          }

        case 503:
          {
            lastfm_connection_state->set
              (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
            set_connection_message
              (_("The radio server is too busy at the moment, please try "
                "again in a few minutes."));
            stop ();
            ::play->request_status (PLAYSTATUS_STOPPED);
            b_lastfm_reconnect->set_sensitive (true);
            break;
          }
  
        case 666:
          {
            lastfm_connection_state->set
              (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
            set_connection_message
              (_("Radio server is down for maintenance, please try again in a few minutes."));
            stop ();
            ::play->request_status (PLAYSTATUS_STOPPED);
            b_lastfm_reconnect->set_sensitive (true);
            break;
          }

        case 667: 
          {
            lastfm_connection_state->set
              (Gtk::StockID (BMP_STOCK_ERROR), Gtk::ICON_SIZE_SMALL_TOOLBAR);
            set_connection_message
              (_("Not enough content (left) to play this station; please tune "
                "in to a different one."));
            stop ();
            ::play->request_status (PLAYSTATUS_STOPPED);
            break;
          }
      }
    }

    void
    LASTFM::tune ()
    {
      try {
        m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
        std::string station = e_station->get_text().c_str();
        lastfm_radio->tune (station);
        clear_error();
        if (!m_metadata.streaming)
          {
              s_stop_request_.emit();
              if (!lastfm_radio->connected())
                {
                  radio_connect ();
                }
              play ();
              s_playback_request_.emit ();
          }
        }
      catch (LastFM::LastFMStreamTuningError& cxe)
        {
          g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
            "StreamTuningError: %s", cxe.what());
          set_error (_("Couldn't tune into station"), false);
          ::play->request_status (PLAYSTATUS_STOPPED);
        }

      CATCH_DEFAULT_ERRORS

    }

    void
    LASTFM::mcs_lastfm_record_to_profile_changed (MCS_CB_DEFAULT_SIGNATURE)
    {
      bool active = boost::get<bool>(value); 

      try {
        lastfm_radio->command (active ? LastFM::Radio::RTP : LastFM::Radio::NORTP);
        clear_error ();
       } 

      CATCH_DEFAULT_ERRORS
    }

    void
    LASTFM::love ()
    {
      try {
          b_love->set_sensitive (false);
          lastfm_radio->command (Bmp::LastFM::Radio::LOVE);
          clear_error ();
        }

      CATCH_DEFAULT_ERRORS

    }

    void
    LASTFM::skip ()
    {
      try {
          b_skip->set_sensitive (false);
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
          s_caps_.emit (m_caps);
          lastfm_radio->command (Bmp::LastFM::Radio::SKIP);
          clear_error ();
        }

      CATCH_DEFAULT_ERRORS

    }

    void
    LASTFM::ban ()
    {
      try {
          b_ban->set_sensitive (false);
          lastfm_radio->command (Bmp::LastFM::Radio::BAN);
          clear_error ();
        }

      CATCH_DEFAULT_ERRORS

    }

    ////////////////////////////////////////////  

    void
    LASTFM::send_title_set_metadata ()
    {
      using namespace Glib;    

      if (!m_metadata.streaming)
        return;

      if ((m_artist && m_album && m_track) &&
            ((m_artist.get() == m_metadata.artist) &&
            (m_album.get() == m_metadata.album) &&
            (m_track.get() == m_metadata.track))) return;

      b_love->set_sensitive (true);
      b_skip->set_sensitive (true);
      b_ban->set_sensitive (true);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
      m_caps = Caps (m_caps | PlaybackSource::CAN_GO_NEXT);
      s_caps_.emit (m_caps);

      lastfm_radio_artist->set_text (m_metadata.artist);
      lastfm_radio_album->set_text  (m_metadata.album);
      lastfm_radio_track->set_text  (m_metadata.track);
      lastfm_radio_station->set_text (m_metadata.station);

      m_artist  = m_metadata.artist;
      m_album   = m_metadata.album;
      m_track   = m_metadata.track;

      Bmp::VFS::Handle handle (m_metadata.albumcover_large);
      vfs->read (handle, VFS::TRANSPORT);
      Glib::RefPtr<Gdk::PixbufLoader> loader = Gdk::PixbufLoader::create(); 
      loader->write (reinterpret_cast<const guint8*>( handle.get_buffer()),
                                                      handle.get_buffer_size());
      loader->close ();
      Glib::RefPtr<Gdk::Pixbuf> p = loader->get_pixbuf();

      if (p)
          lastfm_coverimage->set (p->scale_simple ( 160, 160, Gdk::INTERP_BILINEAR ));
      else
          lastfm_coverimage->clear ();

      Bmp::SimpleTrackInfo sti;

      sti.artist    = m_metadata.artist;
      sti.album     = m_metadata.album;
      sti.title     = m_metadata.track;
      sti.duration  = m_metadata.duration;

      if (p)
        sti.image = p->scale_simple ( 128, 128, Gdk::INTERP_BILINEAR );
      else
        sti.image = m_audio_empty->scale_simple ( 128, 128, Gdk::INTERP_BILINEAR );


      ustring text ("<b>");
      text.append(Glib::Markup::escape_text(m_metadata.track)).append("</b> (");
      text.append(Glib::Markup::escape_text(m_metadata.artist)).append(": <i>");
      text.append(Glib::Markup::escape_text(m_metadata.album)).append("</i>)");

      s_track_info_.emit (text, sti); 

      loader.clear();
    }

    void
    LASTFM::set_uri (Glib::ustring const& uri)
    {
      using namespace Glib;

      URI u (uri);
      u.unescape();
      e_station->set_text (ustring(u));
      tune ();
    }

    LASTFM::TreeViewStations::TreeViewStations (BaseObjectType                       * obj,
                                                Glib::RefPtr<Gnome::Glade::Xml> const& xml)
      : Gtk::TreeView (obj)
    {
      using namespace Glib;
      using namespace Gdk;

      m_pb_station = 
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-station-small.png"));

      m_pb_neighbour =
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-neighbour.png"));

      m_pb_friend =
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-friend.png"));

      m_pb_user =
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-user.png"));

      m_pb_mainuser =
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-mainuser.png"));

      m_pb_tag =
          Pixbuf::create_from_file
            (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-tag.png"));

      m_store = Gtk::TreeStore::create (columns);

      Gtk::TreeViewColumn *column = Gtk::manage (new Gtk::TreeViewColumn ());
      column->set_resizable (false);
      column->set_expand (false);

      Gtk::CellRendererPixbuf *cell1 = Gtk::manage (new Gtk::CellRendererPixbuf ());
      cell1->property_ypad () = 2;
      cell1->property_xpad () = 2;
      column->pack_start (*cell1, false);
      column->set_cell_data_func (*cell1,
        sigc::mem_fun (this, &Bmp::UiPart::LASTFM::TreeViewStations::cell_data_func));

      Gtk::CellRendererText *cell2 = Gtk::manage (new Gtk::CellRendererText ());
      cell2->property_ypad () = 2;
      cell2->property_xpad () = 2;
      column->pack_start (*cell2, false);
      column->add_attribute (*cell2, "markup", 0);

      append_column (*column);

      get_selection()->set_mode (Gtk::SELECTION_SINGLE);  
      set_model (m_store);
    }

    LASTFM::TreeViewStations::~TreeViewStations () {}

    void
    LASTFM::station_list_update ()
    {
      m_treeview_stations->set_user
          (mcs->key_get <std::string> ("lastfm","username"), m_session.subscriber);
    }

    void
    LASTFM::TreeViewStations::set_user (Glib::ustring const& username, bool subscriber)
    {
      using boost::algorithm::split;
      using boost::algorithm::is_any_of;
      using namespace std;

      static boost::format f_neighbours ("lastfm://user/%s/neighbours");      
      static boost::format f_personal   ("lastfm://user/%s/personal");      
      static boost::format f_loved      ("lastfm://user/%s/loved");      
      static boost::format f_recommend  ("lastfm://user/%s/recommended/%i");
      static boost::format f_globaltag  ("lastfm://globaltags/%s");      

      static boost::format t_personal  (_("%s's personal radio"));      
      static boost::format t_loved     (_("%s's loved tracks radio"));      
      static boost::format t_neighbour (_("%s's neighbour radio"));      
      static boost::format t_recommend (_("%s's recommended radio"));      

      m_store->clear ();
      set_sensitive (false);

      while (gtk_events_pending())
        gtk_main_iteration ();

      Gtk::TreeModel::iterator m_iter_parent_stations; 
      Gtk::TreeModel::iterator m_iter_parent_neighbours; 
      Gtk::TreeModel::iterator m_iter_parent_friends; 
      Gtk::TreeModel::iterator m_iter_parent_favtags; 
      Gtk::TreeModel::iterator m_iter_parent_globaltags; 
  
      m_iter_parent_stations = m_store->append ();
      (*m_iter_parent_stations)[columns.name] = _("My Personal Stations");
      (*m_iter_parent_stations)[columns.type] = TreeViewStations::ROW_MAINUSER; 

      m_iter_parent_neighbours = m_store->append ();
      (*m_iter_parent_neighbours)[columns.name] = _("My Neighbours");
      (*m_iter_parent_neighbours)[columns.type] = TreeViewStations::ROW_NEIGHBOUR_ROOT; 

      m_iter_parent_friends = m_store->append ();
      (*m_iter_parent_friends)[columns.name] = _("My Friends");
      (*m_iter_parent_friends)[columns.type] = TreeViewStations::ROW_FRIEND_ROOT; 

      m_iter_parent_favtags = m_store->append ();
      (*m_iter_parent_favtags)[columns.name] = _("My Favorite Tags");
      (*m_iter_parent_favtags)[columns.type] = TreeViewStations::ROW_TAG_ROOT; 

      m_iter_parent_globaltags = m_store->append ();
      (*m_iter_parent_globaltags)[columns.name] = _("Top Global Tags");
      (*m_iter_parent_globaltags)[columns.type] = TreeViewStations::ROW_TAG_ROOT; 

      Gtk::TreeModel::iterator m_iter; 

      m_iter = m_store->append (m_iter_parent_stations->children());
      (*m_iter)[columns.name] = Glib::Markup::escape_text ((t_neighbour % username).str()); 
      (*m_iter)[columns.station]  = (f_neighbours % username.c_str()).str();
      (*m_iter)[columns.type]  = TreeViewStations::ROW_STATION; 

      m_iter = m_store->append (m_iter_parent_stations->children());
      (*m_iter)[columns.name] = Glib::Markup::escape_text ((t_recommend % username).str()); 
      (*m_iter)[columns.station]  = (f_recommend % username.c_str() % 100).str();
      (*m_iter)[columns.type]  = TreeViewStations::ROW_STATION; 

      if (subscriber)
        {
          m_iter = m_store->append (m_iter_parent_stations->children());
          (*m_iter)[columns.name] = Glib::Markup::escape_text ((t_personal % username).str()); 
          (*m_iter)[columns.station]  = (f_personal % username.c_str()).str();
          (*m_iter)[columns.type]  = TreeViewStations::ROW_STATION; 

          m_iter = m_store->append (m_iter_parent_stations->children());
          (*m_iter)[columns.name] = Glib::Markup::escape_text ((t_loved % username).str()); 
          (*m_iter)[columns.station]  = (f_loved % username.c_str()).str();
          (*m_iter)[columns.type]  = TreeViewStations::ROW_STATION; 
        }

      // Append Neighbours 
      try {
          LastFM::Friends neighbours;
          LastFM::get_neighbours (username, neighbours); 
          for (LastFM::Friends::const_iterator i  = neighbours.begin(); 
                                               i != neighbours.end(); ++i)
          {
            LastFM::Friend const& f (*i);
            m_iter = m_store->append (m_iter_parent_neighbours->children());
            (*m_iter)[columns.name] = Glib::Markup::escape_text (f.username);
            (*m_iter)[columns.type]  = TreeViewStations::ROW_NEIGHBOUR; 

            Gtk::TreeModel::iterator n, p, l, r;
            n = m_store->append (m_iter->children());
            r = m_store->append (m_iter->children());
            p = m_store->append (m_iter->children());
            l = m_store->append (m_iter->children());

            (*n)[columns.name] =
              Glib::Markup::escape_text ((t_neighbour % f.username.c_str()).str());
            (*n)[columns.station]  = (f_neighbours % f.username.c_str()).str();
            (*n)[columns.type]  = TreeViewStations::ROW_STATION; 

            (*r)[columns.name] =
              Glib::Markup::escape_text ((t_recommend % f.username.c_str()).str());
            (*r)[columns.station]  = (f_recommend % f.username.c_str() % 100).str();
            (*r)[columns.type]  = TreeViewStations::ROW_STATION; 

            (*p)[columns.name] =
              Glib::Markup::escape_text ((t_personal % f.username.c_str()).str());
            (*p)[columns.station]  = (f_personal % f.username.c_str()).str();
            (*p)[columns.type]  = TreeViewStations::ROW_STATION; 

            (*l)[columns.name] =
              Glib::Markup::escape_text ((t_loved % f.username.c_str()).str());
            (*l)[columns.station]  = (f_loved % f.username.c_str()).str();
            (*l)[columns.type]  = TreeViewStations::ROW_STATION; 

            while (gtk_events_pending())
              gtk_main_iteration ();

          }
        }
      catch  (...) {}

      // Append Friends 
      try {
          LastFM::Friends friends;
          LastFM::get_friends (username, friends); 
          for (LastFM::Friends::const_iterator i  = friends.begin(); 
                                               i != friends.end(); ++i)
          {
            LastFM::Friend const& f (*i);
            m_iter = m_store->append (m_iter_parent_friends->children());
            (*m_iter)[columns.name] = Glib::Markup::escape_text (f.username);
            (*m_iter)[columns.type]  = TreeViewStations::ROW_FRIEND; 

            Gtk::TreeModel::iterator n, p, l;
            n = m_store->append (m_iter->children());
            p = m_store->append (m_iter->children());
            l = m_store->append (m_iter->children());

            (*n)[columns.name] =
              Glib::Markup::escape_text ((t_neighbour % f.username.c_str()).str());
            (*n)[columns.station]  = (f_neighbours % f.username.c_str()).str();
            (*n)[columns.type]  = TreeViewStations::ROW_STATION; 

            (*p)[columns.name] =
              Glib::Markup::escape_text ((t_personal % f.username.c_str()).str());
            (*p)[columns.station]  = (f_personal % f.username.c_str()).str();
            (*p)[columns.type]  = TreeViewStations::ROW_STATION; 

            (*l)[columns.name] =
              Glib::Markup::escape_text ((t_loved % f.username.c_str()).str());
            (*l)[columns.station]  = (f_loved % f.username.c_str()).str();
            (*l)[columns.type]  = TreeViewStations::ROW_STATION; 

            while (gtk_events_pending())
              gtk_main_iteration ();

          }
        }
      catch  (...) {}

      // Append Favorite Tag Stations
      try {
          LastFM::Tags tags;
          LastFM::get_tags (LastFM::TAGS_TOPTAGS, LastFM::TAG_ORIGIN_USER, tags, username); 
          for (LastFM::Tags::const_iterator t = tags.begin(), e = tags.end(); t != e; ++t)
          {
            LastFM::Tag const& tag (*t);
            m_iter = m_store->append (m_iter_parent_favtags->children());
            (*m_iter)[columns.name] = Glib::Markup::escape_text (tag.name);
            (*m_iter)[columns.station] = (f_globaltag % tag.name).str();
            (*m_iter)[columns.type]  = TreeViewStations::ROW_TAG; 

            while (gtk_events_pending())
              gtk_main_iteration ();

          }
        }
      catch  (...) {}

      // Append Global Tag Stations
      try {
          LastFM::Tags tags;
          LastFM::get_tags (LastFM::TAGS_TOPTAGS, LastFM::TAG_ORIGIN_GLOBAL, tags); 
          for (LastFM::Tags::const_iterator t = tags.begin(), e = tags.end(); t != e; ++t)
          {
            LastFM::Tag const& tag (*t);
            m_iter = m_store->append (m_iter_parent_globaltags->children());
            (*m_iter)[columns.name] = Glib::Markup::escape_text (tag.name);
            (*m_iter)[columns.station] = (f_globaltag % tag.name).str();
            (*m_iter)[columns.type]  = TreeViewStations::ROW_TAG; 

            while (gtk_events_pending())
              gtk_main_iteration ();

          }
        }
      catch  (...) {}

      set_sensitive (true);
    }

    void
    LASTFM::TreeViewStations::cell_data_func
      (Gtk::CellRenderer * _cell, Gtk::TreeModel::iterator const& m_iter)
    {
      Gtk::CellRendererPixbuf * cell = dynamic_cast<Gtk::CellRendererPixbuf *>(_cell); 
      switch ((*m_iter)[columns.type])
        {
          case ROW_MAINUSER:
              cell->property_pixbuf() = m_pb_mainuser;
              return;

          case ROW_STATION:
              cell->property_pixbuf() = m_pb_station;
              return;

          case ROW_TAG_ROOT:
              cell->property_pixbuf() = m_pb_tag;
              return;
          case ROW_TAG:
              cell->property_pixbuf() = m_pb_station;
              return;

          case ROW_NEIGHBOUR_ROOT:
              cell->property_pixbuf() = m_pb_neighbour;
              return;
          case ROW_NEIGHBOUR:
              cell->property_pixbuf() = m_pb_user;
              return;

          case ROW_FRIEND_ROOT:
              cell->property_pixbuf() = m_pb_friend;
              return;
          case ROW_FRIEND:
              cell->property_pixbuf() = m_pb_user;
              return;

          default: return;
        }
    }

    void
    LASTFM::TreeViewStations::on_row_activated
      (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn * column)
    {
      using namespace Gtk;

      TreeModel::iterator m_iter = m_store->get_iter (path);
      RowType type ((*m_iter)[columns.type]);
      if ((type == ROW_STATION) || (type == ROW_TAG))
        {
          station_activated_.emit ((*m_iter)[columns.station]); 
        }
    }

  } // namespace UiPart
} // namespace Bmp 
