//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_MAIN_UI_HH 
#define BMP_MAIN_UI_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtkmm.h>
#include <libglademm.h>
#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>
#include <mcs/mcs.h>

#include "playbacksource.hh"
#include "popup.hh"
#include "vfs.hh"

#ifdef HAVE_GAIM
#  include "gaim-dbus.hh"
#endif //HAVE_GAIM

#include <widgets/bmp_tray_icon.h>
#include <widgets/bmp_status_icon.h>

#include "dialog-about.hh"
#include "preferences-ui.hh"
#include "button.hh"

#ifdef HAVE_VISUALIZATIONS
#  include "video-widget.hh"
#endif //HAVE_VISUALIZATIONS

namespace Bmp
{
  class Scroll;
  class PlayerShell : public Gtk::Window
  {
    protected:

      virtual bool
      on_key_press_event (GdkEventKey*);

      virtual bool
      on_window_state_event (GdkEventWindowState *event);

    private:

#ifdef HAVE_VISUALIZATIONS
      VideoWidget *vis;
      void xoverlay_setup (GstElement *element);
      void toggle_vis ();
      Gtk::Notebook *notebook_over;
#endif //HAVE_VISUALIZATIONS

      Glib::RefPtr<Gnome::Glade::Xml> m_ref_xml;
      Gtk::Notebook *notebook_main;

      Glib::RefPtr<Gtk::UIManager>   m_ui_manager;
      Glib::RefPtr<Gtk::ActionGroup> m_actions_main;

      typedef sigc::signal<void, int> SignalSourceSelected;
    
      class SourceSelection : public Gtk::VBox
      {
        public:

      	  SourceSelection (BaseObjectType                       *cobject,
                       		 const Glib::RefPtr<Gnome::Glade::Xml> &xml);
          ~SourceSelection ();

          void select_source (PlaybackSourceID source);
          void set_active_source (PlaybackSourceID source);

          PlaybackSourceID get_active_source ();
         
          SignalSourceSelected& 
          source_selected () { return source_selected_; }

        protected:
  
          virtual bool on_expose_event (GdkEventExpose *event);

        private:

          void source_activated (PlaybackSourceID source);

          SignalSourceSelected      source_selected_;
          PlaybackSourceID          m_active_source;
          PlaybackSourceID          m_selected_source;

          typedef std::map<PlaybackSourceID, Bmp::SourceButton *> Buttons;
          Buttons b_map;

          typedef std::map<PlaybackSourceID, unsigned int> MSourceNotebookTab;
          MSourceNotebookTab st_map;
      };

      typedef std::map <PlaybackSourceID, Bmp::PlaybackSource::Caps> SourceCaps;
      typedef std::map <PlaybackSourceID, Bmp::PlaybackSource::Flags> SourceFlags;
      typedef std::map <PlaybackSourceID, boost::shared_ptr<Bmp::PlaybackSource> > Sources;

      boost::optional<SimpleTrackInfo>    m_sti;

      SourceSelection                   * m_source_selection;
      PlaybackSourceID                    m_active_source;

      SourceCaps                          m_source_caps;
      SourceFlags                         m_source_flags;

      Sources                             m_sources;

      void display_about_dialog ();

      // Playback Controls
      void playstatus_changed ();
      void play ();
      void pause ();
      void prev ();
      void next ();
      void stop ();

      void play_position (int position);
      void play_seek     (int position);
 
      void seek_display ();
      bool seek_event (GdkEvent *event);
    
      void buffering (double buffered);
      void play_set_title (const char *title);

 

      // Source Handling
      bool source_restore_context (GdkEventButton *event);
      void source_caps (Bmp::PlaybackSource::Caps caps, PlaybackSourceID source);
      void source_flags (Bmp::PlaybackSource::Flags flags, PlaybackSourceID source);
      void source_track_info (Glib::ustring title, SimpleTrackInfo sti, PlaybackSourceID source);
      void source_playback_request (PlaybackSourceID source);
      void source_changed (int n);

      void connect_source_signals (PlaybackSourceID source);

      void error_set    (Glib::ustring message, PlaybackSourceID source);
      bool error_clear  (GdkEventButton*);
  
      Glib::RefPtr<Gdk::Pixbuf>           yy_stopped;
      Glib::RefPtr<Gdk::Pixbuf>           yy_seeking;
      Glib::RefPtr<Gdk::PixbufAnimation>  yy_playing;
      Glib::RefPtr<Gdk::PixbufAnimation>  yy_paused;

      Glib::RefPtr<Gdk::Pixbuf>   m_warning;
      Glib::RefPtr<Gdk::Pixbuf>   m_stop_image;

      Gtk::Image    *status_image;
      Gtk::Label    *status_label;
      Gtk::Label    *time_label;
      Gtk::HScale   *seekrange;
      Gtk::EventBox *evb;
      Gtk::EventBox *status_evbox;

      Bmp::Scroll   *volume;

      Bmp::Popup    *m_popup;
      BmpStatusIcon *m_status_icon;

      Glib::RefPtr<Gdk::Pixbuf>      m_status_icon_image_default;
      Glib::RefPtr<Gdk::Pixbuf>      m_status_icon_image_paused;
      Glib::RefPtr<Gdk::Pixbuf>      m_status_icon_image_playing;

      static void create_popup  (GtkPlug *plug, gpointer userdata);
      static void status_icon_click (BmpStatusIcon *icon, gpointer data);
      static void status_icon_scroll_up (BmpStatusIcon *icon, gpointer data);
      static void status_icon_scroll_down (BmpStatusIcon *icon, gpointer data);
      static void status_icon_popup_menu (BmpStatusIcon *icon, guint arg1, guint arg2, gpointer data);
      static gboolean status_icon_enter (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data); 
      static gboolean status_icon_leave (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data); 

      sigc::connection delay_popup;
      bool display_popup_delayed ();

      void display_popup (SimpleTrackInfo& info);

      void recommend ();
      void lyrics ();

#ifdef HAVE_GAIM

      void gaim_send ();
      void gaim_present ();
      void gaim_lost ();
      boost::shared_ptr<Bmp::Gaim::DBUS> gaim_dbus;

#endif //HAVE_GAIM

      bool m_visible;

      bool m_seeking;
      int  m_seek_request;

      enum SeekDirection
      {
        M_SEEK_FORWARD,
        M_SEEK_BACKWARD
      };

      SeekDirection m_seek_direction;

      bool m_sent;
      bool m_error_condition;
      bool m_monitor;

      // Additional UIs
      boost::shared_ptr<Bmp::Preferences> m_preferences;
      boost::shared_ptr<Bmp::AboutDialog> m_about_dialog;

      void import_request (Bmp::VUri& list);
      void enqueue_request (Bmp::VUri& list);

      Gtk::UIManager::ui_merge_id m_ui_merge_id;

    public:

#include "exception.hh"

      EXCEPTION(InvalidSourceError)
      EXCEPTION(NoMetadataError)

      PlayerShell (BaseObjectType                       *cobject,
              const Glib::RefPtr<Gnome::Glade::Xml> &xml);
      ~PlayerShell ();
      static PlayerShell* create ();

      // Inquiries
      PlaybackSource::Caps  get_source_caps (PlaybackSourceID source);
      PlaybackSource::Flags get_source_flags (PlaybackSourceID source);

      PlaybackSourceID      get_current_source ();
      GHashTable*           get_metadata_from_source (PlaybackSourceID source);

      // Actions
      void play_uris (VUri& list, bool playback);
      void play_lastfm_uri (Glib::ustring const& lastfm_uri);
      void raise_ui ();

      // Playback Controls
      void play_ext ();
      void pause_ext ();
      void prev_ext ();
      void next_ext ();
      void stop_ext ();
  };
}
#endif
