/*
*
*  BlueZ - Bluetooth protocol stack for Linux
*
*  Copyright (C) 2004-2005  Marcel Holtmann <marcel@holtmann.org>
*
*
*  This library is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Lesser General Public
*  License as published by the Free Software Foundation; either
*  version 2.1 of the License, or (at your option) any later version.
*
*  This library is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Lesser General Public License for more details.
*
*  You should have received a copy of the GNU Lesser General Public
*  License along with this library; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define FASTTIMEOUTS 1

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/select.h>
#include <sys/poll.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/l2cap.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include <netinet/in.h>

#include "a2dpd_output_a2dp.h"
#include "a2dpd_tools.h"
#include "a2dpd_protocol.h"
#include "a2dpd_sdp.h"
#include "a2dpd_timer.h"
#include "a2dpd_ipc.h"
#include "a2dpd_dbus.h"
#include "sbc.h"
#include "../a2dp.h"

#define A2DPD_FLAGS_NONSPECAUDIO    1
#define NBSDPRETRIESMAX             0
#define BUFS                        2048
#define MAXMESSAGETIME              3 /* seconds */

//FIXME how should I choose the int_seid??
// The SEID used is the one that is advertised
#define A2DPD_SEID                  1

enum { A2DPD_NOROLE, A2DPD_INITIATOR, A2DPD_ACCEPTOR };

typedef enum {
	AVDTP_STATE_DISCONNECTED,
	AVDTP_STATEX_DISCONNECTING,
	AVDTP_STATEX_SDP_CONNECTING,
	AVDTP_STATEX_SDP_CONNECTING_WAIT,
	AVDTP_STATEX_CTL_CONNECTING,
	AVDTP_STATEX_CTL_CONNECTING_WAIT,
	AVDTP_STATEX_IDLE_CONNECTING,
	AVDTP_STATE_IDLE,
	AVDTP_STATEX_DISCOVERING,
	AVDTP_STATEX_DISCOVERING_RESP,
	AVDTP_STATEX_GETTING_CAPABILITIES,
	AVDTP_STATEX_GETTING_CAPABILITIES_RESP,
	AVDTP_STATEX_SETTING_CONFIGURATION,
	AVDTP_STATEX_SETTING_CONFIGURATION_RESP,
	AVDTP_STATEX_STREAM_OPENING,
	AVDTP_STATEX_STREAM_OPENING_RESP,
	AVDTP_STATE_CONFIGURED,
	AVDTP_STATEX_STREAM_CONNECTING,
	AVDTP_STATEX_STREAM_CONNECTING_WAIT,
	AVDTP_STATE_OPEN_START_STREAMING,
	AVDTP_STATE_OPEN,
	AVDTP_STATEX_STREAM_STARTING,
	AVDTP_STATEX_STREAM_STARTING_WAIT,
	AVDTP_STATEX_STREAM_SUSPENDING,
	AVDTP_STATEX_STREAM_SUSPENDING_WAIT,
	AVDTP_STATEX_STREAM_CLOSING,
	AVDTP_STATEX_STREAM_CLOSING_WAIT,
	AVDTP_STATE_STREAMING,
	AVDTP_STATE_CLOSING,
	AVDTP_STATE_ABORTING,
} AVDTP_STATE;

typedef struct {
	int nb_seid;
	int curr_seid;
	int curr_seid_acp_seid;
	struct sepd_resp resp;
	struct getcap_resp cap_resp;
} discover_state_info;

typedef struct {
	time_t message_time;
	discover_state_info discover;
} STATE_INFO;

typedef struct snd_pcm_a2dp {
	A2DPSETTINGS settings;
	bdaddr_t src;
	bdaddr_t dst;
	char bdaddr[20];
	sdp_session_t *sdp_session;
	int role;
	int sk_sdp;
	int sk;
	int control_sk;
	sbc_t sbc;
	int flags;
	int user_flags;
	unsigned char buf[1024];	// contain sbc encoded data, incrementally filled
	unsigned int len;		// number of valid bytes in buf
	unsigned int frame_bytes;	// fixed when initializing

	AVDTP_STATE state;		// State of a2dp
	STATE_INFO state_info;
	char bufe[BUFS];		// temporary encoding buffer
	int lenbufe;			//=0; Size of encoded data
	int samples;            // Number of encoded samples
	int samplescumul;            // Number of encoded samples

	time_t timestamp;		//=0;
	uint16_t seq_num;		//=1;
	int frame_count;		//=0; // Number of sbc frames in one AVDTP packet

	int mtu;			//=A2DPMAXIMUMTRANSFERUNITSIZE
	uint8_t seid;
	unsigned short psm_stream;
	unsigned short psm_cmd;

	// Bluetooth bandwith used
	int bandwithcount;
	struct timeval bandwithtimestamp;
	struct timeval samplescumultimestamp;

	// Used to control stream from headset
	int pause_writing;	// = 0;
} snd_pcm_a2dp_t;

// In fact sbc blocks are 76 bytes long, so a group of them is either 608 or 684 bytes
// So 650 or 678 makes no differences!
// However some devices may have longer transfer unit up to I saw omtu=733?

#define a2dp_set_state(new_state) do { DBG("State %s", #new_state); a2dp->state = new_state; } while (0)
#define a2dp_filter_state(new_state) do { DBG("State %s", #new_state); a2dp->state = new_state; } while (0)

#define FREQUENCY_16000     (1<<3)
#define FREQUENCY_32000     (1<<2)
#define FREQUENCY_44100     (1<<1)
#define FREQUENCY_48000     (1<<0)
unsigned int frequency_table [] = { 16000, 32000, 44100, 48000 };

#define BLOCK_LENGTH_4      (1<<3)
#define BLOCK_LENGTH_8      (1<<2)
#define BLOCK_LENGTH_12     (1<<1)
#define BLOCK_LENGTH_16     (1<<0)
unsigned int block_length_table [] = { 4, 8, 12, 16 };

#define CHANNEL_MODE_MONO   (1<<3)
#define CHANNEL_MODE_DUAL   (1<<2)
#define CHANNEL_MODE_STEREO (1<<1)
#define CHANNEL_MODE_JOINT  (1<<0)
enum { MONO, DUAL_CHANNEL, STEREO, JOINT_STEREO };
unsigned int channel_mode_table [] = { MONO, DUAL_CHANNEL, STEREO, JOINT_STEREO };

#define ALLOCATION_SNR      (1<<1)
#define ALLOCATION_LOUDNESS (1<<0)
enum { LOUDNESS, SNR };
unsigned int allocation_method_table [] = { LOUDNESS, SNR };

#define SUBBANDS_4          (1<<1)
#define SUBBANDS_8          (1<<0)
unsigned int subbands_table [] = { 4, 8 };

char* get_msgtype(int sigval)
{
	char*sigtxt = "Unknown";
	if(sigval==MESSAGE_TYPE_COMMAND)
		sigtxt="MESSAGE_TYPE_COMMAND";
	if(sigval==MESSAGE_TYPE_ACCEPT)
		sigtxt="MESSAGE_TYPE_ACCEPT";
	if(sigval==MESSAGE_TYPE_REJECT)
		sigtxt="MESSAGE_TYPE_REJECT";
	return sigtxt;
}

char* get_signal(int sigval)
{
	char*sigtxt = "Unknown";
	if(sigval==AVDTP_DISCOVER)
		sigtxt="AVDTP_DISCOVER";
	if(sigval==AVDTP_GET_CAPABILITIES)
		sigtxt="AVDTP_GET_CAPABILITIES";
	if(sigval==AVDTP_SET_CONFIGURATION)
		sigtxt="AVDTP_SET_CONFIGURATION";
	if(sigval==AVDTP_GET_CONFIGURATION)
		sigtxt="AVDTP_GET_CONFIGURATION";
	if(sigval==AVDTP_RECONFIGURE)
		sigtxt="AVDTP_RECONFIGURE";
	if(sigval==AVDTP_OPEN)
		sigtxt="AVDTP_OPEN";
	if(sigval==AVDTP_START)
		sigtxt="AVDTP_START";
	if(sigval==AVDTP_CLOSE)
		sigtxt="AVDTP_CLOSE";
	if(sigval==AVDTP_SUSPEND)
		sigtxt="AVDTP_SUSPEND";
	if(sigval==AVDTP_ABORT)
		sigtxt="AVDTP_ABORT";
	if(sigval==AVDTP_SECURITY_CONTROL)
		sigtxt="AVDTP_SECURITY_CONTROL";
	return sigtxt;
}


void memcpy_changeendian(void *dst, const void *src, int size)
{
	int i;
	const uint16_t *ptrsrc = src;
	uint16_t *ptrdst = dst;
	for (i = 0; i < size / 2; i++) {
		*ptrdst++ = htons(*ptrsrc++);
	}
}

// Prepare packet headers
void init_request(struct avdtp_header *header, int request_id)
{
	static int transaction = 0;

	header->packet_type = PACKET_TYPE_SINGLE;
	header->message_type = MESSAGE_TYPE_COMMAND;
	header->transaction_label = transaction;
	header->signal_id = request_id;

	// clear rfa bits
	header->rfa0 = 0;

	transaction = (transaction + 1) & 0xf;
}

uint16_t get_socket_omtu(int sockfd)
{
	// Get mtu size
	uint16_t mtu = A2DPMAXIMUMTRANSFERUNITSIZE;
	struct l2cap_options opts;
	unsigned int iOptSize = sizeof(opts);
	if(getsockopt(sockfd, SOL_L2CAP, L2CAP_OPTIONS, &opts, &iOptSize) >= 0) {
		if (opts.omtu)
			mtu = min(opts.imtu, opts.omtu);
		if (!(mtu))
			mtu = A2DPMAXIMUMTRANSFERUNITSIZE;

		DBG("Socket %d imtu=%d, omtu=%d, flush_to=%d", sockfd, opts.imtu, opts.omtu, opts.flush_to);
	}
	return mtu;
}

int a2dp_ctl_use_socket(LPA2DP a2dp, int cmdfd)
{
	if(a2dp->state != AVDTP_STATE_DISCONNECTED)
		return -1;

	if(cmdfd < 0)
		return -1;

	if(a2dp->control_sk > 0)
		return -1;

	a2dp->control_sk = cmdfd;
	a2dp_set_state(AVDTP_STATE_IDLE);

	return 0;
}

int a2dp_state_use_socket(LPA2DP a2dp, int cmdfd)
{
	int ret = -1;

	if(cmdfd < 0)
		return -1;

	// First connection?
	if(a2dp->control_sk <= 0) {
		// It's a control socket
		DBG("Incoming socket %d : used as control socket", cmdfd);
		ret = a2dp_ctl_use_socket(a2dp, cmdfd);
		if(ret>=0) {
			// Passive Acceptor role
			a2dp->role = A2DPD_ACCEPTOR;
			DBG("Role is ACCEPTOR");
			// Skip sdp browse and connection intermediate state to go to idle state
			a2dp_set_state(AVDTP_STATE_IDLE);
		}
	} else if(a2dp->role == A2DPD_ACCEPTOR && a2dp->sk <= 0) {
		// Already connected, then it's a stream socket
		a2dp->sk = cmdfd;
		a2dp->mtu = get_socket_omtu(cmdfd);
		DBG("Incoming socket %d : used as stream socket (mtu=%d)", cmdfd, a2dp->mtu);
		a2dp_set_state(AVDTP_STATE_OPEN);
		ret = 0;
	} else {
		DBG("Incoming socket %d : refused", cmdfd);
		ret = -1;
	}

	return ret;
}

int a2dp_state_connect(LPA2DP a2dp)
{
	if(strlen(a2dp->bdaddr)==0)
		return -1;

	switch(a2dp->state) {
	case AVDTP_STATE_DISCONNECTED:	if(a2dp->role==A2DPD_NOROLE)
						a2dp_set_state(AVDTP_STATEX_SDP_CONNECTING);
					break;
	case AVDTP_STATE_OPEN:		if(a2dp->role==A2DPD_INITIATOR)
						a2dp_set_state(AVDTP_STATEX_STREAM_STARTING);
					break;
	default:                       return -1;
	}

	return 0;
}

int a2dp_is_streaming(LPA2DP a2dp)
{
	// Connected to a device but not necessarily streaming
	return
		(a2dp->state == AVDTP_STATE_STREAMING)
		;
}

int a2dp_is_connected(LPA2DP a2dp)
{
	// Connected to a device but not necessarily streaming
	return
		(a2dp->state == AVDTP_STATE_OPEN) ||
		(a2dp->state == AVDTP_STATEX_STREAM_STARTING) ||
		(a2dp->state == AVDTP_STATEX_STREAM_STARTING_WAIT) ||
		(a2dp->state == AVDTP_STATE_STREAMING) ||
		(a2dp->state == AVDTP_STATEX_STREAM_SUSPENDING_WAIT) ||
		(a2dp->state == AVDTP_STATEX_STREAM_SUSPENDING)
		;
}

int a2dp_is_connecting(LPA2DP a2dp)
{
	// Include connected state
	return
		(a2dp->state != AVDTP_STATE_DISCONNECTED)
		;
}

void a2dp_state_suspend(LPA2DP a2dp)
{
	if(a2dp->state == AVDTP_STATE_STREAMING) {
		a2dp_set_state(AVDTP_STATEX_STREAM_SUSPENDING);
	} else {
		DBG("Filtering state : State invalid for suspending");
	}
}

void a2dp_state_startstream(LPA2DP a2dp)
{
	if(a2dp->state == AVDTP_STATE_OPEN) {
		a2dp_set_state(AVDTP_STATEX_STREAM_STARTING);
	} else {
		DBG("State invalid for starting stream");
	}
}

int a2dp_reset_encoding_buffer(LPA2DP a2dp)
{
	DBG("");
	// Reset buffer of data to send
	a2dp->len = sizeof(struct media_packet_header) + sizeof(struct media_payload_header);
	a2dp->frame_count = 0;
	a2dp->seq_num = 1;
	a2dp->lenbufe = 0;
	a2dp->samples = 0;
	a2dp->samplescumul = 0;
	return 0;
}

// We have pcm data to send through bluetooth
int a2dp_transfer_raw(LPA2DP a2dp, const char *pcm_buffer, int pcm_buffer_size, AUDIOPACKETHEADER* hdr)
{
	// No error
	int result = 0;
	struct media_packet_header packet_header;
	struct media_payload_header payload_header;
	int codesize, datatoread;
	int written;

	// Check parameter
	if (a2dp == 0 || pcm_buffer == 0 || pcm_buffer_size == 0)
		RETURNERROR("Invalid Parameter");

	// How much data can be encoded by sbc at a time?
	// 16 bits * 2 channels * 16 blocks * 8 subbands = 4096bits = 512 o
	codesize = a2dp->sbc.subbands * a2dp->sbc.blocks * a2dp->sbc.channels * 2;
	// 44 bitpool?
	//codesize=a2dp->sbc.bitpool*a2dp->sbc.subbands*a2dp->sbc.blocks/8;
	datatoread = min((BUFS - a2dp->lenbufe), pcm_buffer_size);

	// If state is not streaming then return
	if(!a2dp_is_streaming(a2dp))
		return datatoread;

	if(a2dp->samplescumul == 0) {
		DBG("Samples starting");
		gettimeofday(&a2dp->samplescumultimestamp, NULL);
	}

	// Enqueue data in bufe
	if (a2dp->lenbufe + datatoread < BUFS) {
		// Append data to bufe, for sbc encoding
		memcpy_changeendian(a2dp->bufe + a2dp->lenbufe, pcm_buffer, datatoread);
		//DBG("Enqueud pcm (codesize=%d, datatoread=%d, lenbufe=%d/%d, mtu=%d)", codesize, datatoread, a2dp->lenbufe, BUFS, a2dp->mtu);
		a2dp->lenbufe += datatoread;
	} else {
		//DBG("Cannot enqueue pcm (codesize=%d, datatoread=%d, lenbufe=%d/%d, mtu=%d)", codesize, datatoread, a2dp->lenbufe, BUFS, a2dp->mtu);
		// Let the encoder dequeue the buffer and retry later
		datatoread = 0;
	}

	result = datatoread;

	// If bufe is full, encode
	if (a2dp->lenbufe >= codesize) {
		// Enough data to encode (sbc wants 1k blocks)
		int encoded;
		encoded = sbc_encode(&(a2dp->sbc), a2dp->bufe, codesize);	//encode

		if (encoded <= 0)
			RETURNERROR("Error while encoding");

		memmove(a2dp->bufe, a2dp->bufe + encoded, a2dp->lenbufe - encoded);	// Shift the bufe
		a2dp->lenbufe -= encoded;

		//DBG8("SBC encoding: len=%d, encoded=%d, total=%d, lenbufe=%d", a2dp->len, encoded, a2dp->sbc.len, a2dp->lenbufe);

		// Send data through bluetooth
		if (a2dp->len + a2dp->sbc.len >= a2dp->mtu) {

			// time to prepare and send the packet
			memset(&payload_header, 0, sizeof(payload_header));
			memset(&packet_header, 0, sizeof(packet_header));
			payload_header.frame_count = a2dp->frame_count;
			packet_header.v = 2;
			packet_header.pt = 1;
			packet_header.cc = 0;
			packet_header.sequence_number = htons(a2dp->seq_num);
			packet_header.timestamp = htonl(a2dp->samplescumul);
			packet_header.ssrc = htonl(1);

			memcpy(a2dp->buf, &packet_header, sizeof(packet_header));
			memcpy(a2dp->buf + sizeof(packet_header), &payload_header, sizeof(payload_header));
			if (a2dp->sk > 0) {
				// Pause?
				// The value 0 have finally been tested ;)
				// However, we may safely simulate a failed write
				if (!a2dp->pause_writing) {
					int count = 0;
					int retry = 0;
					static struct timeval send_date = { 0,0 };
					static struct timeval prev_date = { 0,0 };
					struct timeval send_delay = { 0,0 };
					struct timeval sendz_delay = { 0,0 };
					do {
						gettimeofday(&send_date, NULL);
						errno = 0;
						written = write(a2dp->sk, a2dp->buf, a2dp->len);
#ifdef A2DPD_ENABLE_RETRY_PATCH
#warning "retry patch support enabled"
						retry = (errno == EAGAIN);
#else
#warning "retry patch support disabled"
#endif
						if(!retry && prev_date.tv_sec != 0) {
							timersub(&send_date, &prev_date, &send_delay);
							timersub(&send_date, &a2dp->samplescumultimestamp, &sendz_delay);
							/*if((long)(sendz_delay.tv_sec*1000 + sendz_delay.tv_usec/1000) -
								(long)(((float)a2dp->samplescumul)/44100.0*1000.0)
								>= 0
								) {*/
							DBG8("send %d (cumul=%d) samples (delay=%ld ms, delay_real=%ld ms - delay_theo=%ld ms = delta=%ld ms).",
								a2dp->samples,
								a2dp->samplescumul,
								(long)(send_delay.tv_sec*1000 + send_delay.tv_usec/1000),
								(long)(sendz_delay.tv_sec*1000 + sendz_delay.tv_usec/1000),
								(long)(((float)a2dp->samplescumul)/44100.0*1000.0),
								(long)(sendz_delay.tv_sec*1000 + sendz_delay.tv_usec/1000) -
								(long)(((float)a2dp->samplescumul)/44100.0*1000.0)
								);
							//}
						}
#ifdef A2DPD_ENABLE_RETRY_PATCH
						if(retry)
							DBG("send (retry).");

						if(written < 0 && retry)
							usleep(150000);
#endif
					} while (count++ < 50 && written < 0 && retry);

					prev_date = send_date;

					// Send our data
					if (written != a2dp->len) {
						// Error while sending data
						DBG("Wrote %d not %d bytes (sbc.len=%d)", written, a2dp->len, a2dp->sbc.len);
						// Return the error
						result = written;
					} else {
						// Measure bandwith usage
						struct timeval now = { 0, 0 };
						struct timeval interval = { 0, 0 };

						if(a2dp->bandwithtimestamp.tv_sec==0)
							gettimeofday(&a2dp->bandwithtimestamp, NULL);

						//DBG("Wrote %d bytes.", written);
						// See if we must wait again
						gettimeofday(&now, NULL);
						timersub(&now, &a2dp->bandwithtimestamp, &interval);
						if(interval.tv_sec>0) {
							if(a2dp->user_flags & A2DPD_FLAGS_DISPLAYBANDWITH) {
								struct timeval now, lat;
								gettimeofday(&now, NULL);
								timersub(&now, &hdr->packet_date, &lat);
								DBG("Bandwith: %d (%d kbps) %d, latency=%d %s",
								a2dp->bandwithcount,
								a2dp->bandwithcount/128,
								a2dp->sbc.bitpool,
								(lat.tv_sec==0)?(int)lat.tv_usec:(int)lat.tv_sec,
								(lat.tv_sec==0)?"us.":"sec.");
							}
							a2dp->bandwithtimestamp = now;
							a2dp->bandwithcount = 0;
						}

						a2dp->bandwithcount += written;
					}
				} else {
					// Make the upper layer believe we sent data
					result = a2dp->len;
				}
			} else {
				DBG("Invalid socket");
			}
			// Reset buffer of data to send
			a2dp->len = sizeof(struct media_packet_header) + sizeof(struct media_payload_header);
			a2dp->frame_count = 0;
			a2dp->samples = 0;
			a2dp->seq_num++;
		}
		// Append sbc encoded data to buf, until buf reaches A2DPMAXIMUMTRANSFERUNITSIZE to send
		a2dp->frame_count++;
		memcpy(a2dp->buf + a2dp->len, a2dp->sbc.data, a2dp->sbc.len);
		a2dp->len += a2dp->sbc.len;
		a2dp->samples += encoded / 4;

		a2dp->samplescumul += encoded / 4;
	}

	return result;
}


void a2dp_stream_stop(snd_pcm_a2dp_t * a2dp)
{
	DBG("Closing stream socket %d", a2dp->sk);
	close_socket(&a2dp->sk);
	DBG("Closed");

	a2dp_set_state(AVDTP_STATE_IDLE);
	a2dp->sk = -1;
}

void a2dp_disconnect(snd_pcm_a2dp_t * a2dp)
{
	DBG("");
	a2dp_stream_stop(a2dp);

	if(a2dp->sdp_session != NULL) {
		sdp_close(a2dp->sdp_session);
		a2dp->sdp_session = NULL;
	}

	DBG("Closing ctl socket %d", a2dp->control_sk);
	close_socket(&a2dp->control_sk);
	DBG("Closed");

	a2dp_set_state(AVDTP_STATE_DISCONNECTED);
}

void a2dp_state_disconnect(LPA2DP a2dp)
{
	if(a2dp->state != AVDTP_STATE_DISCONNECTED) {
		switch(a2dp->state) {
		case AVDTP_STATE_OPEN:
		case AVDTP_STATEX_STREAM_STARTING:
		case AVDTP_STATEX_STREAM_STARTING_WAIT:
		case AVDTP_STATEX_STREAM_SUSPENDING:
		case AVDTP_STATEX_STREAM_SUSPENDING_WAIT:
		case AVDTP_STATEX_STREAM_CLOSING:
		case AVDTP_STATEX_STREAM_CLOSING_WAIT:
		case AVDTP_STATE_STREAMING:
		case AVDTP_STATE_CLOSING:
		case AVDTP_STATE_ABORTING:
			a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING);
			break;
		default:
			a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			break;
		}
	} else {
		DBG("Filtering state : already disconnected");
	}
}

void a2dp_set_dst_addr(LPA2DP a2dp, char* bdaddr)
{
	bacpy(&a2dp->src, BDADDR_ANY);

	if(bdaddr && bdaddr[0]) {
		strncpy(a2dp->bdaddr, bdaddr, sizeof(a2dp->bdaddr));
		a2dp->bdaddr[sizeof(a2dp->bdaddr)-1] = 0;
		str2ba(a2dp->bdaddr, &a2dp->dst);
	} else {
		a2dp->bdaddr[0] = 0;
	}

	// Disconnect so that we reconnect
	if(a2dp->state != AVDTP_STATE_DISCONNECTED)
		a2dp_state_disconnect(a2dp);
}

snd_pcm_a2dp_t *a2dp_alloc(void)
{
	snd_pcm_a2dp_t *a2dp;
	DBG("");
	a2dp = malloc(sizeof(*a2dp));
	if (!a2dp)
		return NULL;

	memset(a2dp, 0, sizeof(*a2dp));
	a2dp->seq_num = 1;
	a2dp->samples = 0;
	a2dp->samplescumul = 0;
	a2dp->mtu = A2DPMAXIMUMTRANSFERUNITSIZE;
	a2dp->len = sizeof(struct media_packet_header) + sizeof(struct media_payload_header);

	sbc_init(&a2dp->sbc, 0L);
	a2dp->sbc.rate = A2DPD_FRAME_RATE;
	a2dp->sbc.subbands = 8;	// safe default
	a2dp->sbc.blocks = 16;	// safe default
	a2dp->sbc.bitpool = 32;	// recommended value 53, safe default is 32
	DBG("(a2dp = %p)", a2dp);

	return a2dp;
}

void a2dp_free(LPA2DP a2dp)
{
	DBG("Disconnecting");
	a2dp_disconnect(a2dp);

	DBG("Freeing sbc");
	sbc_finish(&a2dp->sbc);

	DBG("(a2dp = %p)", a2dp);
	free(a2dp);

	DBG("OK");
}

void a2dp_init(void)
{
	a2dpd_register_sdp();
}

void a2dp_exit(void)
{
	a2dpd_unregister_sdp();
}

LPA2DP a2dp_new(A2DPSETTINGS* settings)
{
	snd_pcm_a2dp_t *a2dp = NULL;

	if(settings) {
		a2dp = a2dp_alloc();

		DBG("%s, %d", settings->bdaddr, settings->framerate);

		if (a2dp) {
			memcpy(&a2dp->settings, settings, sizeof(a2dp->settings));
			a2dp->sbc.rate = settings->framerate;
			a2dp->sbc.channels = max(1, min(settings->channels, 2));
			a2dp->sbc.bitpool = settings->sbcbitpool;
			a2dp->user_flags = settings->flags;
			if(settings->channels==1)
				a2dp->sbc.joint=1;

			a2dp_set_dst_addr(a2dp, settings->bdaddr);

			a2dp_set_state(AVDTP_STATE_DISCONNECTED);
		}
	}
	return a2dp;
}

void a2dp_destroy(LPA2DP *a2dp)
{
	DBG("a2dp = %p", a2dp);
	a2dp_free(*a2dp);
	*a2dp = NULL;
}

void a2dp_set_user_flags(LPA2DP a2dp, int flags)
{
	a2dp->user_flags = flags;
}

int a2dp_get_user_flags(LPA2DP a2dp)
{
	return a2dp->user_flags;
}

int a2dp_make_listen_socket(unsigned short psm)
{
	struct sockaddr_l2 addr;
	int on = 1;
	int sockfd = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);

	if (sockfd < 0)
		RETURNERROR("Cannot create socket for psm %d", (int)psm);

	(void)setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

	memset(&addr, 0, sizeof(addr));
	addr.l2_family = AF_BLUETOOTH;
	bacpy(&addr.l2_bdaddr, BDADDR_ANY);
	addr.l2_psm = htobs(psm);

	if (bind(sockfd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		close(sockfd);
		RETURNERROR("Cannot bind socket %d for psm %d", sockfd, (int)psm);
	}

	if (listen(sockfd, 5) < 0) {
		close(sockfd);
		RETURNERROR("Cannot listen socket %d for psm %d", sockfd, (int)psm);
	}

	return sockfd;
}

int a2dp_wait_connection(int sockfd, char *szRemote, int iRemoteSize, uint16_t * mtu)
{
	// Wait client connection
	struct sockaddr_l2 addr;
	int new_fd = -1;
	socklen_t addrlen = sizeof(addr);

	// Timeouts each second to read variables
	struct timeval t = { 1, 0 };
	(void)setsockopt(sockfd, SOL_SOCKET, SO_SNDTIMEO, &t, sizeof(t));
	(void)setsockopt(sockfd, SOL_SOCKET, SO_RCVTIMEO, &t, sizeof(t));

	if (szRemote)
		*szRemote = '\0';

	if(poll_accept(sockfd, 0)) {
		new_fd = accept(sockfd, (struct sockaddr *) &addr, &addrlen);

		if (new_fd >= 0) {
			if(mtu)
				*mtu = get_socket_omtu(sockfd);

			if(fcntl(new_fd, F_SETFL, O_NONBLOCK)<0)
				DBG("Failed to set non blocking socket %d", new_fd);

			if (szRemote) {
				char* tmpaddr = batostr(&addr.l2_bdaddr);
				strncpy(szRemote, tmpaddr, iRemoteSize);
				free(tmpaddr);
				szRemote[iRemoteSize - 1] = '\0';
			}
		}
	} else {
		sleep(1);
	}
	return new_fd;
}

// This function handle the bluetooth connection
int a2dp_ctl_handle_commands(LPA2DP a2dp, int sockfd, char* lpFrame, int lpFrameSize)
{
	int result = 0;
	int accepted = 0;
	int wrresult = 0;
	struct avdtp_header *pkt_hdr = (struct avdtp_header *) lpFrame;
	int ans_size = sizeof(*pkt_hdr);

	if (pkt_hdr->signal_id == AVDTP_DISCOVER) {
		struct sepd_resp* discover_resp = (struct sepd_resp*)lpFrame;
		DBG("Received signal AVDTP_DISCOVER(%d) from set", pkt_hdr->signal_id);
		ans_size = sizeof(*pkt_hdr)+sizeof(discover_resp->infos[0]);
		memset(lpFrame+sizeof(*pkt_hdr), 0, ans_size-sizeof(*pkt_hdr));
		accepted = 1;

		//Fill in the values of the structure
		discover_resp->infos[0].inuse0 = (a2dp&&a2dp_is_connected(a2dp))?1:0;
		discover_resp->infos[0].acp_seid = A2DPD_SEID;
		discover_resp->infos[0].media_type = AUDIO_MEDIA_TYPE;
		discover_resp->infos[0].tsep = 0; // 0 source / 1 sink

	} else if (pkt_hdr->signal_id == AVDTP_GET_CAPABILITIES) {
		struct getcap_resp* cap_resp = (struct getcap_resp*)lpFrame;
		DBG("Received signal AVDTP_GET_CAPABILITIES(%d) from set", pkt_hdr->signal_id);
		ans_size = sizeof(*cap_resp);
		memset(lpFrame+sizeof(*pkt_hdr), 0, ans_size-sizeof(*pkt_hdr));
		accepted = 1;

		//Fill in the values of the structure
		cap_resp->serv_cap=MEDIA_TRANSPORT_CATEGORY;
		cap_resp->serv_cap_len=0;
		cap_resp->cap_type=MEDIA_CODEC;
		cap_resp->media_type=AUDIO_MEDIA_TYPE;
		cap_resp->length=6;//sizeof(cap_resp->codec_elements.sbc_elements);
		cap_resp->media_codec_type=SBC_MEDIA_CODEC_TYPE;
		cap_resp->codec_elements.sbc_elements.channel_mode=(CHANNEL_MODE_MONO|/*CHANNEL_MODE_DUAL|*/CHANNEL_MODE_STEREO/*|CHANNEL_MODE_JOINT*/);
		cap_resp->codec_elements.sbc_elements.frequency=(FREQUENCY_44100);
		cap_resp->codec_elements.sbc_elements.allocation_method=(ALLOCATION_LOUDNESS/*|ALLOCATION_SNR*/);
		cap_resp->codec_elements.sbc_elements.subbands=(/*SUBBANDS_4|*/SUBBANDS_8);  
		cap_resp->codec_elements.sbc_elements.min_bitpool=2;
		cap_resp->codec_elements.sbc_elements.max_bitpool=250;
		cap_resp->codec_elements.sbc_elements.block_length=(/*BLOCK_LENGTH_4|BLOCK_LENGTH_8|BLOCK_LENGTH_12|*/BLOCK_LENGTH_16);

	} else if (pkt_hdr->signal_id == AVDTP_SET_CONFIGURATION) {
		struct set_config* cfg_req = (struct set_config*)lpFrame;
		struct set_config_resp* cfg_resp = (struct set_config_resp*)lpFrame;
		struct sbc_codec_specific_elements* sbc_elements = 0;
		int cfgok = 1;
		int v = 0;

		DBG("Received signal AVDTP_SET_CONFIGURATION(%d) from set", pkt_hdr->signal_id);

		if(a2dp) {
			// Check we support the settings proposed
			//cfgok = cfgok && (cfg_req->acp_seid == A2DPD_SEID);
			DBG("acp_seid: %d => %s", cfg_req->acp_seid, cfgok?"supported":"unsupported");
	
			cfgok = cfgok && (cfg_req->media_type == AUDIO_MEDIA_TYPE);
			DBG("media_type: %d => %s", cfgok, cfgok?"supported":"unsupported");
	
			cfgok = cfgok && (cfg_req->media_codec_type == SBC_MEDIA_CODEC_TYPE);
			DBG("media_codec_type: %d => %s", cfg_req->media_codec_type, cfgok?"supported":"unsupported");
	
			sbc_elements = &cfg_req->codec_elements.sbc_elements;
	
			// channel mode
			if(sbc_elements->channel_mode&CHANNEL_MODE_MONO) {
				if(a2dp) {
					a2dp->sbc.channels = 1;
					a2dp->sbc.joint = 0;
				}
			} else if(sbc_elements->channel_mode&CHANNEL_MODE_DUAL) {
				// Don't know how to set sbc
			} else if(sbc_elements->channel_mode&CHANNEL_MODE_STEREO) {
				if(a2dp) {
					a2dp->sbc.channels = 2;
					a2dp->sbc.joint = 0;
				}
			} else if(sbc_elements->channel_mode&CHANNEL_MODE_JOINT) {
				if(a2dp) {
					a2dp->sbc.channels = 2;
					a2dp->sbc.joint = 1;
				}
			} else {
				cfgok = 0;
			}
			DBG("channel_mode: %d, joint: %d => %s", a2dp->sbc.channels, a2dp->sbc.joint, cfgok?"supported":"unsupported");
	
			if(sbc_elements->frequency & FREQUENCY_16000) {
				v = 16000;
				if(a2dp->sbc.rate != 16000) {
					cfgok = 0;
				}
			} else if(sbc_elements->frequency & FREQUENCY_32000) {
				v = 32000;
				if(a2dp->sbc.rate != 32000) {
					cfgok = 0;
				}
			} else if(sbc_elements->frequency & FREQUENCY_44100) {
				v = 44100;
				if(a2dp->sbc.rate != 44100) {
					cfgok = 0;
				}
			} else if(sbc_elements->frequency & FREQUENCY_48000) {
				v = 48000;
				if(a2dp->sbc.rate != 48000) {
					cfgok = 0;
				}
			} else {
				v = -1;
				cfgok = 0;
			}
			DBG("frequency: %d => %s", v, cfgok?"supported":"unsupported");
	
			if(sbc_elements->allocation_method&ALLOCATION_SNR) {
				// SBC support SNR?
			} else if(sbc_elements->allocation_method&ALLOCATION_LOUDNESS) {
				// Mandatory
				cfgok = 1;
			} else {
				cfgok = 0;
			}
			DBG("allocation_method: %d => %s", sbc_elements->allocation_method, cfgok?"supported":"unsupported");
	
			if(sbc_elements->allocation_method&SUBBANDS_4) {
				a2dp->sbc.subbands = 4;
			} else if(sbc_elements->allocation_method&SUBBANDS_8) {
				a2dp->sbc.subbands = 8;
			} else {
				cfgok = 0;
			}
			DBG("subbands: %d => %s", sbc_elements->subbands, cfgok?"supported":"unsupported");
	
			if(sbc_elements->allocation_method&BLOCK_LENGTH_4) {
				a2dp->sbc.blocks = 4;
			} else if(sbc_elements->allocation_method&BLOCK_LENGTH_8) {
				a2dp->sbc.blocks = 8;
			} else if(sbc_elements->allocation_method&BLOCK_LENGTH_12) {
				a2dp->sbc.blocks = 12;
			}  else if(sbc_elements->allocation_method&BLOCK_LENGTH_16) {
				a2dp->sbc.blocks = 16;
			}  else {
				cfgok = 0;
			}
			DBG("block_length: %d => %s", sbc_elements->block_length, cfgok?"supported":"unsupported");

			DBG("A2DPD  Prefered bitpool %d", a2dp->settings.sbcbitpool);
			a2dp->sbc.bitpool = a2dp->settings.sbcbitpool;

			// PTS sends min and max to the same value so use what is wanted
			if(sbc_elements->min_bitpool == sbc_elements->max_bitpool) {
				if((a2dp->sbc.bitpool < 2) || (a2dp->sbc.bitpool > 250)) {
					DBG("Invalid bitpool %d wanted", sbc_elements->min_bitpool);
					cfgok = 0;
				} else {
					a2dp->sbc.bitpool = sbc_elements->min_bitpool;
				}
			} else {
				// This dumb of HBH DS 970 send 2 as min and 250 as max so use ours
				// We just check ours is within the supported range
				// Plantronics send 2 as min and 32 as max even if 53 is accepted
				if((a2dp->sbc.bitpool < sbc_elements->min_bitpool) || (a2dp->sbc.bitpool > sbc_elements->max_bitpool)) {
					DBG("bitpool %d not supported by device", a2dp->sbc.bitpool);
					if(sbc_elements->max_bitpool<a2dp->sbc.bitpool) {
						a2dp->sbc.bitpool = sbc_elements->max_bitpool;
						DBG("Align to device capabilities : choosing bitpool %d", a2dp->sbc.bitpool);
					} else {
						DBG("Can't align to device capabilities");
						cfgok = 0;
					}
				}
			}
			DBG("min_bitpool: %d => %s", sbc_elements->min_bitpool, cfgok?"supported":"unsupported");
			DBG("max_bitpool: %d => %s", sbc_elements->max_bitpool, cfgok?"supported":"unsupported");
			DBG("A2DPD  Selected bitpool %d", a2dp->sbc.bitpool);

			if(!cfgok)
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
		}
		// If config is OK then we reply
		ans_size = sizeof(*cfg_resp);
		memset(lpFrame+sizeof(*pkt_hdr), 0, ans_size-sizeof(*pkt_hdr));
		accepted = cfgok;
		cfg_resp->serv_cat = MEDIA_TRANSPORT_CATEGORY;
		cfg_resp->error_code = 0;

	} else if (pkt_hdr->signal_id == AVDTP_GET_CONFIGURATION) {
		DBG("Received signal AVDTP_GET_CONFIGURATION(%d) from set", pkt_hdr->signal_id);
		// Not implemented
		//accepted = 1;
		
	} else if (pkt_hdr->signal_id == AVDTP_RECONFIGURE) {
		DBG("Received signal AVDTP_RECONFIGURE(%d) from set", pkt_hdr->signal_id);
		// Not implemented
		//accepted = 1;
	
	} else if (pkt_hdr->signal_id == AVDTP_OPEN) {
		struct open_stream_rsp* open_resp = (struct open_stream_rsp*)lpFrame;
		DBG("Received signal AVDTP_OPEN(%d) from set", pkt_hdr->signal_id);
		ans_size = sizeof(*open_resp);
		memset(lpFrame+sizeof(*pkt_hdr), 0, ans_size-sizeof(*pkt_hdr));

		if(a2dp) {
			if(a2dp->state == AVDTP_STATE_IDLE && a2dp->role == A2DPD_ACCEPTOR) {
				a2dp->pause_writing = 0;
				a2dp_set_state(AVDTP_STATE_OPEN);

				//Fill in the values of the structure
				open_resp->error = 0;
				accepted = 1;
			} else {
				DBG("Filtering state : AVDTP_OPEN received while not in AVDTP_STATE_IDLE or not ACCEPTOR");
			}
		}
	} else if (pkt_hdr->signal_id == AVDTP_START) {
		DBG("Received signal AVDTP_START(%d) from set", pkt_hdr->signal_id);

		if(a2dp) {
			if(a2dp->state == AVDTP_STATE_OPEN) {
				a2dp->pause_writing = 0;
				a2dp_set_state(AVDTP_STATE_STREAMING);
				accepted = 1;
				a2dpd_signal_command("down", "avdtp_start");
				a2dpd_signal_command("up", "avdtp_start");
			} else {
				DBG("Filtering state : AVDTP_START received while not in AVDTP_STATE_OPEN");
			}
		}

	} else if (pkt_hdr->signal_id == AVDTP_CLOSE) {
		DBG("Received signal AVDTP_CLOSE(%d) from set", pkt_hdr->signal_id);
		if(a2dp) {
			a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			accepted = 1;
			a2dpd_signal_command("down", "avdtp_close");
			a2dpd_signal_command("up", "avdtp_close");
		}

	} else if (pkt_hdr->signal_id == AVDTP_SUSPEND) {
		DBG("Received signal AVDTP_SUSPEND(%d) from set", pkt_hdr->signal_id);
		if(a2dp) {
			if(a2dp->state == AVDTP_STATE_STREAMING) {
				a2dp->pause_writing = 1;
				a2dp_set_state(AVDTP_STATE_OPEN);
				accepted = 1;
				a2dpd_signal_command("down", "avdtp_suspend");
				a2dpd_signal_command("up", "avdtp_suspend");
			} else {
				DBG("Filtering state : AVDTP_SUSPEND received while not in AVDTP_STATE_STREAMING");
			}
		}

	} else if (pkt_hdr->signal_id == AVDTP_ABORT) {
		DBG("Received signal AVDTP_ABORT(%d) from set", pkt_hdr->signal_id);
		if(a2dp) {
			if(a2dp->role == A2DPD_INITIATOR) {
				a2dp_set_state(AVDTP_STATE_IDLE);
			}
			if(a2dp->role == A2DPD_INITIATOR) {
				a2dp_set_state(AVDTP_STATE_IDLE);
			}
			a2dpd_signal_command("down", "avdtp_abort");
			a2dpd_signal_command("up", "avdtp_abort");
			accepted = 1;
		}
	} else if (pkt_hdr->signal_id == AVDTP_SECURITY_CONTROL) {
		DBG("Received signal AVDTP_SECURITY_CONTROL(%d) from set", pkt_hdr->signal_id);
		// Not implemented
		//accepted = 1;
	}  else {
		DBG("Unexpected headset directive %d", pkt_hdr->signal_id);
		//accepted = 1;
	}

	pkt_hdr->message_type = accepted ? MESSAGE_TYPE_ACCEPT : MESSAGE_TYPE_REJECT;
	DBG("Answering command packet (msgtype=%s,signal=%s)", get_msgtype(pkt_hdr->message_type), get_signal(pkt_hdr->signal_id));

	// Reply to command received
	wrresult = write(sockfd, pkt_hdr, ans_size);

	if (wrresult != ans_size) {
		DBG("FAILED Answering command packet (msgtype=%s,signal=%s) wrresult=%d/%d", get_msgtype(pkt_hdr->message_type), get_signal(pkt_hdr->signal_id),
		wrresult, ans_size);
	}
	return result;
}

// This function handle the bluetooth connection
int a2dp_handle_avdtp_message(LPA2DP a2dp, int sockfd, struct avdtp_header *sent_packet, struct avdtp_header *answer, size_t answer_size)
{
	int result = 0;
	char lpFrame[A2DPMAXIMUMTRANSFERUNITSIZE];

	// Data to read?
	if(poll_accept(sockfd, 0)) {
		ssize_t iReceived = recv(sockfd, lpFrame, sizeof(lpFrame), MSG_WAITALL|MSG_NOSIGNAL);
		struct avdtp_header *pkt_hdr = (struct avdtp_header *) lpFrame;
		if (iReceived > 0) {
			// Manage the packet
			if (sent_packet == NULL) {
				//int i;
				DBG("socket %d: Received %d bytes", sockfd, iReceived);
				// dump_raw(lpFrame, iReceived);
				result = 0;
			} else if ((pkt_hdr->message_type == MESSAGE_TYPE_ACCEPT) && (pkt_hdr->signal_id == sent_packet->signal_id) && answer) {
				// Got expected answer
				memcpy(answer, lpFrame, (ssize_t)answer_size > iReceived ? answer_size : iReceived);
				result = iReceived;
			} else {
				// Got bad answer
				result = 0;
			}
	
			// Handle the packet if the incoming message is a command
			if (pkt_hdr->message_type == MESSAGE_TYPE_COMMAND) {
				a2dp_ctl_handle_commands(a2dp, sockfd, lpFrame, iReceived);
			} else {
				DBG("Read non command packet (msgtype=%s,signal=%s)", get_msgtype(pkt_hdr->message_type), get_signal(pkt_hdr->signal_id));
			}
		} else {
			result = iReceived;
			if (iReceived < 0 && errno != EAGAIN) {
				DBG("socket %d: Receive failed %d", sockfd, iReceived);
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
		}
	} else {
		if(errno != EAGAIN && errno != EINTR) {
			DBG("socket %d: poll_accept returned error", sockfd);
			a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
		}
	}

	return result;
}

int get_avdtp_psm(LPA2DP a2dp)
{
	sdp_list_t *attrid, *search, *seq, *next;
	sdp_data_t *pdlist;
	uuid_t group;
	uint32_t range = 0x0000ffff;
	int err;
	int psm = 25;

	a2dp->flags = 0;

	// Remove non blocking attribute
	fcntl(a2dp->sdp_session->sock, F_SETFL, fcntl(a2dp->sdp_session->sock, F_GETFL, 0)&~O_NONBLOCK);

	/* 0x1108->all? 0x1101->rf sink 0x111e->handsfree 0x1108->headset */
	sdp_uuid16_create(&group, 0x110d);
	search = sdp_list_append(0, &group);
	attrid = sdp_list_append(0, &range);
	err = sdp_service_search_attr_req(a2dp->sdp_session, search, SDP_ATTR_REQ_RANGE, attrid, &seq);
	sdp_list_free(search, 0);
	sdp_list_free(attrid, 0);

	if (err) {
		RETURNERROR("Service Search failed.");
	}

	if (!seq) {
		DBG("Service Search record empty.");
	} else {
		DBG("Parsing results");
	}

	// Free sdp record
	for (; seq; seq = next) {
		sdp_record_t *rec = (sdp_record_t *) seq->data;
		DBG("Record");
		next = seq->next;
		free(seq);
		sdp_record_free(rec);
	}

	DBG("Service Search OK");

	sdp_uuid16_create(&group, PNP_INFO_SVCLASS_ID);
	search = sdp_list_append(0, &group);
	attrid = sdp_list_append(0, &range);
	err = sdp_service_search_attr_req(a2dp->sdp_session, search, SDP_ATTR_REQ_RANGE, attrid, &seq);
	sdp_list_free(search, 0);
	sdp_list_free(attrid, 0);

	if (err) {
		RETURNERROR("Service Search failed.");
	}

	DBG("Checking non spec audio");

	for (; seq; seq = next) {
		sdp_record_t *rec = (sdp_record_t *) seq->data;
		uint16_t vendor, product, version;

		pdlist = sdp_data_get(rec, 0x0201);
		vendor = pdlist ? pdlist->val.uint16 : 0x0000;

		pdlist = sdp_data_get(rec, 0x0202);
		product = pdlist ? pdlist->val.uint16 : 0x0000;

		pdlist = sdp_data_get(rec, 0x0203);
		version = pdlist ? pdlist->val.uint16 : 0x0000;

		DBG("Product ID %04x:%04x:%04x", vendor, product, version);

		if (vendor == 0x1310 && product == 0x0100 && version == 0x0104) {
			DBG("Enabling GCT media payload workaround");

			a2dp->flags |= A2DPD_FLAGS_NONSPECAUDIO;
		}

		next = seq->next;
		free(seq);
		sdp_record_free(rec);
	}

	return psm;
}

void a2dp_add_fd_to_poll(LPA2DP a2dp, struct pollinfo* pollinfos)
{
	switch(a2dp->state) {
		case AVDTP_STATEX_SDP_CONNECTING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->sk_sdp, POLLOUT, -1, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_CTL_CONNECTING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLOUT, -1, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_STREAM_CONNECTING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->sk, POLLOUT, -1, NULL, NULL, NULL);
			break;
		case AVDTP_STATE_IDLE:
//			if(a2dp->role == A2DPD_ACCEPTOR)
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, -1, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_DISCOVERING_RESP:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_GETTING_CAPABILITIES_RESP:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_SETTING_CONFIGURATION_RESP:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_STREAM_OPENING_RESP:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATE_OPEN:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_STREAM_STARTING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATE_STREAMING:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_STREAM_SUSPENDING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATEX_STREAM_CLOSING_WAIT:
			add_fd_to_poll(pollinfos, a2dp->control_sk, POLLIN, 50, NULL, NULL, NULL);
			break;
		case AVDTP_STATE_DISCONNECTED:
			// We want to block infinitely
			break;
		default:
			// We don't want to block until we arrive in state AVDTP_STATE_DISCONNECTED
			pollinfos->polltimeout = (pollinfos->polltimeout == -1)?50:min(pollinfos->polltimeout, 50);
			break;
	}
}


// This function handle the bluetooth connection
int a2dp_state_machine(LPA2DP a2dp)
{
	struct pollfd pollfds[1];
	int errcode = 0;
	unsigned int opt_size = sizeof(errcode);
	struct sockaddr_l2 addr;
	struct sepd_req discover_req;
	struct getcap_req cap_req;
	//struct getcap_resp cap_resp;
	struct set_config cfg_req;
	struct set_config_resp cfg_resp;
	struct stream_cmd open_req;
	struct open_stream_rsp open_resp;
	struct stream_cmd start_req;
	struct start_stream_rsp start_resp;
	struct stream_cmd suspend_req;
	struct start_stream_rsp suspend_resp;
	struct stream_cmd close_req;
	struct close_stream_rsp close_resp;

	int v,size;

	switch(a2dp->state) {
		case AVDTP_STATE_DISCONNECTED:
			// Delightly do nothing
			a2dp->role = A2DPD_NOROLE;
			break;
		case AVDTP_STATEX_DISCONNECTING:
			a2dp_disconnect(a2dp);
			break;
		case AVDTP_STATEX_SDP_CONNECTING:
			a2dp->role = A2DPD_INITIATOR;
			DBG("Role is INITIATOR");

			a2dp->sdp_session = sdp_connect_async(&a2dp->dst);

			if(a2dp->sdp_session != NULL) {
				a2dp->sk_sdp = sdp_get_socket(a2dp->sdp_session);
				DBG("SDP connection on socket %d", a2dp->sk_sdp);
				a2dp_set_state(AVDTP_STATEX_SDP_CONNECTING_WAIT);
			} else  {
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_SDP_CONNECTING_WAIT:
			// If SDP connection is established
			pollfds[0].fd = a2dp->sk_sdp;
			pollfds[0].events = POLLOUT;
			pollfds[0].revents = 0;
			if(poll(pollfds, 1, 0)>0) {
				DBG("SDP connection terminated");

				if((getsockopt(a2dp->sk_sdp, SOL_SOCKET, SO_ERROR, &errcode, &opt_size) == 0)
				&& (errcode == 0)) {
					int psm = get_avdtp_psm(a2dp);
					DBG("Found psm %d", (int)psm);
					// Free sdp data
					if(a2dp->sdp_session)
						sdp_close(a2dp->sdp_session);
					a2dp->sdp_session = NULL;
					// Retrieve channel
					if(psm>0) {
						a2dp->psm_cmd = psm;
						a2dp_set_state(AVDTP_STATEX_CTL_CONNECTING);
					} else {
						a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
					}
				} else {
					errno = errcode;
					DBG("SDP connection failed");
					a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
				}
			}
			break;
		case AVDTP_STATEX_CTL_CONNECTING:
			// Start async connection to channel
			a2dp->control_sk = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
			DBG("Socket is %d", a2dp->control_sk);

			if(fcntl(a2dp->control_sk, F_SETFL, O_NONBLOCK)<0)
				DBG("Failed to set non blocking socket %d", a2dp->control_sk);

			memset(&addr, 0, sizeof(addr));
			addr.l2_family = AF_BLUETOOTH;
			bacpy(&addr.l2_bdaddr, &a2dp->dst);
			addr.l2_psm = htobs(a2dp->psm_cmd);

			if((connect(a2dp->control_sk, (struct sockaddr *)&addr, sizeof(addr)) < 0) && ((errno == EAGAIN) || (errno == EINPROGRESS))) {
				a2dp_set_state(AVDTP_STATEX_CTL_CONNECTING_WAIT);
			} else {
				DBG("Connection failed");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_CTL_CONNECTING_WAIT:
			// Poll ctl socket
			pollfds[0].fd = a2dp->control_sk;
			pollfds[0].events = POLLOUT;
			pollfds[0].revents = 0;
			if(poll(pollfds, 1, 0)>0) {
				if((getsockopt(a2dp->control_sk, SOL_SOCKET, SO_ERROR, &errcode, &opt_size) == 0)
				&& (errcode == 0)) {
					a2dp_set_state(AVDTP_STATE_IDLE);
				} else {
					errno = errcode;
					DBG("CTL connection failed");
					a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
				}
			}
			break;
		case AVDTP_STATE_IDLE:
			if(a2dp->role == A2DPD_INITIATOR) {
				a2dp_set_state(AVDTP_STATEX_DISCOVERING);
			} else if(a2dp->role == A2DPD_ACCEPTOR) {
				a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, NULL, NULL, 0);
			} else {
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_DISCOVERING:
			// Start configuration
			memset(&discover_req, 0, sizeof(discover_req));
			init_request(&discover_req.header, AVDTP_DISCOVER);

			if(write(a2dp->control_sk, &discover_req, sizeof(discover_req)) == sizeof(discover_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp_set_state(AVDTP_STATEX_DISCOVERING_RESP);
			} else {
				DBG("Discovery failed %d (socket = %d)", errno, a2dp->control_sk);
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_DISCOVERING_RESP:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&discover_req, 0, sizeof(discover_req));
				init_request(&discover_req.header, AVDTP_DISCOVER);
				memset(&a2dp->state_info.discover.resp, 0, sizeof(a2dp->state_info.discover.resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &discover_req.header, &a2dp->state_info.discover.resp.header, sizeof(a2dp->state_info.discover.resp));
				if (size > 0) {
					DBG("AVDTP_DISCOVER success");
					a2dp->state_info.discover.curr_seid = 0;
					a2dp->state_info.discover.nb_seid = (size - sizeof(a2dp->state_info.discover.resp.header)) / sizeof(struct acp_seid_info);
					a2dp_set_state(AVDTP_STATEX_GETTING_CAPABILITIES);
				} else {
					//DBG("AVDTP_DISCOVER got bad answer");
				}
			} else {
				DBG("Too many answer for AVDTP_DISCOVER");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_GETTING_CAPABILITIES:
			if(a2dp->state_info.discover.curr_seid<a2dp->state_info.discover.nb_seid) {
				memset(&cap_req, 0, sizeof(cap_req));
				init_request(&cap_req.header, AVDTP_GET_CAPABILITIES);
				cap_req.acp_seid = a2dp->state_info.discover.resp.infos[a2dp->state_info.discover.curr_seid].acp_seid;
				a2dp->state_info.discover.curr_seid_acp_seid = cap_req.acp_seid;
				if (write(a2dp->control_sk, &cap_req, sizeof(cap_req)) == sizeof(cap_req)) {
					DBG("Getting capabilities for SEID=%d", cap_req.acp_seid);
					a2dp->state_info.message_time = time(NULL);
					a2dp_set_state(AVDTP_STATEX_GETTING_CAPABILITIES_RESP);
				} else {
					DBG("AVDTP_GET_CAPABILITIES failed");
					a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
				}
			} else {
				DBG("No more seid to browse");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_GETTING_CAPABILITIES_RESP:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&cap_req, 0, sizeof(cap_req));
				init_request(&cap_req.header, AVDTP_GET_CAPABILITIES);
				memset(&a2dp->state_info.discover.cap_resp, 0, sizeof(a2dp->state_info.discover.cap_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &cap_req.header, (void*)&a2dp->state_info.discover.cap_resp, sizeof(a2dp->state_info.discover.cap_resp));
				if (size > 0) {
					DBG("AVDTP_GET_CAPABILITIES success for seid %d",
						a2dp->state_info.discover.curr_seid_acp_seid);
					DBG("servcap_cap=%d, servcap_len=%d,",
						a2dp->state_info.discover.cap_resp.serv_cap,
						a2dp->state_info.discover.cap_resp.serv_cap_len);
					DBG("cap_type=%d, length=%d", 
						a2dp->state_info.discover.cap_resp.cap_type,
						a2dp->state_info.discover.cap_resp.length);
					DBG("media_type=%d, codec=%d",
						a2dp->state_info.discover.cap_resp.media_type,
						a2dp->state_info.discover.cap_resp.media_codec_type);

					//FIXME Check SEID is audio & SBC
					// Else next seid
					if(a2dp->state_info.discover.cap_resp.media_type == AUDIO_MEDIA_TYPE && a2dp->state_info.discover.cap_resp.media_codec_type == SBC_MEDIA_CODEC_TYPE) {
						DBG("A2DPD  Prefered bitpool %d", a2dp->settings.sbcbitpool);
						a2dp->sbc.bitpool = a2dp->settings.sbcbitpool;
						DBG("Device Prefered bitpool %d -> %d", 
							a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.min_bitpool,
							a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.max_bitpool);
	
						if(a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.max_bitpool < a2dp->settings.sbcbitpool) {
							a2dp->sbc.bitpool = a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.max_bitpool;
						}
						DBG("A2DPD  Selected bitpool %d", a2dp->sbc.bitpool);

						a2dp_set_state(AVDTP_STATEX_SETTING_CONFIGURATION);
					} else {
						// Next SEID
						DBG("SEID %d is not SBC audio", a2dp->state_info.discover.curr_seid);

						a2dp->state_info.discover.curr_seid++;
						a2dp_set_state(AVDTP_STATEX_GETTING_CAPABILITIES);
					}
				} else {
					//DBG("AVDTP_GET_CAPABILITIES got bad answer");
				}
			} else {
				DBG("Too many answer for AVDTP_GET_CAPABILITIES");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_SETTING_CONFIGURATION:
			memset(&cfg_req, 0, sizeof(cfg_req));
			init_request(&cfg_req.header, AVDTP_SET_CONFIGURATION);
			cfg_req.serv_cap = MEDIA_TRANSPORT_CATEGORY;
			cfg_req.acp_seid = a2dp->state_info.discover.curr_seid_acp_seid;
			cfg_req.int_seid = A2DPD_SEID;
			cfg_req.cap_type = MEDIA_CODEC;
			cfg_req.length = 6;
			cfg_req.media_type = AUDIO_MEDIA_TYPE;
			cfg_req.media_codec_type = SBC_MEDIA_CODEC_TYPE;

			switch (a2dp->sbc.channels) {
			case 1:
				v = 8;
				break;
			case 2:
			default:
				v = 2;
				break;
			}
			cfg_req.codec_elements.sbc_elements.channel_mode = v;
		
			switch (a2dp->sbc.rate) {
			case 16000:
				v = 8;
				break;
			case 32000:
				v = 4;
				break;
			case 48000:
				v = 1;
				break;
			case 44100:
			default:
				v = 2;
				break;
			}
			cfg_req.codec_elements.sbc_elements.frequency = v;
			cfg_req.codec_elements.sbc_elements.allocation_method = 1 << 1;
		
			switch (a2dp->sbc.subbands) {
			case 4:
				v = 2;
				break;
			case 8:
			default:
				v = 1;
				break;
			}
			cfg_req.codec_elements.sbc_elements.subbands = v;
		
			switch (a2dp->sbc.blocks) {
			case 4:
				v = 8;
				break;
			case 8:
				v = 4;
				break;
			case 12:
				v = 2;
				break;
			case 16:
			default:
				v = 1;
				break;
			}
			cfg_req.codec_elements.sbc_elements.block_length = v;
			cfg_req.codec_elements.sbc_elements.min_bitpool = a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.min_bitpool;
			cfg_req.codec_elements.sbc_elements.max_bitpool = a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.max_bitpool;
		
			if (!(a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.channel_mode & cfg_req.codec_elements.sbc_elements.channel_mode)) {
				DBG("headset does not support this channel mode");
			}
		
			if (!(a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.frequency & cfg_req.codec_elements.sbc_elements.frequency)) {
				DBG("headset does not support this frequency");
			}
		
			if (!(a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.allocation_method & cfg_req.codec_elements.sbc_elements.allocation_method)) {
				DBG("headset does not support this allocation_method");
			}
		
			if (!(a2dp->state_info.discover.cap_resp.codec_elements.sbc_elements.subbands & cfg_req.codec_elements.sbc_elements.subbands)) {
				DBG("headset does not support this subbands setting");
			}
			if(write(a2dp->control_sk, &cfg_req, sizeof(cfg_req)) == sizeof(cfg_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp_set_state(AVDTP_STATEX_SETTING_CONFIGURATION_RESP);
			} else {
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_SETTING_CONFIGURATION_RESP:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&cfg_req, 0, sizeof(cfg_req));
				init_request(&cfg_req.header, AVDTP_SET_CONFIGURATION);
				memset(&cfg_resp, 0, sizeof(cfg_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &cfg_req.header, (void*)&cfg_resp, sizeof(cfg_resp));
				if (size > 0) {
					DBG("AVDTP_SET_CONFIGURATION success");
					a2dp->mtu = A2DPMAXIMUMTRANSFERUNITSIZE;
					a2dp->seid = a2dp->state_info.discover.curr_seid_acp_seid;
					a2dp->psm_stream = 25;
					a2dp_set_state(AVDTP_STATEX_STREAM_OPENING);
				} else {
					//DBG("AVDTP_SET_CONFIGURATION got bad answer");
				}
			} else {
				DBG("Too many answer for AVDTP_SET_CONFIGURATION");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_STREAM_OPENING:
			memset(&open_req, 0, sizeof(open_req));
			init_request(&open_req.header, AVDTP_OPEN);
			open_req.acp_seid = a2dp->seid;

			if (write(a2dp->control_sk, &open_req, sizeof(open_req)) == sizeof(open_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp_set_state(AVDTP_STATEX_STREAM_OPENING_RESP);
			} else {
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_STREAM_OPENING_RESP:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&open_req, 0, sizeof(open_req));
				init_request(&open_req.header, AVDTP_OPEN);
				memset(&open_resp, 0, sizeof(open_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &open_req.header, (void*)&open_resp, sizeof(open_resp));
				if (size > 0) {
					DBG("AVDTP_OPEN success");
					a2dp_set_state(AVDTP_STATE_CONFIGURED);
				} else {
					//DBG("AVDTP_OPEN got bad answer");
				}
			} else {
				DBG("Too many answer for AVDTP_OPEN");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATE_CONFIGURED:
			a2dp_set_state(AVDTP_STATEX_STREAM_CONNECTING);
			break;
		case AVDTP_STATEX_STREAM_CONNECTING:
			// Start async connection to channel
			a2dp->sk = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
			DBG("Socket is %d", a2dp->sk);

			if(fcntl(a2dp->sk, F_SETFL, O_NONBLOCK)<0)
				DBG("Failed to set non blocking socket %d", a2dp->sk);

			memset(&addr, 0, sizeof(addr));
			addr.l2_family = AF_BLUETOOTH;
			bacpy(&addr.l2_bdaddr, &a2dp->dst);
			addr.l2_psm = htobs(a2dp->psm_stream);

			if((connect(a2dp->sk, (struct sockaddr *)&addr, sizeof(addr)) < 0) && ((errno == EAGAIN) || (errno == EINPROGRESS))) {
				a2dp_set_state(AVDTP_STATEX_STREAM_CONNECTING_WAIT);
			} else {
				DBG("Connection failed");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_STREAM_CONNECTING_WAIT:
			// Poll ctl socket
			pollfds[0].fd = a2dp->sk;
			pollfds[0].events = POLLOUT;
			pollfds[0].revents = 0;
			if(poll(pollfds, 1, 0)>0) {
				if((getsockopt(a2dp->sk, SOL_SOCKET, SO_ERROR, &errcode, &opt_size) == 0)
				&& (errcode == 0)) {
					DBG("Stream connected");
					a2dp->mtu = get_socket_omtu(a2dp->sk);
					a2dp_set_state(AVDTP_STATE_OPEN_START_STREAMING);
				} else {
					errno = errcode;
					DBG("STREAM connection failed");
					a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
				}
			}
			break;
		case AVDTP_STATE_OPEN_START_STREAMING:
			a2dp_set_state(AVDTP_STATEX_STREAM_STARTING);
			break;
		case AVDTP_STATE_OPEN:
			a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, NULL, NULL, 0);
			break;
		case AVDTP_STATEX_STREAM_STARTING:
			memset(&start_req, 0, sizeof(start_req));
			init_request(&start_req.header, AVDTP_START);
			start_req.acp_seid = a2dp->seid;

			if (write(a2dp->control_sk, &start_req, sizeof(start_req)) == sizeof(start_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp_set_state(AVDTP_STATEX_STREAM_STARTING_WAIT);
			} else {
				a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING);
			}
			break;
		case AVDTP_STATEX_STREAM_STARTING_WAIT:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&start_req, 0, sizeof(start_req));
				init_request(&start_req.header, AVDTP_START);
				memset(&start_resp, 0, sizeof(start_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &start_req.header, (void*)&start_resp, sizeof(start_resp));
				if (size > 0) {
					DBG("AVDTP_START success");
					a2dp->pause_writing = 0;
					a2dp_set_state(AVDTP_STATE_STREAMING);
				} else {
				}
			} else {
				DBG("Too many answer for AVDTP_START");
				a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING);
			}
			break;
		case AVDTP_STATE_STREAMING:
			a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &cfg_req.header, (void*)&cfg_resp, sizeof(cfg_resp));
			break;
		case AVDTP_STATEX_STREAM_SUSPENDING:
			memset(&suspend_req, 0, sizeof(suspend_req));
			init_request(&suspend_req.header, AVDTP_SUSPEND);
			suspend_req.acp_seid = a2dp->seid;

			if (write(a2dp->control_sk, &suspend_req, sizeof(suspend_req)) == sizeof(suspend_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp->pause_writing = 0;
				a2dp_set_state(AVDTP_STATEX_STREAM_SUSPENDING_WAIT);
			} else {
				a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING);
			}
			break;
		case AVDTP_STATEX_STREAM_SUSPENDING_WAIT:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&suspend_req, 0, sizeof(suspend_req));
				init_request(&suspend_req.header, AVDTP_SUSPEND);
				memset(&suspend_resp, 0, sizeof(suspend_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &suspend_req.header, (void*)&suspend_resp, sizeof(suspend_resp));
				if (size > 0) {
					DBG("AVDTP_SUSPEND success");
					a2dp_set_state(AVDTP_STATE_OPEN);
				} else {
				}
			} else {
				DBG("Too many answer for AVDTP_SUSPEND");
				a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING);
			}
			break;
		case AVDTP_STATE_CLOSING:
		case AVDTP_STATEX_STREAM_CLOSING:
			memset(&close_req, 0, sizeof(close_req));
			init_request(&close_req.header, AVDTP_CLOSE);
			close_req.acp_seid = a2dp->seid;

			if (write(a2dp->control_sk, &close_req, sizeof(close_req)) == (ssize_t)sizeof(close_req)) {
				a2dp->state_info.message_time = time(NULL);
				a2dp->pause_writing = 1;
				a2dp_set_state(AVDTP_STATEX_STREAM_CLOSING_WAIT);
			} else {
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATEX_STREAM_CLOSING_WAIT:
			if(time(NULL)-a2dp->state_info.message_time<MAXMESSAGETIME) {
				memset(&close_req, 0, sizeof(close_req));
				init_request(&close_req.header, AVDTP_CLOSE);
				memset(&close_resp, 0, sizeof(close_resp));
				size = a2dp_handle_avdtp_message(a2dp, a2dp->control_sk, &close_req.header, (void*)&close_resp, sizeof(close_resp));
				if (size > 0) {
					DBG("AVDTP_CLOSE success");
					a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
				} else {
				}
			} else {
				DBG("Too many answer for AVDTP_CLOSE");
				a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			}
			break;
		case AVDTP_STATE_ABORTING:
			a2dp_set_state(AVDTP_STATEX_DISCONNECTING);
			break;
		default:
			DBG("Unhandled state %d", a2dp->state);
			break;
	}
	return 0;
}
