/*
 * Bluetooth Headset ALSA Plugin
 *
 * Copyright (c) 2006-2007 by Fabien Chevalier
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>

#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>

/* For bluetooth addresses manipulation functions */
#include <bluetooth/bluetooth.h>
/* For SCO constants */
#include <bluetooth/sco.h>

#include <config.h>

/* Debug */

#ifdef NDEBUG
	#define DBG(fmt, arg...)
#else
	#define DBG(fmt, arg...)  printf("DEBUG(pid=%d): %s: " fmt "\n" , getpid(), __FUNCTION__ , ## arg)
#endif

/* Defines */

#define PCM_SERVER_SOCKET "\0bluez-headset-pcm"

#define SCO_PACKET_LEN        48
#define SCO_RATE              8000
#define SCO_SAMPLE_SIZE       2

#define OPENED_PLAYBACK 1
#define OPENED_CAPTURE  2

#define PERIODS_MIN_PLAYBACK 2
#define PERIODS_MIN_CAPTURE  4
#define PERIODS_MAX 200

#ifndef SCO_TXBUFS
#define SCO_TXBUFS 0x03
#endif

#ifndef SCO_RXBUFS
#define SCO_RXBUFS 0x04
#endif

  /* IPC packet types */
#define PKT_TYPE_CFG_BDADDR        0
#define PKT_TYPE_CFG_PAGINGTIMEOUT 1
#define PKT_TYPE_CFG_ACK           2
#define PKT_TYPE_CFG_NACK          3
#define PKT_TYPE_ERROR_IND         4
#define PKT_TYPE_STREAMING_IND     5

/* Macros */
#define ARRAY_NELEMS(a) (sizeof(a) / sizeof(a[0]) )

/* Constants */

/* Data types */

typedef struct ipc_packet {
	unsigned char type;
	union {
		bdaddr_t     bdaddr;	               /* PKT_TYPE_CFG_BDADDR        */
		int32_t      timeout;                  /* PKT_TYPE_CFG_PAGINGTIMEOUT */
		int32_t	     errorcode;		       /* PKT_TYPE_ERROR_IND        */
	};
} ipc_packet_t;

typedef struct sco_packet {
	char sco_data[SCO_PACKET_LEN];
} sco_packet_t;

typedef struct snd_pcm_sco_headset {
	snd_pcm_ioplug_t io;
	snd_pcm_sframes_t hw_ptr;
	sco_packet_t pkt;
	unsigned int sco_data_count;
} snd_pcm_sco_headset_t;

/* Global variables */
static int          serverfd   = -1;
static int          scofd      = -1;
static unsigned int opened_for =  0;

/* Function declarations */

static int  do_cfg(int serverfd, const bdaddr_t * bdaddr, long timeout);
static int  get_scofd(int serverfd, int *scofd, int nonblock);

/* Function definitions */

static snd_pcm_sframes_t sco_headset_write(snd_pcm_ioplug_t *io,
				   const snd_pcm_channel_area_t *areas,
				   snd_pcm_uframes_t offset,
				   snd_pcm_uframes_t size)
{
	snd_pcm_sco_headset_t *bt_headset = io->private_data;
	snd_pcm_sframes_t ret = 0;
	snd_pcm_uframes_t frames_to_read;

	DBG("areas->step=%u, areas->first=%u, offset=%lu, size=%lu, io->nonblock=%u", areas->step, areas->first, offset, size, io->nonblock);

	if((bt_headset->sco_data_count + SCO_SAMPLE_SIZE * size) <= SCO_PACKET_LEN) {
		frames_to_read = size;
	}
	else {
		frames_to_read = (SCO_PACKET_LEN - bt_headset->sco_data_count) / SCO_SAMPLE_SIZE;
	}

	/* Ready for more data */
	unsigned char *buff;
	buff = (unsigned char *) areas->addr + (areas->first + areas->step * offset) / 8;
	memcpy(bt_headset->pkt.sco_data + bt_headset->sco_data_count, buff, areas->step / 8 * frames_to_read);	

	if((bt_headset->sco_data_count + areas->step / 8 * frames_to_read) == SCO_PACKET_LEN) {
		int rsend;
		/* Actually send packet */
		rsend = send(scofd, &bt_headset->pkt, sizeof(sco_packet_t), io->nonblock ? MSG_DONTWAIT : 0);
		if(rsend > 0) {
			/* Reset sco_data_count pointer */		
			bt_headset->sco_data_count = 0;
		
			/* Increment hardware transmition pointer */
			bt_headset->hw_ptr = (bt_headset->hw_ptr + SCO_PACKET_LEN / SCO_SAMPLE_SIZE) % io->buffer_size;

			ret = frames_to_read;
		}
		else {
			/* EPIPE means device underrun in ALSA world. But we mean we lost contact
                           with server, so we have to find another error code */
			ret = (errno == EPIPE ? -EIO : -errno);
			if(errno == EPIPE) SYSERR("Lost contact with headsetd");
		}	
	}
	else {
		/* Remember we have some frame in the pipe now */
		bt_headset->sco_data_count += areas->step / 8 * frames_to_read;
		/* Ask for more */
		ret = frames_to_read;
	}

	DBG("returning %d", (int)ret);
	return ret;
}

static snd_pcm_sframes_t sco_headset_read(snd_pcm_ioplug_t *io,
				  const snd_pcm_channel_area_t *areas,
				  snd_pcm_uframes_t offset,
				  snd_pcm_uframes_t size)
{
	snd_pcm_sco_headset_t *bt_headset = io->private_data;
	snd_pcm_sframes_t ret = 0;
	DBG("areas->step=%u, areas->first=%u, offset=%lu, size=%lu, io->nonblock=%u", areas->step, areas->first, offset, size, io->nonblock);
	if(bt_headset->sco_data_count == 0) {
		int nrecv = recv(scofd, &bt_headset->pkt, sizeof(sco_packet_t),
			MSG_WAITALL | (io->nonblock ? MSG_DONTWAIT : 0 ));
		if(nrecv == sizeof(sco_packet_t)) {
			ret = 0;
			/* Increment hardware transmition pointer */
			bt_headset->hw_ptr = (bt_headset->hw_ptr + SCO_PACKET_LEN / 2) % io->buffer_size;
		}
		else if(nrecv > 0) {
			ret = -EIO;
			SNDERR(strerror(-ret));
		}
		else if(nrecv == -1 && errno == EAGAIN) {
			ret = -EAGAIN;
		}
		else { /* nrecv < 0 */
			/* EPIPE means device underrun in ALSA world. But we mean we lost contact
                           with server, so we have to find another error code */
			ret = (errno == EPIPE ? -EIO : -errno);
			SYSERR("Lost contact with headsetd");
		}	
	}
	if(ret == 0) { /* Still ok, proceed */
		snd_pcm_uframes_t frames_to_write;
		unsigned char *buff;
		buff = (unsigned char *) areas->addr + (areas->first + areas->step * offset) / 8;
		
		if((bt_headset->sco_data_count + 2 * size) <= SCO_PACKET_LEN) {
			frames_to_write = size;
		}
		else {
			frames_to_write = (SCO_PACKET_LEN - bt_headset->sco_data_count) / 2;
		}
		memcpy(buff, bt_headset->pkt.sco_data + bt_headset->sco_data_count, areas->step / 8 * frames_to_write);	
		bt_headset->sco_data_count += (areas->step / 8 * frames_to_write);
		bt_headset->sco_data_count %= SCO_PACKET_LEN;
		/* Return written frames count */
		ret = frames_to_write;
	}

	DBG("returning %d", (int)ret);
	return ret;
}

static snd_pcm_sframes_t sco_headset_pointer(snd_pcm_ioplug_t *io)
{
	snd_pcm_sco_headset_t *bt_headset = io->private_data;

	DBG("returning bt_headset->hw_ptr=%lu", bt_headset->hw_ptr);
	return bt_headset->hw_ptr;
}

static int sco_headset_start(snd_pcm_ioplug_t *io)
{
	DBG("");
	return 0;
}

static int sco_headset_stop(snd_pcm_ioplug_t *io)
{
	DBG("");
	return 0;
}

static int sco_headset_prepare(snd_pcm_ioplug_t *io)
{
	snd_pcm_sco_headset_t *bt_headset = io->private_data;

	DBG("Preparing with io->period_size = %lu, io->buffer_size = %lu", io->period_size, io->buffer_size);

	if(io->stream == SND_PCM_STREAM_PLAYBACK) {
		/* If not null for playback, xmms doesn't display time correctly */
		bt_headset->hw_ptr = 0;
	}
	else {
		/* ALSA library is really picky on the fact hw_ptr is not null. If it is, capture won't start */
		bt_headset->hw_ptr = io->period_size;
	}
	return 0;
}

static int sco_headset_hw_params(snd_pcm_ioplug_t *io, snd_pcm_hw_params_t *params)
{
	uint32_t period_count = io->buffer_size / io->period_size;

	DBG("period_count = %d", period_count);

	if(setsockopt(scofd, SOL_SCO, 
			io->stream == SND_PCM_STREAM_PLAYBACK ? SCO_TXBUFS : SCO_RXBUFS,
			&period_count,
			sizeof(period_count)) == 0) {
		return 0;
	} else if(setsockopt(scofd, SOL_SCO,
                        io->stream == SND_PCM_STREAM_PLAYBACK ? SO_SNDBUF : SO_RCVBUF,
                        &period_count,
                        sizeof(period_count)) == 0) {
                return 0;
        } else {
		SNDERR("Unable to set number of SCO buffers : please upgrade your Kernel !");
		return -EINVAL;
	}
}

static int sco_headset_hw_constraint(snd_pcm_sco_headset_t *bt_headset)
{
	snd_pcm_ioplug_t *io = &bt_headset->io; 
	static const snd_pcm_access_t access_list[] = {
		SND_PCM_ACCESS_RW_INTERLEAVED,
		/* Mmap access is really useless from this driver point of view, 
                   but we support it because some pieces of software out there insist on using it */
		SND_PCM_ACCESS_MMAP_INTERLEAVED
	};
	static const unsigned int format[] = {
		SND_PCM_FORMAT_S16_LE
	};
	int err;

	/* Access type */
	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_ACCESS,
		ARRAY_NELEMS(access_list), access_list);
	if (err < 0)
		return err;	

	/* supported formats */
	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_FORMAT,
		ARRAY_NELEMS(format), format);
	if (err < 0)
		return err;

	/* supported channels */
	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_CHANNELS,
		1, 1);
	if (err < 0)
		return err;

	/* supported rates */
	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_RATE, SCO_RATE, SCO_RATE);
	if (err < 0)
		return err;

	/* period size */
	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIOD_BYTES,
					    SCO_PACKET_LEN, SCO_PACKET_LEN);
	if (err < 0)
		return err;
	/* periods */
	if(io->stream == SND_PCM_STREAM_PLAYBACK) {
		err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIODS, PERIODS_MIN_PLAYBACK, PERIODS_MAX);
	}
	else {
		/* Experiments have shown that with HZ=250, Minimum packets we must be able to buffer is 3 - so
		let it be 4 just for safety -- Experiments done on x86 Intel Centrino */
		err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIODS, PERIODS_MIN_CAPTURE, PERIODS_MAX);
	}
	if (err < 0)
		return err;
	if (err < 0)
		return err;

	return 0;
}

static int sco_headset_close(snd_pcm_ioplug_t *io)
{
	snd_pcm_sco_headset_t *bt_headset = io->private_data;
	DBG("closing ioplug=%p", io);
	switch(io->stream) {
	case SND_PCM_STREAM_PLAYBACK:
		DBG("Closing Playback stream");
		opened_for &= ~OPENED_PLAYBACK;
		break;
	case SND_PCM_STREAM_CAPTURE:
		DBG("Closing Capture stream");
		opened_for &= ~OPENED_CAPTURE;
		break;
	default:
		SNDERR("Unexpected ioplug received !!");
		return -EINVAL;	
	}

	/* If not any opened stream anymore, close files */
	if(opened_for == 0) {
		if(scofd != -1) {
			close(scofd);
			scofd = -1;
		}
		close(serverfd);
		serverfd = -1;
	}
	io->private_data = 0;
	free(bt_headset);
	return 0;
}

static int  do_cfg(int serverfd, const bdaddr_t * bdaddr, long timeout)
{
	ipc_packet_t pkt;
	int res;

	/* Sending bd_addr */
	pkt.type = PKT_TYPE_CFG_BDADDR;
	bacpy(&pkt.bdaddr, bdaddr);
	res = send(serverfd, &pkt, sizeof(ipc_packet_t), 0);	
	if(res < 0) {
		return errno;
	}
	do {
		res = recv(serverfd, &pkt, sizeof(ipc_packet_t), 0);
	} while((res < 0) && (errno == EINTR));
	if(res < 0) {
		return errno;
	}
	if((pkt.type != PKT_TYPE_CFG_ACK) && (pkt.type != PKT_TYPE_CFG_NACK)) {
		SNDERR("Unexpected packet type received: type = %d", pkt.type);
		return EINVAL;
	}

	/* Sending timeout */
	pkt.type = PKT_TYPE_CFG_PAGINGTIMEOUT;
	pkt.timeout = timeout;
	res = send(serverfd, &pkt, sizeof(ipc_packet_t), 0);	
	if(res < 0) {
		return errno;
	}
	do {
		res = recv(serverfd, &pkt, sizeof(ipc_packet_t), 0);
	} while((res < 0) && (errno == EINTR));
	if(res < 0) {
		return errno;
	}
	if((pkt.type != PKT_TYPE_CFG_ACK) && (pkt.type != PKT_TYPE_CFG_NACK)) {
		SNDERR("Unexpected packet type received: type = %d", pkt.type);
		return EINVAL;
	}

	return 0;
}

static int get_scofd(int serverfd, int* scofd, int nonblock)
{
	/* receiving SCO fd through ancilliary data*/	
        char cmsg_b[CMSG_SPACE(sizeof(int))];  /* ancillary data buffer */
	ipc_packet_t pkt;
	int ret;
 	struct iovec iov = {
              .iov_base = &pkt,        /* Starting address */
              .iov_len  = sizeof(pkt)  /* Number of bytes  */
        };
	struct msghdr msgh = {
		.msg_name       = 0,
		.msg_namelen    = 0,
		.msg_iov        = &iov,
		.msg_iovlen     = 1,
		.msg_control    = &cmsg_b,
		.msg_controllen = CMSG_LEN(sizeof(int)),
		.msg_flags      = 0
	};

	ret = recvmsg(serverfd, &msgh, nonblock ? MSG_DONTWAIT : 0);
	if(ret > 0) {
		if(pkt.type == PKT_TYPE_STREAMING_IND) {
			struct cmsghdr *cmsg;
			/* Receive auxiliary data in msgh */
			for (cmsg = CMSG_FIRSTHDR(&msgh);
				cmsg != NULL;
				cmsg = CMSG_NXTHDR(&msgh,cmsg)) {
				if (cmsg->cmsg_level == SOL_SOCKET
					&& cmsg->cmsg_type == SCM_RIGHTS) {
					sco_packet_t sco_pkt;
					/* yep - got it !! */
					*scofd = (*(int *) CMSG_DATA(cmsg));
					/* It is possible there is some outstanding
					data in the pipe - we have to empty it */ 
					while(recv(*scofd, &sco_pkt, sizeof(sco_packet_t),
						MSG_DONTWAIT) > 0);
					return 0;
				}
			}
			return -EINVAL;
		}
		else if(pkt.type == PKT_TYPE_ERROR_IND){
			return -pkt.errorcode;
		}
		else {
			SNDERR("Unexpected packet type received: type = %d", pkt.type);
			return -EINVAL;
		}
	}
	else if(ret < 0 && errno == EAGAIN) {
		return -EAGAIN;
	}
	else {
		int err = errno;
		SNDERR("Unable to receive SCO fd: %s", strerror(errno));
		return -err;
	}
}


static snd_pcm_ioplug_callback_t sco_headset_playback_callback = {
	.close = sco_headset_close,
	.hw_params = sco_headset_hw_params,
	.start = sco_headset_start,
	.stop = sco_headset_stop,
	.prepare = sco_headset_prepare,
	.transfer = sco_headset_write,
	.pointer = sco_headset_pointer,
};

static snd_pcm_ioplug_callback_t sco_headset_capture_callback = {
	.close = sco_headset_close,
	.hw_params = sco_headset_hw_params,
	.start = sco_headset_start,
	.stop = sco_headset_stop,
	.prepare = sco_headset_prepare,
	.transfer = sco_headset_read,
	.pointer = sco_headset_pointer,
};


SND_PCM_PLUGIN_DEFINE_FUNC(sco)
{
	snd_config_iterator_t i, next;
	int err;
	snd_pcm_sco_headset_t *headset = 0;
	bdaddr_t     hs_bdaddr = {{0, 0, 0, 0, 0, 0}};
	long timeout = -1;
	struct sockaddr_un  socket_location = {
		AF_UNIX, PCM_SERVER_SOCKET
	};
	
	DBG("Starting pcm_sco_headset plugin.");
	DBG("Open mode is for %s.", stream == SND_PCM_STREAM_PLAYBACK ? "Playback" : "Capture");

	snd_config_for_each(i, next, conf) {
		snd_config_t *n = snd_config_iterator_entry(i);
		const char *id;
		if (snd_config_get_id(n, &id) < 0)
			continue;
		if (strcmp(id, "comment") == 0 || strcmp(id, "type") == 0)
			continue;
		if (!strcmp(id, "bdaddr")) {
			const char *addr;
			if (snd_config_get_string(n, &addr) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			str2ba(addr, &hs_bdaddr);
			continue;
		}
		if (!strcmp(id, "timeout")) {
			if (snd_config_get_integer(n, &timeout) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}
		SNDERR("Unknown field %s", id);
		return -EINVAL;
	}

	if((stream == SND_PCM_STREAM_PLAYBACK) && (opened_for & OPENED_PLAYBACK)) {
		SNDERR("Cannot open Bluetooth Headset PCM plugin twice for playback.");
		return -EINVAL;
	}
	if((stream == SND_PCM_STREAM_CAPTURE) && (opened_for & OPENED_CAPTURE)) {
		SNDERR("Cannot open Bluetooth Headset PCM plugin twice for capture.");
		return -EINVAL;
	}

	opened_for |= (stream == SND_PCM_STREAM_PLAYBACK ? OPENED_PLAYBACK : OPENED_CAPTURE);
	
	if(serverfd == -1) {
		/* First PCM to be opened, try to connect socket to headsetd */
		serverfd = socket(PF_LOCAL, SOCK_STREAM, 0);
	
		err = connect(serverfd,
			(struct sockaddr *)&socket_location, sizeof(socket_location));
		if(err == 0) {
			err = do_cfg(serverfd, &hs_bdaddr, timeout);
			if(err != 0 || ((err = get_scofd(serverfd, &scofd, 0)) != 0)) {
				goto error3;
			}	
		}
		else {
			err = -errno;
			SNDERR("Socket connection returned %s", strerror(errno));
			goto error3;
		}
	}

	headset = calloc(1, sizeof(snd_pcm_sco_headset_t));

	headset->io.version = SND_PCM_IOPLUG_VERSION;
	headset->io.name = "Bluetooth SCO Audio";
	headset->io.mmap_rw     =  0; /* No direct mmap com */
	headset->io.poll_fd     =  scofd;
	headset->io.poll_events =  stream == SND_PCM_STREAM_PLAYBACK ? POLLOUT : POLLIN;
	headset->io.callback = stream == SND_PCM_STREAM_PLAYBACK ?
		&sco_headset_playback_callback : &sco_headset_capture_callback;
	
	err = snd_pcm_ioplug_create(&headset->io, name, stream, mode);
	if (err < 0)
		goto error2;

	if ((err = sco_headset_hw_constraint(headset)) < 0) {
		goto error; 
	}
	
	headset->io.private_data = headset;
	*pcmp = headset->io.pcm;

	DBG("opened as ioplug=%p, pcm=%p, ioplug->callback = %p", &headset->io, headset->io.pcm, headset->io.callback);

	return 0;

error:
	snd_pcm_ioplug_delete(&headset->io);

error2:
	free(headset);

error3:
	opened_for &= (stream == SND_PCM_STREAM_PLAYBACK ? ~OPENED_PLAYBACK : ~OPENED_CAPTURE);
	if(!opened_for) {
		close(serverfd);
		serverfd = -1;
	}
	return err;
}

SND_PCM_PLUGIN_SYMBOL(sco);
