/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include "a2dpd_uinput.h"
#include "a2dpd_protocol.h"

#ifdef ENABLE_UINPUT
#include <linux/input.h>
#include <linux/uinput.h>
#define UINPUT_DEVICE          "/dev/input/uinput"

static int uinput_fd = -1;
#endif

void send_key(unsigned short key)
{
#ifdef ENABLE_UINPUT
	unsigned short code = KEY_MAX;
	struct input_event ev = {
		.type = EV_KEY,
		.code = code,
		.time = {0,}
	};

	if (uinput_fd == -1)
		return;

	switch(key) {
	case A2DPD_UINPUT_PLAY: code = KEY_PLAY; break;
	case A2DPD_UINPUT_STOP: code = KEY_STOP; break;
	case A2DPD_UINPUT_PAUSE: code = KEY_PAUSE; break;
	case A2DPD_UINPUT_PREV: code = KEY_PREVIOUSSONG; break;
	case A2DPD_UINPUT_NEXT: code = KEY_NEXTSONG; break;
	}

	if (code > KEY_MAX)
		return;

	ev.value = 1;		// press...
	write(uinput_fd, &ev, sizeof(ev));

	ev.value = 0;		// then release
	write(uinput_fd, &ev, sizeof(ev));
#endif
}

int init_uinput()
{
#ifdef ENABLE_UINPUT
	int fd, i;
	struct uinput_dev dev = {
		.id = {
		.bustype = BUS_BLUETOOTH,
		.version = 0x0001,
		}
	};

	if ((fd = open(UINPUT_DEVICE, O_WRONLY)) < 0) {
		DBG("Cannot open " UINPUT_DEVICE);
		goto shutdown;
	}

	if (write(fd, &dev, sizeof(dev)) < sizeof(dev)) {
		DBG("Cannot create a uinput device");
		goto release;
	}

	if (ioctl(fd, UI_SET_EVBIT, EV_KEY))
		goto release;

	for (i = 0; i <= KEY_MAX; i++)
		if (ioctl(fd, UI_SET_KEYBIT, i))
			goto release;
	if (ioctl(fd, UI_DEV_CREATE))
		goto release;

	uinput_fd = fd;

	return 0;
release:
	if(fd>0)
		ioctl(fd, UI_DEV_DESTROY);
shutdown:
	if(fd>0)
		close(fd);
#endif
	return 1;
}

void kill_uinput()
{
#ifdef ENABLE_UINPUT
#warning "uinput support enabled"
	if (uinput_fd == -1)
		return;

	ioctl(uinput_fd, UI_DEV_DESTROY);
	close(uinput_fd);
#else
#warning "uinput support disabled"
#endif
}

void a2dpd_register_uinput()
{
	init_uinput();
}

void a2dpd_unregister_uinput()
{
	kill_uinput();
}

