# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gtk import STOCK_DIALOG_INFO
from ConfigParser import ConfigParser, NoSectionError
from base64 import b64decode
from thread import start_new_thread
from os.path import join
from time import sleep

try:
    from xmpp import NS_PUBSUB, NS_PRIVACY, Node, Iq, Client, Presence
    NS_TUNE = 'http://jabber.org/protocol/tune'
    xmpp_imported = True
except ImportError:
    xmpp_imported = False

from common.config import ConfigLoader

from plugins.jabber.config import Config

class Jabber:
    def __init__(self, extensionsloader):
        self.extensions = extensionsloader
        self.plugin = {'name': 'Jabber',
                       'version': 0.2,
                       'logo': 'plugins/jabber/jabber_small.png',
                       'configurable': True,
                       'authors': 'Erwan Briand <erwan@codingteam.net>',
                       'license': 'GNU General Public License 3',
                       'description': _('Send PEP notifications')}

    def start_plugin(self):
        # python-xmpp is an optional depends
        if not xmpp_imported:
            self.stop_plugin()
            return

        # Start configuration
        name = self.plugin['name'].lower() + '.cfg'
        self.config = ConfigLoader()
        self.config_file = join(self.config.confdir, 'plugins', name)

        self.config_gui = Config(self.plugin, self.config_file,
                                 self.get_configuration)

        # Start a new Jabber client
        config = self.get_configuration()

        # Get Jabber account
        self.is_jabber_account = False
        if config['username'] != '' and config['password'] != '':
            self.jid = config['username']

            pwd = b64decode(config['password'])

            try:
                user = self.jid.split('@')[0]
                server = self.jid.split('@')[1]
            except IndexError:
                return

            # Start the connection to the Jabber server
            self.connection = Client(server, debug=[])
            self.connection.RegisterDisconnectHandler(
              self.connection.reconnectAndReauth)

            def connect_to_account(server, user, pwd):
                """Connect to the Jabber account of the user."""
                if not self.connection.connect():
                    raise Exception('Unable to connect to %s' % server)
                if not self.connection.auth(user=user, password=pwd,
                                            resource='Bluemindo'):
                    raise Exception('Unable to authenticate as %s' % self.jid)

                self.is_jabber_account = True

                # Use an invisible list (Omega, I love you <3)
                if not self.check_invisible_list():
                    self.create_invisible_list()
                self.activate_invisible_list()

                self.connection.send(Presence(priority=-42))

                while 1:
                    try:
                        self.connection.Process(5)
                    except:
                        exit(0)

            # Thread the conection
            start_new_thread(connect_to_account, (server, user, pwd))

        def loadconfig(args):
            self.config_gui.configuration(args)

        def saveconfig(args):
            self.config_gui.save_config(args)

        def load_new_song(song):
            title = song[0]
            artist = song[1]
            album = song[2]
            track = song[6]
            length = song[7]

            if self.is_jabber_account:
                # Test if connection is active, if not, reconnect and reauth
                if self.connection.isConnected() is None:
                    self.connection.reconnectAndReauth()

                # Create the PEP stanza
                item = Node('tune', {'xmlns': NS_TUNE})

                itemchild = item.addChild('artist')
                itemchild.addData(artist)

                itemchild = item.addChild('source')
                itemchild.addData(album)

                itemchild = item.addChild('title')
                itemchild.addData(title)

                itemchild = item.addChild('track')
                itemchild.addData(track)

                itemchild = item.addChild('length')
                itemchild.addData(length)

                query = Iq('set', to=self.jid)
                querychild = query.addChild('pubsub', namespace=NS_PUBSUB)
                querychild = querychild.addChild('publish', {'node': NS_TUNE})
                querychild = querychild.addChild('item', {}, [item])

                # Send the PEP notification
                self.connection.send(query)

        # Connect to Bluemindo's signals
        self.extensions.connect('OnPlayNewSong', load_new_song)
        self.extensions.connect('OnStopPressed', self.stop_publish_anything)
        self.extensions.connect('OnBluemindoQuitted', self.quit_jabber_account)
        self.extensions.connect('OnModuleConfiguration', loadconfig)
        self.extensions.connect('OnModuleConfigurationSave', saveconfig)

    def stop_publish_anything(self):
        """Send empty PEP notification."""
        iq = Iq(typ='set')
        iq.pubsub = iq.addChild('pubsub', namespace=NS_PUBSUB)
        iq.pubsub.publish = iq.pubsub.addChild('publish',
                                                     attrs={'node': NS_TUNE})
        iq.pubsub.publish.item = iq.pubsub.publish.addChild('item',
                                                     attrs={'id': 'current'})

        tune = iq.pubsub.publish.item.addChild('tune')
        tune.setNamespace(NS_TUNE)

        self.connection.send(iq)

    def quit_jabber_account(self):
        """Quit Bluemindo, so disconnect from Jabber network."""
        if self.is_jabber_account:
            self.is_jabber_account = False
            self.stop_publish_anything()
            self.connection.send(Presence(frm=self.jid, typ='unavailable'))
            sleep(1)
            del self.connection

    def check_invisible_list(self):
        """Return TRUE if this account has a privacy list for invisibility."""
        has_privacy = False

        request = Iq(typ='get')
        request.addChild('query', namespace=NS_PRIVACY)

        query = self.connection.SendAndWaitForResponse(request, timeout=25)
        children = query.getQueryChildren()

        if children is not None:
            for child in children:
                if (child.getName() == 'list' and
                    child.getAttr('name') == 'bluemindo'):
                    has_privacy = True

        return has_privacy

    def create_invisible_list(self):
        """Create an invisible list."""
        iq = Iq(typ='set')
        iq.query = iq.addChild('query', namespace=NS_PRIVACY)
        iq.query.list = iq.query.addChild('list', attrs={'name': 'bluemindo'})
        iq.query.list.item = iq.query.list.addChild('item', attrs={
                                                    'action': 'deny',
                                                    'order': '1'})
        iq.query.list.item.addChild('presence-out')
        self.connection.send(iq)

    def activate_invisible_list(self):
        """Activate the invisible list."""
        iq = Iq(typ='set')
        iq.query = iq.addChild('query', namespace=NS_PRIVACY)
        iq.query.addChild('active', attrs={'name': 'bluemindo'})
        self.connection.send(iq)

    def stop_plugin(self):
        self.quit_jabber_account()

    def get_configuration(self):
        configparser = ConfigParser()
        configparser.read(self.config_file)
        config = {}

        try:
            for item in configparser.items(self.plugin['name']):
                try:
                    value = int(item[1])
                except ValueError:
                    value = str(item[1])

                config[item[0]] = value
        except NoSectionError:
            config['username'] = ''
            config['password'] = ''

        return config