//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using Rss;
using Atom.Core;
using Atom.Core.Collections;
using System;
using System.IO;
using System.Net;
using System.Xml;
using System.Xml.XPath;

//using System.Web;

namespace Imendio.Blam {
    class CertCheck : ICertificatePolicy
    {
        public CertCheck()
        {
        }

        public bool CheckValidationResult(ServicePoint sp,
            System.Security.Cryptography.X509Certificates.X509Certificate cert,
            WebRequest request, int problem)
        {
            bool ignore = Conf.Get(Preference.IGNORE_SSL_ERR, false);
            if(ignore){
                System.Console.WriteLine("I was asked! allowed!");         
                return true;
            } else {
                System.Console.WriteLine("not allowing");
                return false;
            }
        }
    }
    public class FeedUpdater {
    
    private static string feed_type(Stream feed)
    {
        string type = null;

        XPathDocument doc = new XPathDocument(feed);
        XPathNavigator nav = doc.CreateNavigator();
        XmlNamespaceManager nsm = new XmlNamespaceManager(nav.NameTable);
        XPathExpression expr = nav.Compile("/atom03:feed|/atom10:feed");
        
        nsm.AddNamespace("atom10", "http://www.w3.org/2005/Atom");
        nsm.AddNamespace("atom03", "http://purl.org/atom/ns#");
        expr.SetContext(nsm);
        
        XPathNodeIterator iter = nav.Select(expr);
        iter.MoveNext();
        
        if(iter.Current != null){
            switch(iter.Current.NamespaceURI){
                case "http://www.w3.org/2005/Atom":
                    type = "Atom 1.0";
                break;
                case "http://purl.org/atom/ns#":
                    type = "Atom 0.3";
                break;
                default:
                    type = "unknown";
                break;
            }
        } else {
            type = "unknown";
        }
        
        return type;
    }
    
    public static bool Update (Channel channel)
    {
        channel.LastRefreshed = DateTime.Now;

        bool updated = false;
        WebResponse res = GetChannelData(channel);
        Stream stream = null;

        if(res == null){
            return false; /* Feed hasn't changed or we couldn't connect. */
        }

        stream = res.GetResponseStream();
	    
        if(channel.Type == ""){

            string type = feed_type(stream);
            
            /* If we know for certain, assign it directly. */
            if(type == "Atom 0.3" || type == "Atom 1.0"){
                channel.Type = "Atom";
            } else {
                try{
                /* We don't know the type of feed. Try RSS. */
                RssReader reader = new RssReader(GetChannelData(channel).GetResponseStream());
                RssElement e = reader.Read();
                    
                if(e != null){ /* If the RSS library will take it, we say it's RSS. */
                    if(e is RssRedirect){ /* Redirection needs special handling. */
                        channel.Url = ((RssRedirect)e).Location.ToString();
                        return Update(channel);
                    } else {
                        channel.Type = "RSS";
                    }
                }
                
                reader.Close();
                res.Close();
                    
                } catch(Exception e){
                    System.Console.WriteLine("Error whilst determining feed type: {0}", e.Message);
                }
            }
            
            /* If it's still unknown, we don't speak it. */
            if(channel.Type == ""){
                return false;
            }
        }

        try {
            
            stream = GetChannelData(channel).GetResponseStream();
            if (channel.Type == "RSS") {
                updated = UpdateRssChannel(channel, stream);
            } else if (channel.Type == "Atom") {
                updated = UpdateAtomChannel(channel, stream);
            }

            if (res.Headers.Get("Last-Modified") != null) {
                channel.LastModified = res.Headers.Get("Last-Modified"); 
            }
            if (res.Headers.Get("ETag") != null) {
                channel.ETag = res.Headers.Get("ETag");
            }

            res.Close ();
        } catch (Exception e) {
            Console.WriteLine("Error whilst updating the feed {0}: {1}", channel.Name, e.Message);
        }
        
        return updated;
	}
	
    private static void set_credentials_from_uri(WebRequest req)
    {
        if (req.RequestUri.UserInfo != "") {
            int i = 0;
            string userInfo = req.RequestUri.UserInfo;
            string userName = "";

            while (i < userInfo.Length && userInfo[i] != ':')
                userName += Uri.HexUnescape(userInfo, ref i);

            if (i != userInfo.Length) {
                string password = "";

                i += 1;

                while (i < userInfo.Length && userInfo[i] != ':')
                    password += Uri.HexUnescape(userInfo, ref i);

                req.Credentials = new NetworkCredential(userName,
                                                        password);
            }
        }
    }

	public static WebResponse GetChannelData (Channel channel)
	{
	    HttpWebRequest req = null;
	    WebResponse res = null;

	    try {
	        req = (HttpWebRequest) WebRequest.Create(channel.Url);
	        
	        if(channel.http_username != ""){
                req.Credentials = new NetworkCredential(channel.http_username,
                                                        channel.http_password);
	        } else {
	            set_credentials_from_uri(req);
	        }

	        if (channel.LastModified != "") {
	            req.IfModifiedSince = DateTime.Parse(channel.LastModified);
	        }

	        if (channel.ETag != "") {
	            req.Headers.Add("If-None-Match", channel.ETag);
	        }

	        req.Timeout = 20000;
	        
	        ServicePointManager.CertificatePolicy = new CertCheck();

	        WebProxy proxy = Proxy.GetProxy();
	        if (proxy != null) {
	            req.Proxy = proxy;
	        }

	        try {
	            res = req.GetResponse();
	        } catch (WebException wE) {
	            switch (wE.Status) {
	            case WebExceptionStatus.ProtocolError:
		        if (((HttpWebResponse)wE.Response).StatusCode == 
			    HttpStatusCode.NotModified) {
		            //Console.WriteLine("No changes to feed.");
		        }
		        if(((HttpWebResponse)wE.Response).StatusCode ==
		        HttpStatusCode.Unauthorized) {
		        	System.Console.WriteLine("Unauthorised: " + wE.ToString());
		        }
		        break;
	            case WebExceptionStatus.Timeout:
		        //Console.WriteLine("Timed out");
		        //Console.WriteLine("Exception: " + wE.ToString());
		        break;
		        case WebExceptionStatus.TrustFailure:
		        System.Console.WriteLine("TrustFailure: " + wE.ToString());
		        break;
	            default:
		        Console.WriteLine("Exception: " + wE.ToString());
		        break;
	            }
	        }
	    } catch (Exception e) {
	        Console.WriteLine("Big WebRequest error: {0}", e.Message);
	    }

	    return res;
	}

	private static bool UpdateRssChannel (Channel channel, Stream stream)
	{
	    bool channelUpdated = false;

	    RssReader reader = new RssReader (stream);
	    RssChannel rssChannel = null;
	    RssElement e;

	    // FIXME: Handle multiple channels per feed?
	    do {
	        e = reader.Read();
	        if (e is RssChannel) {
	            rssChannel = (RssChannel)e;
	        }
	    } while (e != null);

	    reader.Close ();

	    if (rssChannel != null) {
	        /* Check if this channel is updated */
	        channel.StartRefresh ();

	        /*if (channel.Image.Equals ("") &&
                      rssChannel.Image != null) {
                      channel.Image = rssChannel.Image.Url.ToString ();
	        }*/

	        if ((channel.Name.Equals ("") || channel.Name.Equals(channel.Url)) &&
		    rssChannel.Title != null) {
	            channel.Name = HtmlUtils.StripHtml(rssChannel.Title);
	            channelUpdated = true;
	        }

	        foreach (RssItem item in rssChannel.Items) {
	            string id;
	            bool   itemUpdated;

	            id = GenerateItemId(item);
	            if (item.Title == null || item.Title.Equals("")) {
		        if (!item.PubDate.Equals (DateTime.MinValue)) {
		            item.Title = item.PubDate.ToString("d MMM yyyy");
		        } else {
		            item.Title = channel.Name;
		        }
	            }

	            // Will add, update or do nothing.
	            itemUpdated = channel.UpdateItem (id, item);
	            if (itemUpdated) {
		        channelUpdated = true;
	            }
	        }
	        channel.FinishRefresh ();
	    }

	    return channelUpdated;
	}

	private static bool UpdateAtomChannel(Channel channel, Stream stream)
	{
	    bool channelUpdated = false;

	    try {
	        AtomFeed feed = AtomFeed.Load(stream);

	        if (feed != null) {
	            channel.StartRefresh();

	            if ((channel.Name == "" || channel.Name == channel.Url) && 
		        feed.Title.Content != null) {
		        channel.Name = HtmlUtils.StripHtml(feed.Title.Content.Trim());
		        channelUpdated = true;
	            }

	            foreach (AtomEntry entry in feed.Entries) {
		        string id;
		        bool   entryUpdated;

		        id = GenerateItemId(entry);
		        if (entry.Title.Content == "") {
		            if (!entry.Modified.DateTime.Equals(DateTime.MinValue)) {
		                entry.Title.Content = 
		                    entry.Modified.DateTime.ToString("d MMM yyyy");
		            } else {
		                entry.Title.Content = channel.Name;
		            }
		        }

		        entryUpdated = channel.UpdateItem (id, entry);
		        if (entryUpdated) {
		            channelUpdated = true;
		        }
	            }

	            channel.FinishRefresh();
	        }
	    } catch (Exception) {
	    }

	  return channelUpdated;
	}

	public static string GenerateItemId (RssItem item)
	{
	    if (item.Guid != null) {
		return item.Guid.Name;
	    } 
	    else if (item.Link != RssDefault.Uri) {
		return item.Link.ToString ();
	    } else {
		return item.Title;
	    }
	}

	public static string GenerateItemId (AtomEntry entry)
	{
	    return entry.Id.ToString();
	}
    }
}
