//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using Gtk;
using System;
using GtkSharp;
using Mono.Posix;
using System.Collections;
using System.Reflection;
using System.IO;
using System.Net;
using System.Text;

namespace Imendio.Blam {
    public class ItemView : HTML {
	private ChannelCollection mCollection;
	private Cache mCache;
	private Imendio.Blam.Item currentItem;
	private ArrayList asyncReaders;
	private bool textReady;
	private bool emittedReady;
	 
	public bool PageLoaded;

	public delegate void ItemLoadedHandler();
	public event ItemLoadedHandler ItemLoaded;

	public Imendio.Blam.Item CurrentItem {
	    get {
		return currentItem;
	    }
	     
	    set {
		currentItem = value;
		Load ();
	    }
	}

	public ItemView(Cache cache, ChannelCollection collection)
	{
	    this.mCollection = collection;
	    this.mCache = cache;
	    Editable = false;
	    UrlRequested += new UrlRequestedHandler(UrlRequestedCb);
	    LinkClicked += new LinkClickedHandler(LinkClickedCb);
	    PageLoaded = false;
	    this.asyncReaders = new ArrayList();
	}

	private void Load() {
	    textReady = emittedReady = false;
	    HTMLStream stream = Begin ("text/html; charset=utf-8");

	    this.Title = currentItem.Title;
	    stream.Write (HtmlHeader());

	    if (currentItem.Text.Length > 0) {
		stream.Write ("<table cellpadding=\"2\"><tr><td>");
		stream.Write(EncodeUnicode(FixMarkup(currentItem.Text)));
		stream.Write ("</td></tr></table>");
	    }

	    stream.Write(HtmlFooter());
	    End(stream, HTMLStreamStatus.Ok);
	    PageLoaded = true;
	    textReady = true;
	    if (!emittedReady && asyncReaders.Count <= 0) {
		EmitItemLoaded();
	    }
	}

	private void ReadFromStream (HTMLStream handle, Stream stream)
	{
	    try {
		byte[] buffer = new byte[8192];
		int n;

		while ((n = stream.Read (buffer, 0, 8192)) != 0) {
		    handle.Write (buffer, n);
		} 
	    }
	    catch (Exception e) {
		//Console.WriteLine("Exception when reading from cache: " + e.Message);
	    }

	    handle.Close (HTMLStreamStatus.Ok);
	}

	private void UrlRequestedCb(object obj, UrlRequestedArgs args)
	{
	    Stream stream;

	    // Special case the images for the grey boxes.
	    if (args.Url == "blam-box-top-left.png" || 
		args.Url == "blam-box-top-right.png" || 
		args.Url == "blam-box-bottom-left.png" || 
		args.Url == "blam-box-bottom-right.png" ||
		args.Url == "blam-box-pad.png") {
		try {
		    stream = Assembly.GetEntryAssembly().GetManifestResourceStream (args.Url);

		    ReadFromStream (args.Handle, stream);
		}
		catch (Exception e) {
		    //Console.WriteLine("Exception when getting box image: " + e.Message);
		    return;
		}

		return;
	    }
	    
	    stream = mCache.GetCachedData(args.Url);

	    if (stream != null) {
		ReadFromStream(args.Handle, stream);
	    } else { 
		AsyncReader reader = new AsyncReader(mCache,
						     args.Handle, 
						     args.Url,
						     currentItem.Id);
		reader.Done += new AsyncReader.DoneHandler(ReaderDone);
		asyncReaders.Add (reader);

		try {
		    reader.Start();
		}
		catch(Exception e) {
		    Console.WriteLine("Couldn't complete request: " + args.Url
				      + "\n" + e.Message);
		}
	    }
	}

	private void LinkClickedCb(object obj, LinkClickedArgs args)
	{
	    try {
		Gnome.Url.Show(args.Url);
	    } 
	    catch (Exception e) {
		Console.WriteLine("Couldn't show URL: " + args.Url);
	    }
	}

	private void ReaderDone(AsyncReader reader)
	{
	    asyncReaders.Remove(reader);
	    if (textReady && asyncReaders.Count <= 0) {
		EmitItemLoaded();
	    }
	}

	private static string EncodeUnicode(string str)
	{
	    StringBuilder output = new StringBuilder();

	    foreach (char c in str) {
		if ((int)c > 128) {
		    output.Append("&#");
		    output.Append(((int)c).ToString());
		    output.Append(";");
		} else {
		    output.Append(c);
		}
	    }

	    return output.ToString();
	}

	private void EmitItemLoaded()
	{
	    emittedReady = true;

	    if (ItemLoaded != null) {
		ItemLoaded();
	    }
	}

	private string HtmlBox (string leftContent, string rightContent)
	{
	    string html = "<table bgcolor=\"#333333\" border=\"0\" width=\"100%\" cellpadding=\"0\" cellspacing=\"0\">" +
		"  <tr>" +
		"    <td bgcolor=\"#ffffff\">" +
		"      <img src=\"blam-box-top-left.png\">" +
		"    </td>" +
		"    <td colspan=2 width=\"100%\">" +
		"      <img src=\"blam-box-pad.png\">" +
		"    </td>" +
		"    <td bgcolor=\"#ffffff\">" +
		"      <img src=\"blam-box-top-right.png\">" +
		"    </td>" +
		"  </tr>" +
		"  <tr>" +
		"    <td>" +
		"      <img src=\"blam-box-pad.png\">" +
		"    </td>" +
		"    <td width=\"75%\" valign=\"top\">" +
		+ leftContent +
		"    </td>" +
		"    <td align=\"right\" valign=\"top\">" +
		+ rightContent + 
		"    </td>" +
		"    <td>" +
		"      <img src=\"blam-box-pad.png\">" +
		"    </td>" +
		"  </tr>" +
		"  <tr>" +
		"    <td bgcolor=\"#ffffff\">" +
		"      <img src=\"blam-box-bottom-left.png\">" +
		"    </td>" +
		"    <td colspan=2 width=\"100%\">" +
		"      <img src=\"blam-box-pad.png\">" +
		"    </td>" +
		"    <td bgcolor=\"#ffffff\">" +
		"      <img src=\"blam-box-bottom-right.png\">" +
		"    </td>" +
		"  </tr>" +
		"</table>";
	    
	    return html;
	}

	private string HtmlHeader ()
	{
	    string date = "";
	    string title;

	    // Don't print the date if it's not set (avoid showing 0001 Jan 01)
	    // in the item header.
	    if (!currentItem.PubDate.Equals (DateTime.MinValue)) {
		date = "<font color=\"#ffffff\" size=\"+1\">" +
		    currentItem.PubDate.ToString ("d MMM yyyy") +
		    "</font>";
	    }
	    
	    StringBuilder html = new StringBuilder ();

	    title = "<font color=\"#ffffff\" size=\"+1\">" + 
		EncodeUnicode (currentItem.Title) +
		"</font>";

	    html.Append (HtmlBox (title, date));

	    return html.ToString();
	}

	private string HtmlFooter()
	{
	    StringBuilder str = new StringBuilder();
	    string link = "";
	    string author;

	    if (currentItem.Link != null && currentItem.Link.Equals ("") != true) {
		link = String.Format("<a href=\"{0}\">" +
				     "<font color=\"#ffff00\">" +
				     Catalog.GetString ("Show in browser") +
				     "</font></a>",
				     currentItem.Link);
	    }

	    if (currentItem.Author != null && currentItem.Author != "") {
		string authorStr = String.Format (Catalog.GetString ("by {0}"), currentItem.Author);
		
		author = "<font color=\"#aaaaaa\">" + authorStr + "</font>";
	    } else {
		author = "";
	    }

	    str.Append("<br><br>");
	    str.Append(HtmlBox(link, author));

	    return str.ToString();
	}

	private string FixMarkup(string str)
	{
	    string tStr = str.Trim().Replace("</p>", "").Replace("</P>", "");

	    if (!tStr.StartsWith("<p>") && !tStr.StartsWith("<P>")) {
		tStr = "<br/><br/>" + tStr;
	    }

	    return tStr.Replace("<p>", "<br/><br/>").Replace("<P>", "<br/><br/>");
	}
    } 

    public class AsyncReader {

	public HTMLStream  htmlStream;
	public string      url;
	public string      itemId;
	public Cache       mCache;
	
	private WebRequest request;
	private Stream     responseStream;

	public static int  BUFFER_SIZE = 8192;
	public byte[]      bufferRead;
	public byte[]      data;

	public bool        done;

	public delegate void DoneHandler(AsyncReader reader);
	public event DoneHandler Done; 
	
	public AsyncReader(Cache cache, HTMLStream htmlStream, 
			   string url, string itemId)
	{
	    mCache = cache;
	    this.htmlStream = htmlStream;
	    this.url    = url;
	    this.itemId = itemId;
	    this.bufferRead = new byte[BUFFER_SIZE];
	    data = new byte[0];
	}

	public void Start()
	{
	    this.request = WebRequest.Create(url);

	    WebProxy proxy = Proxy.GetProxy ();
	    if (proxy != null) {
		this.request.Proxy = proxy;
	    }

	    IAsyncResult result = (IAsyncResult) request.BeginGetResponse(new AsyncCallback(ResponseCallback), this);

	    done = false;
	    GLib.Timeout.Add(10, new GLib.TimeoutHandler(WriteTimeout));
	}
	    
	public void Cancel()
	{
	
	}
	
	public bool WriteTimeout()
	{
	    if (done) {
		if (data.Length > 0) {
		    htmlStream.Write(data, data.Length);
		    htmlStream.Close(HTMLStreamStatus.Ok);
		} else {
		    htmlStream.Close(HTMLStreamStatus.Error);
		}
		
		if (Done != null) {
		    Done(this);
		}
		return false;
	    }

	    return true;
	}
	
	private static void ResponseCallback(IAsyncResult asyncResult)
	{
	    AsyncReader reader = (AsyncReader) asyncResult.AsyncState;
	    
	    try {
		WebResponse response = reader.request.EndGetResponse(asyncResult);
		reader.responseStream = response.GetResponseStream();
		
		IAsyncResult readResult = 
		    reader.responseStream.BeginRead(reader.bufferRead, 0, 
						    BUFFER_SIZE,
						    new AsyncCallback(ReadCallback),
						    reader);
	    }
	    catch (Exception e) {
		//Console.WriteLine("Exception in ResponseCallback: " + e.ToString());
		reader.done = true;
	    }
	}

	private static byte[] ConcatByteArrays(byte[] a1, byte[] a2, int length)
	{
	    byte[] retVal = new byte[a1.Length + length];
	 
	    Array.Copy(a1, 0, retVal, 0, a1.Length);
	    Array.Copy(a2, 0, retVal, a1.Length, length);

	    return retVal;
	}

	private static void ReadCallback(IAsyncResult asyncResult)
	{
	    AsyncReader reader = (AsyncReader) asyncResult.AsyncState;

	    try {
		Stream stream = reader.responseStream;

		int read = stream.EndRead(asyncResult);
		if (read > 0) {
		    reader.data = ConcatByteArrays(reader.data,
						   reader.bufferRead, 
						   read);

		    reader.responseStream.BeginRead(reader.bufferRead, 0,
						    BUFFER_SIZE, 
						    new AsyncCallback(ReadCallback), 
						    reader);
		} else {
		    if (reader.data.Length > 0) {
			reader.mCache.AddCacheData(reader.url, reader.itemId, reader.data);
		    }

		    stream.Close();
		    reader.done = true;
		}
	    } 
	    catch (Exception e) {
		//Console.WriteLine("Exception in ReadCallback: " + e.ToString());
		reader.data = new byte[0];
		reader.done = true;
	    }
	}
    }    
}
