
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include "glow.image.h"

// COLOR TRIPLET

int Triplet::R(int triplet) {
  return((triplet>>16)&0xff);
}

int Triplet::G(int triplet) {
  return((triplet>>8)&0xff);
}

int Triplet::B(int triplet) {
  return(triplet&0xff);
}

int Triplet::value(int r,int g,int b) {
  int v=((r&0xff)<<16)|((g&0xff)<<8)|(b&0xff);
  return(v);
}

int Triplet::safeValue(int r,int g,int b) {
  int v;
  if (r>255) r=255;
  if (g>255) g=255;
  if (b>255) b=255;
  if (r<0) r=0;
  if (g<0) g=0;
  if (b<0) b=0;
  v=((r&0xff)<<16)|((g&0xff)<<8)|(b&0xff);
  return(v);
}

int Triplet::RGB2HSV(int triplet) {
  
  float r = (float)Triplet::R(triplet), 
        g = (float)Triplet::G(triplet), 
        b = (float)Triplet::B(triplet), v, x, f; 
  float a[3];

  int i; 

  r/=255.0;
  g/=255.0;
  b/=255.0;

  // RGB are each on [0, 1]. S and V are returned on [0, 1] and H is 
  // returned on [0, 6]. 

  x = GMIN(r, g, b); 
  v = GMAX(r, g, b); 
  if (v == x) {
    a[0]=0.0;
    a[1]=0.0;
    a[2]=v;
  } else {
    f = (r == x) ? g - b : ((g == x) ? b - r : r - g); 
    i = (r == x) ? 3 : ((g == x) ? 5 : 1);
    a[0]=((float)i)-f/(v-x);
    a[1]=(v-x)/v;
    a[2]=v;
  }

  // (un)normalize

  a[0]*=255.0/6.0;
  a[1]*=255.0;
  a[2]*=255.0;
  
  return(Triplet::value((int)a[0],(int)a[1],(int)a[2]));
}

int Triplet::HSV2RGB(int triplet) {

  // H is given on [0, 6]. S and V are given on [0, 1]. 
  // RGB are each returned on [0, 1]. 
  float h = (float)Triplet::H(triplet), 
        s = (float)Triplet::S(triplet), 
        v = (float)Triplet::V(triplet), m, n, f; 
  float a[3];
  int i; 

  h/=255.0/6.0;
  s/=255.0;
  v/=255.0;

  if (s==0.0) {
    a[0]=a[1]=a[2]=v;
  } else {
    i = (int) floor(h); 
    f = h - (float)i; 
    if(!(i & 1)) f = 1 - f; // if i is even 
    m = v * (1 - s); 
    n = v * (1 - s * f); 
    switch (i) { 
    case 6: 
    case 0: a[0]=v; a[1]=n; a[2]=m; break;
    case 1: a[0]=n; a[1]=v; a[2]=m; break;
    case 2: a[0]=m; a[1]=v; a[2]=n; break;
    case 3: a[0]=m; a[1]=n; a[2]=v; break;
    case 4: a[0]=n; a[1]=m; a[2]=v; break;
    case 5: a[0]=v; a[1]=m; a[2]=n; break;
    } 
  }

  // (un)normalize
  for(i=0;i<3;i++)
    a[i]*=255;

  return(Triplet::value((int)a[0],(int)a[1],(int)a[2]));
}

int Triplet::RGB2YCbCr(int triplet) {
  float y,cb,cr;

  y=0.257*(float)Triplet::R(triplet)+
    0.504*(float)Triplet::G(triplet)+
    0.098*(float)Triplet::B(triplet)+16.0;

  cb=-0.148*(float)Triplet::R(triplet)-
      0.291*(float)Triplet::G(triplet)+
      0.439*(float)Triplet::B(triplet)+
      128.0;

  cr=0.439*(float)Triplet::R(triplet)-
     0.368*(float)Triplet::G(triplet)-
     0.071*(float)Triplet::B(triplet)+
     128.0;

  return(Triplet::value((int)y,(int)cb,(int)cr));
}

int Triplet::YCbCr2RGB(int triplet) {
  float r,g,b;

  r=1.164*((float)Triplet::Y(triplet)-16.0)+
    1.596*((float)Triplet::Cr(triplet)-128.0);

  g=1.164*((float)Triplet::Y(triplet)-16.0)-
    0.813*((float)Triplet::Cr(triplet)-128.0)-
    0.392*((float)Triplet::Cb(triplet)-128.0);
  
  b=1.164*((float)Triplet::Y(triplet)-16.0)+
    2.017*((float)Triplet::Cb(triplet)-128.0);

  return(Triplet::safeValue((int)r,(int)g,(int)b));
}

// 8-BIT PLANE

BitPlane::BitPlane(int width,int height) {
  int i;

  Width=width;
  Height=height;

  data=(unsigned char *)malloc(width*height*sizeof(char));
  rowoffset=(int *)malloc(height*sizeof(int));

  if ((data==NULL)||(rowoffset==NULL)) {
    cerr << "**[pid " << (int)getpid() << 
            "]** class BitPlane: unable to allocate memory, exited (11)\n\n";
    exit(11);
  }

  for(i=0;i<Height;i++)
    rowoffset[i]=Width*i;

  fillRow(0,0);
  for(i=1;i<Height;i++)
    copyRow(i,0);
}

BitPlane::~BitPlane() {
  free(data);
  free(rowoffset);
}

int inline BitPlane::getPixel(int x,int y) {
  return( (int) (data[rowoffset[y]+x]) );
}

void inline BitPlane::setPixel(int x,int y,int value) {
  data[rowoffset[y]+x]=(unsigned char)value;
}

void BitPlane::fillRow(int y,int value) {
  memset(data+rowoffset[y],(unsigned char)value,Width);
}

void BitPlane::copyRow(int desty,int srcy) {
  memcpy(data+rowoffset[desty],data+rowoffset[srcy],Width);
}

unsigned char *BitPlane::getBase() {
  return(data);
}

// rgb image


Image::Image(int width,int height) {
  Width=width;
  Height=height;

  red=new BitPlane(Width,Height);
  green=new BitPlane(Width,Height);
  blue=new BitPlane(Width,Height);
  
  SavePlanes=0;
}

Image::Image(char *filename) {
  FILE *f;

  SavePlanes=0;
  red=green=blue=NULL;

  if (filename==NULL)
    f=stdin;
  else
    f=fopen(filename,"r");
  if (!f) {
    cerr << "** Image::Image(char *) : unable to open " << filename << endl;
    exit(1);
  }
  iread(f);
  fclose(f);
}

Image::~Image() {
  if (red) delete red;
  if (green) delete green;
  if (blue) delete blue;
}

int inline Image::getPixel(int x,int y) {
  return(Triplet::value(red->getPixel(x,y),
			green->getPixel(x,y),
			blue->getPixel(x,y)));
}

void inline Image::setPixel(int x,int y,int value) {
  red->setPixel(x,y,Triplet::R(value));
  green->setPixel(x,y,Triplet::G(value));
  blue->setPixel(x,y,Triplet::B(value));
}


void Image::fillRow(int y,int value) {
  red->fillRow(y,Triplet::R(value));
  green->fillRow(y,Triplet::G(value));
  blue->fillRow(y,Triplet::B(value));
}

void Image::copyRow(int desty,int srcy) {
  red->copyRow(desty,srcy);
  green->copyRow(desty,srcy);
  blue->copyRow(desty,srcy);
}

int Image::saveP6(char *filename) {
  FILE *f,*fh=NULL,*fs=NULL,*fv=NULL;
  int s;
  unsigned char *p[3];

  if (filename==NULL)
    f=stdout;
  else
    f=fopen(filename,"w");
  if (!f)
    return -1;

  if (SavePlanes) {
    fh=fopen("R.ppm","w");
    fs=fopen("G.ppm","w");
    fv=fopen("B.ppm","w");    
    if ((!fh)||(!fv)||(!fs))
      return -1;
  }
  
  fprintf(f,"P6\n%d %d\n%d\n",Width,Height,255);

  if (SavePlanes) {
    fprintf(fh,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fs,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fv,"P5\n%d %d\n%d\n",Width,Height,255);
  }

  s=Width*Height;
  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  for(;s;s--,p[0]++,p[1]++,p[2]++) {
    fputc(p[0][0],f);
    fputc(p[1][0],f);
    fputc(p[2][0],f);
    if (SavePlanes) { 
      fputc(p[0][0],fh);
      fputc(p[1][0],fs);
      fputc(p[2][0],fv);
    }
  }

  if (SavePlanes) {
    fclose(fh);
    fclose(fs);
    fclose(fv);
  }

  fclose(f);
  return 0;
}

int Image::saveHSV(char *filename) {
  FILE *f,*fh=NULL,*fs=NULL,*fv=NULL;
  int s;
  unsigned char *p[3];
  int triple;

  if (filename==NULL)
    f=stdout;
  else
    f=fopen(filename,"w");
  if (!f)
    return -1;

  if (SavePlanes) {
    fh=fopen("hue.ppm","w");
    fs=fopen("sat.ppm","w");
    fv=fopen("val.ppm","w");    
    if ((!fh)||(!fv)||(!fs))
      return -1;
  }
  
  fprintf(f,"HSV\n%d %d\n%d\n",Width,Height,255);

  if (SavePlanes) {
    fprintf(fh,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fs,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fv,"P5\n%d %d\n%d\n",Width,Height,255);
  }

  s=Width*Height;
  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  for(;s;s--,p[0]++,p[1]++,p[2]++) {
    triple=Triplet::RGB2HSV(Triplet::value(p[0][0],p[1][0],p[2][0]));
    fputc(Triplet::H(triple),f);
    if (SavePlanes) fputc(Triplet::H(triple),fh);
    fputc(Triplet::S(triple),f);
    if (SavePlanes) fputc(Triplet::S(triple),fs);
    fputc(Triplet::V(triple),f);
    if (SavePlanes) fputc(Triplet::V(triple),fv);
  }

  fclose(f);

  if (SavePlanes) {
    fclose(fh);
    fclose(fs);
    fclose(fv);
  }

  return 0;
}

int Image::saveYCbCr(char *filename) {
  FILE *f,*fh=NULL,*fs=NULL,*fv=NULL;
  int s;
  unsigned char *p[3];
  int triple;

  if (filename==NULL)
    f=stdout;
  else
    f=fopen(filename,"w");
  if (!f)
    return -1;

  if (SavePlanes) {
    fh=fopen("Y.ppm","w");
    fs=fopen("Cb.ppm","w");
    fv=fopen("Cr.ppm","w");    
    if ((!fh)||(!fv)||(!fs))
      return -1;
  }
  
  fprintf(f,"YCBCR\n%d %d\n%d\n",Width,Height,255);

  if (SavePlanes) {
    fprintf(fh,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fs,"P5\n%d %d\n%d\n",Width,Height,255);
    fprintf(fv,"P5\n%d %d\n%d\n",Width,Height,255);
  }

  s=Width*Height;
  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  for(;s;s--,p[0]++,p[1]++,p[2]++) {
    triple=Triplet::RGB2YCbCr(Triplet::value(p[0][0],p[1][0],p[2][0]));
    fputc(Triplet::Y(triple),f);
    if (SavePlanes) fputc(Triplet::Y(triple),fh);
    fputc(Triplet::Cb(triple),f);
    if (SavePlanes) fputc(Triplet::Cb(triple),fs);
    fputc(Triplet::Cr(triple),f);
    if (SavePlanes) fputc(Triplet::Cr(triple),fv);
  }

  fclose(f);

  if (SavePlanes) {
    fclose(fh);
    fclose(fs);
    fclose(fv);
  }

  return 0;
}

void Image::iread(FILE *f) {
  char first[256];

  fgets(first,255,f);

  readHeader(f);
  
  if (!strcmp(first,"P6\n"))    { readP6(f);    return;   }
  if (!strcmp(first,"HSV\n"))   { readHSV(f);   return;   }
  if (!strcmp(first,"YCBCR\n")) { readYCbCr(f); return;   }
 
  cerr << "** Image::iread(FILE *) : file format not recognized" << endl;
}

void Image::readYCbCr(FILE *f) {
  unsigned char *p[3];
  int tri;
  int i,s;

  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  s=Width*Height;
  for(i=0;i<s;i++,p[0]++,p[1]++,p[2]++) {
    tri=Triplet::value(fgetc(f),fgetc(f),fgetc(f));
    tri=Triplet::YCbCr2RGB(tri);
    p[0][0]=(unsigned char)Triplet::R(tri);
    p[1][0]=(unsigned char)Triplet::G(tri);
    p[2][0]=(unsigned char)Triplet::B(tri);
  }
  fclose(f);
}

void Image::readP6(FILE *f) {
  unsigned char *p[3];
  int i,s;

  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  s=Width*Height;
  for(i=0;i<s;i++,p[0]++,p[1]++,p[2]++) {
    p[0][0]=(unsigned char)fgetc(f);
    p[1][0]=(unsigned char)fgetc(f);
    p[2][0]=(unsigned char)fgetc(f);
  }
  fclose(f);
}

void Image::readHSV(FILE *f) {
  unsigned char *p[3];
  int tri;
  int i,s;

  p[0]=red->getBase();
  p[1]=green->getBase();
  p[2]=blue->getBase();

  s=Width*Height;
  for(i=0;i<s;i++,p[0]++,p[1]++,p[2]++) {
    tri=Triplet::value(fgetc(f),fgetc(f),fgetc(f));
    tri=Triplet::HSV2RGB(tri);
    p[0][0]=(unsigned char)Triplet::R(tri);
    p[1][0]=(unsigned char)Triplet::G(tri);
    p[2][0]=(unsigned char)Triplet::B(tri);
  }
  fclose(f);
}

void Image::readHeader(FILE *f) {
  char line[256];
  fgets(line,255,f);
  Width=atoi(strtok(line," \t\n"));
  Height=atoi(strtok(NULL," \t\n"));
  fgets(line,255,f);
  red=new BitPlane(Width,Height);
  green=new BitPlane(Width,Height);
  blue=new BitPlane(Width,Height);
}

