var bm = require("../shared-modules/bookmarks");

const TIMEOUT = 5000;

const LOCAL_TEST_FOLDER = collector.addHttpResource('../test-files/');
const LOCAL_TEST_PAGE = LOCAL_TEST_FOLDER + 'test.html';
const LOCAL_TEST_FEED = LOCAL_TEST_FOLDER + 'feed.atom';

var setupModule = function(module) {
    module.controller = mozmill.getBrowserController();
    module.jum = {}
    module.sync = {}
    Cu.import("resource://mozmill/modules/jum.js", module.jum);
    Cu.import("resource://bindwood/sync.jsm", module.sync);
    module.synchroniser = null;
    bm.clearBookmarks();
};


var setupTest = function() {
    synchroniser = new sync.Synchroniser(null, "profile_name");
};

var teardownTest = function() {
    bm.clearBookmarks();
};

var test_create_bookmark = function() {
    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var bookmark_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getItemType(bookmark_id),
                     bm.bookmarksService.TYPE_BOOKMARK);
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.toolbarFolder);
    jum.assertEquals(bm.bookmarksService.getItemTitle(bookmark_id),
                     'Bookmark title');
    jum.assertEquals(bm.bookmarksService.getBookmarkURI(bookmark_id).spec,
                     LOCAL_TEST_PAGE);
};

var test_update_bookmark = function() {
    doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(doc);
    var bookmark_id = synchroniser.guid_to_id('12345');
    doc.title = 'New title';
    doc.uri = LOCAL_TEST_PAGE + "#updated";
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(bookmark_id) + 1);
    synchroniser.processRecord(doc);

    jum.assertEquals(bm.bookmarksService.getItemTitle(bookmark_id),
                     'New title');
    jum.assertEquals(bm.bookmarksService.getBookmarkURI(bookmark_id).spec,
                     LOCAL_TEST_PAGE + "#updated");
};

var test_create_folder = function() {
    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: [],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var bookmark_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getItemType(bookmark_id),
                     bm.bookmarksService.TYPE_FOLDER);
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.toolbarFolder);
    jum.assertEquals(
        bm.bookmarksService.getItemTitle(bookmark_id), 'Folder');
    // Reordering of folder's children has been scheduled.
    jum.assertNotEquals(synchroniser.folders_to_reorder['12345'], null);
};

var test_update_folder = function() {
    doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: [],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(doc);
    var bookmark_id = synchroniser.guid_to_id('12345');
    doc.title = 'Updated folder';
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(bookmark_id) + 1);
    synchroniser.processRecord(doc);

    jum.assertEquals(bm.bookmarksService.getItemTitle(bookmark_id),
                     'Updated folder');
    // Reordering of folder's children has been scheduled.
    jum.assertNotEquals(synchroniser.folders_to_reorder['12345'], null);
};

var test_create_livemark = function() {
    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/feed',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Livemark',
        site_uri: "http://www.example.com/",
        feed_uri: LOCAL_TEST_FEED,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var bookmark_id = synchroniser.guid_to_id('12345');
    jum.assertTrue(bm.livemarkService.isLivemark(bookmark_id))
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.toolbarFolder);
    jum.assertEquals(
        bm.bookmarksService.getItemTitle(bookmark_id), 'Livemark');
    jum.assertEquals(bm.livemarkService.getSiteURI(bookmark_id).spec,
                     "http://www.example.com/");
    jum.assertEquals(bm.livemarkService.getFeedURI(bookmark_id).spec,
                     LOCAL_TEST_FEED);
};

var test_update_livemark = function() {
    doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/feed',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Livemark',
        site_uri: "http://www.example.com/",
        feed_uri: LOCAL_TEST_FEED,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(doc);
    var bookmark_id = synchroniser.guid_to_id('12345');
    doc.title = 'New title';
    doc.feed_uri = LOCAL_TEST_FEED + "#updated";
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(bookmark_id) + 1);
    synchroniser.processRecord(doc);

    jum.assertEquals(bm.bookmarksService.getItemTitle(bookmark_id),
                     'New title');
    jum.assertEquals(bm.livemarkService.getFeedURI(bookmark_id).spec,
                     LOCAL_TEST_FEED + "#updated");
};

var test_create_separator = function() {
    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/separator',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var bookmark_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getItemType(bookmark_id),
                     bm.bookmarksService.TYPE_SEPARATOR);
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.toolbarFolder);
};

var test_update_separator = function() {
    doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/separator',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(doc);
    var bookmark_id = synchroniser.guid_to_id('12345');
    // No properties to change on the separator, but verify that the
    // update code path runs anyway.
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(bookmark_id) + 1);
    synchroniser.processRecord(doc);

    jum.assertEquals(bm.bookmarksService.getItemType(bookmark_id),
                     bm.bookmarksService.TYPE_SEPARATOR);
};

var test_update_item_reparent = function() {
    doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(doc);
    var bookmark_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.toolbarFolder);

    // Move to a new parent folder.
    doc.parent_guid = "menu_profile_name";
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(bookmark_id) + 1);
    synchroniser.processRecord(doc);

    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(bookmark_id),
                     bm.bookmarksService.bookmarksMenuFolder);
};

var test_create_parent_then_child = function() {
    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: ['67890'],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var parent_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(parent_id),
                     bm.bookmarksService.toolbarFolder);
    // Reordering of folder's children has been scheduled.
    jum.assertNotEquals(synchroniser.folders_to_reorder['12345'], null);

    synchroniser.processRecord({
        _id: '67890',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: '12345',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var child_id = synchroniser.guid_to_id('67890');
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(child_id),
                     parent_id);
};

var test_create_child_then_parent = function() {
    synchroniser.processRecord({
        _id: '67890',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: '12345',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var child_id = synchroniser.guid_to_id('67890');
    // The bookmark is temporarily stashed in unfiled bookmarks.
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(child_id),
                     bm.bookmarksService.unfiledBookmarksFolder);

    synchroniser.processRecord({
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: ['67890'],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var parent_id = synchroniser.guid_to_id('12345');
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(parent_id),
                     bm.bookmarksService.toolbarFolder);
    // The child has been reparented.
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(child_id),
                     parent_id);
    // Reordering of folder's children has been scheduled.
    jum.assertNotEquals(synchroniser.folders_to_reorder['12345'], null);
};

var test_reorder_children = function() {
    synchroniser.processRecord({
        _id: 'parent',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: ['child1', 'child2', 'child3', 'missing_child'],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    var parent_id = synchroniser.guid_to_id('parent');
    // Add an untracked bookmark to the folder.
    var untracked_child = bm.bookmarksService.insertBookmark(
        parent_id, bm.createURI(LOCAL_TEST_PAGE),
        bm.bookmarksService.DEFAULT_INDEX,
        "Untracked bookmark");

    // Add expected children.
    synchroniser.processRecord({
        _id: 'child1',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'parent',
        title: 'Child 1',
        uri: LOCAL_TEST_PAGE + "#child1",
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    synchroniser.processRecord({
        _id: 'child3',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'parent',
        title: 'Child 3',
        uri: LOCAL_TEST_PAGE + "#child1",
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });
    synchroniser.processRecord({
        _id: 'child2',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'parent',
        title: 'Child 2',
        uri: LOCAL_TEST_PAGE + "#child1",
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    });

    // The parent has been scheduled for reordering, so trigger
    // reordering.
    jum.assertNotEquals(synchroniser.folders_to_reorder['parent'], null);
    synchroniser.reorder_children();

    // Children should now be in the requested order, with the
    // untracked child at the end.
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 0),
                     synchroniser.guid_to_id('child1'));
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 1),
                     synchroniser.guid_to_id('child2'));
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 2),
                     synchroniser.guid_to_id('child3'));
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 3),
                     untracked_child);
};

var test_reorder_after_move = function() {
    // Create parent folder and children.
    parent_doc = {
        _id: 'parent',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        parent_guid: 'toolbar_profile_name',
        title: 'Folder',
        children: ['child1', 'child2', 'child3'],
        application_annotations: {
            Firefox: {
                profile: "profile_name"
            }
        }
    };
    synchroniser.processRecord(parent_doc);
    var parent_id = synchroniser.guid_to_id('parent');

    for each (var child_name in ['child1' ,'child2', 'child3']) {
        synchroniser.processRecord({
            _id: child_name,
            record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
            parent_guid: 'parent',
            title: child_name,
            uri: LOCAL_TEST_PAGE + '#' + child_name,
        });
    }
    synchroniser.reorder_children();

    // Move one of the children to a different folder.
    var child2_id = synchroniser.guid_to_id('child2');
    bm.bookmarksService.moveItem(
        child2_id, bm.bookmarksService.bookmarksMenuFolder,
        bm.bookmarksService.DEFAULT_INDEX);

    // Now impose a new ordering to the parent folder, including a
    // position for the moved item.
    parent_doc.children = ['child2', 'child3', 'child1'];
    synchroniser.processRecord(parent_doc);
    synchroniser.reorder_children();

    // Children should now be in the requested order, with the
    // untracked child at the end.
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 0),
                     synchroniser.guid_to_id('child3'));
    jum.assertEquals(bm.bookmarksService.getIdForItemAt(parent_id, 1),
                     synchroniser.guid_to_id('child1'));

    // The moved bookmark should still be in its new parent folder.
    jum.assertEquals(bm.bookmarksService.getFolderIdForItem(child2_id),
                     bm.bookmarksService.bookmarksMenuFolder);
};

var test_update_after_remove = function() {
    // Create a bookmark.
    var bookmark_doc = {
        _id: '12345',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    synchroniser.processRecord(bookmark_doc);

    // Remove the bookmark (as the user might do).
    var bookmark_id = synchroniser.guid_to_id(bookmark_doc._id);
    bm.bookmarksService.removeItem(bookmark_id);

    // Process another change for the record from CouchDB.
    synchroniser.processRecord(bookmark_doc);

    // The change should have been ignored, and no new bookmark should
    // have been created.
    bookmark_id = synchroniser.guid_to_id(bookmark_doc._id);
    if (bookmark_id) {
        // Check to see whether the bookmark ID really exists.
        try {
            bm.bookmarksService.getItemType(bookmark_id)
        } catch (e) {
            bookmark_id = null;
        }
    }
    jum.assertNull(bookmark_id);
};
