var bm = require("../shared-modules/bookmarks");

const TIMEOUT = 5000;

const RECORD_PREFIX = (
    "http://www.freedesktop.org/wiki/Specifications/desktopcouch/");
const TYPE_BOOKMARK = RECORD_PREFIX + "bookmark";
const TYPE_FOLDER = RECORD_PREFIX + "folder";
const TYPE_FEED = RECORD_PREFIX + "feed";
const TYPE_SEPARATOR = RECORD_PREFIX + "separator";


var setupModule = function(module) {
    module.controller = mozmill.getBrowserController();
    module.jum = {};
    module.desktopcouch = {};
    module.sync = {};
    module.migration = {};
    Cu.import("resource://mozmill/modules/jum.js", module.jum);
    Cu.import("resource://bindwood/desktopcouch.jsm", module.desktopcouch);
    Cu.import("resource://bindwood/sync.jsm", module.sync);
    Cu.import("resource://bindwood/migration.jsm", module.migration);
    module.couch = null;
    module.synchroniser = null;
    bm.clearBookmarks();
};


var setupTest = function(test) {
    var done = false;
    desktopcouch.connect_desktopcouch("test_bookmarks", function(db) {
            couch = db;
            done = true;
        }, function (message) {});
    controller.waitFor(
        function() { return done; }, "Could not connect to CouchDB", TIMEOUT);
    jum.assertNotEquals(couch, null);

    try {
        couch.createDb();
    } catch (e) {
        if (e.error != 'file_exists')
            throw(e);
    }
    synchroniser = new sync.Synchroniser(couch, "profile_name");
    synchroniser.ensureCouchViews();
};


var teardownTest = function(test) {
    synchroniser.uninit();
    couch.deleteDb();
    bm.clearBookmarks();
};


var test_get_schema_version = function() {
    var sm = new migration.SchemaMigration(couch, 'profile_name');
    // With no root document, the database must be from 0.4.2.
    jum.assertEquals(sm.get_schema_version(), 0);

    // Create a 1.0.x style root document.
    var root_doc = {
        _id: 'root_profile_name',
        children: ['old_toolbar', 'old_menu', 'old_unfiled'],
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1
            }
        }
    };
    couch.save(root_doc);
    jum.assertEquals(sm.get_schema_version(), 1);

    // Now modify it to match this version.
    root_doc.record_type = TYPE_FOLDER;
    root_doc.record_type_version = 2;
    root_doc.children = [
        'toolbar_profile_name', 'menu_profile_name', 'unfiled_profile_name'];
    couch.save(root_doc);
    jum.assertEquals(sm.get_schema_version(), 2);

    // Future schema versions will update record_type_version.
    root_doc.record_type_version = 42;
    couch.save(root_doc);
    jum.assertEquals(sm.get_schema_version(), 42);
};


// Build up bookmarks that match the sample data below.
var make_sample_bookmarks = function() {
    var folder1_id = bm.bookmarksService.createFolder(
        bm.bookmarksService.toolbarFolder, 'Folder 1',
        bm.bookmarksService.DEFAULT_INDEX);
    bm.setGuid(folder1_id, 'folder1');

    var livemark1_id = bm.livemarkService.createLivemark(
        folder1_id, 'Livemark 1',
        bm.createURI('http://www.example.com/'),
        bm.createURI('http://www.example.com/feed'),
        bm.bookmarksService.DEFAULT_INDEX);
    bm.setGuid(livemark1_id, 'livemark1');

    var bookmark1_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.toolbarFolder,
        bm.createURI('http://www.example.com/bookmark1'),
        bm.bookmarksService.DEFAULT_INDEX, 'Bookmark 1');
    bm.setGuid(bookmark1_id, 'bookmark1');

    var bookmark2_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.bookmarksMenuFolder,
        bm.createURI('http://www.example.com/bookmark2'),
        bm.bookmarksService.DEFAULT_INDEX, 'Bookmark 2');
    bm.setGuid(bookmark2_id, 'bookmark2');

    var separator1_id = bm.bookmarksService.insertSeparator(
        bm.bookmarksService.bookmarksMenuFolder,
        bm.bookmarksService.DEFAULT_INDEX);
    bm.setGuid(separator1_id, 'separator1');

    var bookmark3_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.unfiledBookmarksFolder,
        bm.createURI('http://www.example.com/bookmark3'),
        bm.bookmarksService.DEFAULT_INDEX, 'Bookmark 3');
    bm.setGuid(bookmark3_id, 'bookmark3');
};


// Make Version 0 (Bindwood 0.4.2) sample data
var make_sample_data_v0 = function() {
    couch.bulkSave([
        {
            _id: 'couchdb_folder1',
            record_type: TYPE_FOLDER,
            title: 'Folder 1',
            application_annotations: {
                uuid: 'folder1',
                folder: 'toolbarFolder',
                profile: 'profile_name'
            }
        },
        {
            _id: 'couchdb_livemark1',
            record_type: TYPE_FOLDER,
            title: 'Livemark 1',
            application_annotations: {
                uuid: 'livemark1',
                folder: 'Folder 1',
                profile: 'profile_name'
            }
        },
        {
            _id: 'couchdb_bookmark1',
            record_type: TYPE_BOOKMARK,
            title: 'Bookmark 1',
            uri: 'http://www.example.com/bookmark1',
            application_annotations: {
                uuid: 'bookmark1',
                folder: 'toolbarFolder',
                profile: 'profile_name'
            }
        },
        {
            _id: 'couchdb_bookmark2',
            record_type: TYPE_BOOKMARK,
            title: 'Bookmark 2',
            uri: 'http://www.example.com/bookmark2',
            application_annotations: {
                uuid: 'bookmark2',
                folder: 'bookmarksMenuFolder',
                profile: 'profile_name'
            }
        },
        {
            _id: 'couchdb_separator1',
            record_type: TYPE_SEPARATOR,
            title: null,
            application_annotations: {
                uuid: 'separator1',
                folder: 'bookmarksMenuFolder',
                profile: 'profile_name'
            }
        },
        {
            _id: 'couchdb_bookmark3',
            record_type: TYPE_BOOKMARK,
            title: 'Bookmark 3',
            uri: 'http://www.example.com/bookmark3',
            application_annotations: {
                uuid: 'bookmark3',
                folder: 'unfiledBookmarksFolder',
                profile: 'profile_name'
            }
        }
    ]);
};


// Make Version 1 (Bindwood 1.0.x) sample data
var make_sample_data_v1 = function() {
    var common_annotations = {
        Firefox: {
            profile: 'profile_name',
            last_modified: 1,
            garbage: ''
        }
    };
    couch.bulkSave([
        {
            _id: 'root_profile_name',
            children: ['obsolete_toolbar', 'obsolete_menu', 'obsolete_unfiled'],
            application_annotations: common_annotations
        },
        {
            _id: 'obsolete_toolbar',
            record_type: TYPE_FOLDER,
            record_type_version: 1,
            title: 'Bookmarks Toolbar',
            children: ['folder1', 'bookmark1'],
            application_annotations: common_annotations
        },
        {
            _id: 'folder1',
            record_type: TYPE_FOLDER,
            record_type_version: 1,
            title: 'Folder 1',
            children: ['livemark1'],
            application_annotations: common_annotations
        },
        {
            _id: 'livemark1',
            record_type: TYPE_FEED,
            record_type_version: 1,
            title: 'Livemark 1',
            feed_uri: 'http://www.example.com/feed',
            site_uri: 'http://www.example.com/',
            application_annotations: common_annotations
        },
        {
            _id: 'bookmark1',
            record_type: TYPE_BOOKMARK,
            record_type_version: 1,
            title: 'Bookmark 1',
            uri: 'http://www.example.com/bookmark1',
            application_annotations: common_annotations
        },
        {
            _id: 'obsolete_menu',
            record_type: TYPE_FOLDER,
            record_type_version: 1,
            title: 'Bookmarks Menu',
            children: ['bookmark2', 'separator1'],
            application_annotations: common_annotations
        },
        {
            _id: 'bookmark2',
            record_type: TYPE_BOOKMARK,
            record_type_version: 1,
            title: 'Bookmark 2',
            uri: 'http://www.example.com/bookmark2',
            application_annotations: common_annotations
        },
        {
            _id: 'separator1',
            record_type: TYPE_SEPARATOR,
            record_type_version: 1,
            application_annotations: common_annotations
        },
        {
            _id: 'obsolete_unfiled',
            record_type: TYPE_FOLDER,
            record_type_version: 1,
            title: 'Unfiled Bookmarks',
            children: ['bookmark3'],
            application_annotations: common_annotations
        },
        {
            _id: 'bookmark3',
            record_type: TYPE_BOOKMARK,
            record_type_version: 1,
            title: 'Bookmark 3',
            uri: 'http://www.example.com/bookmark3',
            application_annotations: common_annotations
        },
        {
            _id: 'orphaned_bookmark',
            record_type: TYPE_BOOKMARK,
            record_type_version: 1,
            title: 'Orphaned Bookmark',
            uri: 'http://www.example.com/orphan',
            application_annotations: common_annotations
        }
    ]);
};

// Verify that the database contains expected v2 sample data.
var verify_sample_data_v2 = function() {
    var root = couch.open('root_profile_name');
    jum.assertNotNull(root);
    jum.assertEquals(root.record_type, TYPE_FOLDER);
    jum.assertEquals(root.record_type_version, 2);
    jum.assertEquals(root.children.length, 3);
    jum.assertEquals(root.children[0], 'toolbar_profile_name');
    jum.assertEquals(root.children[1], 'menu_profile_name');
    jum.assertEquals(root.children[2], 'unfiled_profile_name');

    var toolbar = couch.open('toolbar_profile_name');
    jum.assertNotNull(toolbar);
    jum.assertEquals(toolbar.record_type, TYPE_FOLDER);
    jum.assertEquals(toolbar.record_type_version, 2);
    jum.assertEquals(toolbar.parent_guid, 'root_profile_name');
    jum.assertEquals(toolbar.children.length, 2);
    jum.assertEquals(toolbar.children[0], 'folder1');
    jum.assertEquals(toolbar.children[1], 'bookmark1');

    var folder1 = couch.open('folder1');
    jum.assertNotNull(folder1);
    jum.assertEquals(folder1.record_type, TYPE_FOLDER);
    jum.assertEquals(folder1.record_type_version, 2);
    jum.assertEquals(folder1.parent_guid, 'toolbar_profile_name');
    jum.assertEquals(folder1.title, 'Folder 1');
    jum.assertEquals(folder1.children.length, 1);
    jum.assertEquals(folder1.children[0], 'livemark1');

    var livemark1 = couch.open('livemark1');
    jum.assertNotNull(livemark1);
    jum.assertEquals(livemark1.record_type, TYPE_FEED);
    jum.assertEquals(livemark1.record_type_version, 2);
    jum.assertEquals(livemark1.parent_guid, 'folder1');
    jum.assertEquals(livemark1.title, 'Livemark 1');
    jum.assertEquals(livemark1.site_uri, 'http://www.example.com/')
    jum.assertEquals(livemark1.feed_uri, 'http://www.example.com/feed')

    var bookmark1 = couch.open('bookmark1');
    jum.assertNotNull(bookmark1);
    jum.assertEquals(bookmark1.record_type, TYPE_BOOKMARK);
    jum.assertEquals(bookmark1.record_type_version, 2);
    jum.assertEquals(bookmark1.parent_guid, 'toolbar_profile_name');
    jum.assertEquals(bookmark1.title, 'Bookmark 1');
    jum.assertEquals(bookmark1.uri, 'http://www.example.com/bookmark1');

    var menu = couch.open('menu_profile_name');
    jum.assertNotNull(menu);
    jum.assertEquals(menu.record_type, TYPE_FOLDER);
    jum.assertEquals(menu.record_type_version, 2);
    jum.assertEquals(menu.parent_guid, 'root_profile_name');
    jum.assertEquals(menu.children.length, 2);
    jum.assertEquals(menu.children[0], 'bookmark2');
    jum.assertEquals(menu.children[1], 'separator1');

    var bookmark2 = couch.open('bookmark2');
    jum.assertNotNull(bookmark2);
    jum.assertEquals(bookmark2.record_type, TYPE_BOOKMARK);
    jum.assertEquals(bookmark2.record_type_version, 2);
    jum.assertEquals(bookmark2.parent_guid, 'menu_profile_name');
    jum.assertEquals(bookmark2.title, 'Bookmark 2');
    jum.assertEquals(bookmark2.uri, 'http://www.example.com/bookmark2');

    var separator1 = couch.open('separator1');
    jum.assertNotNull(separator1);
    jum.assertEquals(separator1.record_type, TYPE_SEPARATOR);
    jum.assertEquals(separator1.record_type_version, 2);
    jum.assertEquals(separator1.parent_guid, 'menu_profile_name');

    var unfiled = couch.open('unfiled_profile_name');
    jum.assertNotNull(unfiled);
    jum.assertEquals(unfiled.record_type, TYPE_FOLDER);
    jum.assertEquals(unfiled.record_type_version, 2);
    jum.assertEquals(unfiled.parent_guid, 'root_profile_name');
    jum.assertEquals(unfiled.children.length, 1);
    jum.assertEquals(unfiled.children[0], 'bookmark3');

    var bookmark3 = couch.open('bookmark3');
    jum.assertNotNull(bookmark3);
    jum.assertEquals(bookmark3.record_type, TYPE_BOOKMARK);
    jum.assertEquals(bookmark3.record_type_version, 2);
    jum.assertEquals(bookmark3.parent_guid, 'unfiled_profile_name');
    jum.assertEquals(bookmark3.title, 'Bookmark 3');
    jum.assertEquals(bookmark3.uri, 'http://www.example.com/bookmark3');
};


var test_upgrade_1_to_2 = function() {
    make_sample_data_v1();
    var sm = new migration.SchemaMigration(couch, 'profile_name');
    sm.upgrade_1_to_2();

    verify_sample_data_v2();

    // The old documents for the special folders have been removed.
    jum.assertNull(couch.open('obsolete_toolbar'));
    jum.assertNull(couch.open('obsolete_menu'));
    jum.assertNull(couch.open('obsolete_unfiled'));

    // Unneeded attributes removed from annotation.
    var bookmark1_doc = couch.open('bookmark1');
    jum.assertUndefined(bookmark1_doc.application_annotations.Firefox.garbage);

    // Orphaned items have been deleted.
    jum.assertNull(couch.open('orphaned_bookmark'));
};


var test_migrate_v0 = function() {
    make_sample_bookmarks();
    make_sample_data_v0();

    synchroniser.last_seq = couch.info().last_seq;
    synchroniser.init();
    synchroniser.sync();

    verify_sample_data_v2();
};

var test_migrate_v1 = function() {
    make_sample_bookmarks();
    make_sample_data_v1();

    synchroniser.last_seq = couch.info().last_seq;
    synchroniser.init();
    synchroniser.sync();

    verify_sample_data_v2();
};
