/*
 * Copyright 2009-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

const EXPORTED_SYMBOLS = ['SchemaMigration'];

const Cu = Components.utils;

Cu.import("resource://bindwood/logging.jsm");


function SchemaMigration(couch, profile) {
    this.couch = couch;
    this.profile = profile;
}

SchemaMigration.prototype = {

    get_schema_version: function() {
        var doc = this.couch.open('root_' + this.profile);

        if (!doc) {
            // If there is no root document, we're dealing with the
            // original Bindwood datbase schema.
            return 0;
        }
        // If the root document does not have a record_type_version,
        // assume it is the second schema version.
        if (!doc.record_type_version) {
            return 1;
        }
        // Otherwise, use the value of record_type_version.
        return doc.record_type_version;
    },

    upgrade: function() {
        var schema_version = this.get_schema_version();
        Log.debug("Current schema version is " + schema_version);
        switch (schema_version) {
        case 0:
            // Haven't written migration code for this format, so just
            // wipe out the bookmarks and assume that we can rebuild
            // from the local places database.
            this.wipe_bookmarks();
            return true;
        case 1:
            this.upgrade_1_to_2();
            return true;
        case 2:
            // Nothing to be done: we're at the most recent version.
            return false;
        default:
            Log.error("Unknown schema version " + schema_version);
            return false;
        }
    },

    wipe_bookmarks: function() {
        Log.debug("Wiping all bookmarks for profile " + this.profile);
        var result = this.couch.view("bindwood/bookmarks", {
            key: this.profile,
            include_docs: true,
        });
        var changes = [];
        for each (var row in result.rows) {
            var doc = row.doc;
            changes.push({
                _id: doc._id,
                _rev: doc._rev,
                _deleted: true
            });
        }
        // Delete all the bookmarks.
        this.couch.bulkSave(changes);
    },

    /* The format used by Bindwood 0.4.2.  Bookmarks were represented as:
     *
     * {
     *   _id: '...',
     *   record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
     *   title: 'Bookmark title',
     *   uri: 'http://www.example.com/',
     *   application_annotations: {
     *     Firefox: {
     *       uuid: '...', // This is the UUID associated with the local item.
     *       folder: 'Title of parent folder',
     *       profile: 'profile name'
     *     }
     *   }
     * }
     *
     * Folders and separators had the appropriate record types, and
     * had no URI property.  Folder child ordering is not stored
     * anywhere.
     *
     * There is no special handling of livemarks.  They are stored as
     * folders, and their children are also stored.
     *
     * The special parent names "toolbarFolder", "bookmarksMenuFolder"
     * and "unfiledBookmarksFolder" were used to represent items at
     * the top levels of each folder hierarchy.
     */

    /* The format used by Bindwood 1.0.x.  Bookmarks look like this:
     *
     * {
     *   _id: '...', // This is the UUID associated with the local item.
     *   record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
     *   record_type_version: 1,
     *   title: 'Bookmark title',
     *   uri: 'http://www.example.com/',
     *   application_annotations: {
     *     Firefox: {
     *       profile: 'profile name'
     *       last_modified: ..., // time in microseconds since epoch
     *     }
     *   }
     * }
     *
     * Folders contain a list of the IDs of their children (the
     * opposite of the 0.4.2 format), which also acts as a way to
     * store the child ordering.
     *
     * Livemarks are stored with their own record type.  They look
     * similar to normal bookmarks, but store site_uri and feed_uri
     * instead of a single bookmark URI.
     */
    upgrade_1_to_2: function() {
        Log.debug("Upgrading bookmarks for profile " + this.profile +
                  " from schema v1 to v2");
        var result = this.couch.view("bindwood/bookmarks", {
            key: this.profile,
            include_docs: true,
        });

        var changes = [];

        // Build a map of document IDs to the documents they reflect.
        var docs_by_id = {};
        for each (var row in result.rows) {
            docs_by_id[row.id] = row.doc;
        }

        // Fix up children of the root document.  The children (which
        // represent the toolbar, bookmarks menu and unfiled bookmarks
        // folders) should be renamed to fixed identifiers.
        var root_doc = docs_by_id['root_' + this.profile];
        root_doc.record_type = 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder';
        var expected_children = [
            'toolbar_' + this.profile,
            'menu_' + this.profile,
            'unfiled_' + this.profile];
        for (var i = 0; i < expected_children.length; i++) {
            var old_id = root_doc.children[i];
            var new_id = expected_children[i];
            // Existing document has expected name.
            if (old_id == new_id) {
                continue;
            }
            var doc = docs_by_id[old_id];
            delete docs_by_id[old_id];

            // Schedule deletion of old document
            changes.push({
                _id: doc._id,
                _rev: doc._rev,
                _deleted: true
            });
            doc._id = new_id;
            delete doc._rev;
            docs_by_id[new_id] = doc;
            root_doc.children[i] = new_id;
        }

        // Walk the bookmarks tree from the root node to determine
        // which documents are reachable, and build up a parent map.
        var reachable = {};
        var parents_by_id = {};

        var remaining = [root_doc];
        reachable[root_doc._id] = true;
        while (remaining.length > 0) {
            var doc = remaining.pop();
            for each (var child_id in doc.children) {
                parents_by_id[child_id] = doc;
                reachable[child_id] = true;

                var child_doc = docs_by_id[child_id];
                if (child_doc && child_doc.record_type == 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder') {
                    remaining.push(child_doc);
                }
            }
        }

        // Now update the documents.
        for each (var doc in docs_by_id) {
            // If the document can not be reached from the root, then
            // it does not belong in the document tree.
            if (!reachable[doc._id]) {
                Log.debug("Deleting bookmark " + doc._id +
                          " is not reachable from root.");
                changes.push({
                    _id: doc._id,
                    _rev: doc._rev,
                    _deleted: true
                });
                continue;
            }

            doc.record_type_version = 2;
            var parent_doc = parents_by_id[doc._id]
            if (parent_doc) {
                doc.parent_guid = parent_doc._id;
                doc.parent_title = parent_doc.title;
            }

            // Bindwood 1.0.x would store a lot of useless data in the
            // Firefox section of the document (basically anything
            // that triggered an onItemChanged callback).  Keep only
            // the information we care about.
            var old_annotations = doc.application_annotations.Firefox;
            var new_annotations = {
                profile: old_annotations.profile,
                last_modified: old_annotations.last_modified
            };
            doc.application_annotations.Firefox = new_annotations;

            changes.push(doc);
        }
        // Now push the updates back to CouchDB.
        Log.debug("Saving modifications to database.");
        this.couch.bulkSave(changes);
        Log.debug("Done.");
    }
};
