/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cwriter.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Tue Dec 17 09:44:20 1991                          */
/*    Last change :  Thu May 19 15:57:50 2005 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Object (that have to be non recursives) printing.                */
/*=====================================================================*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Les recuperations externes                                       */
/*---------------------------------------------------------------------*/
extern obj_t c_constant_string_to_string( char *c_string );
extern obj_t write_object( obj_t, obj_t );
extern obj_t write_ucs2( obj_t, obj_t );
extern obj_t display_ucs2string( obj_t, obj_t );
extern obj_t real_to_string( double );
extern bool_t symbol_case_sensitivep( obj_t );
extern obj_t create_string_for_read( obj_t, int );
extern obj_t llong_to_string( BGL_LONGLONG_T x, long radix );

/*---------------------------------------------------------------------*/
/*    Les noms des caracateres                                         */
/*---------------------------------------------------------------------*/
static char *char_name[] = {
   "","","","","","","","",
   "",  "tab", "newline", "", "", "return", "", "",
   "", "","","","","","","",
   "", "", "","","", "", "", "",
   "space", "!", "\"","#","$","%","&","'",
   "(", ")", "*", "+", ",", "-", ".", "/",
   "0", "1", "2", "3", "4", "5", "6", "7",
   "8", "9", ":", ";", "<", "=", ">", "?",
   "@", "A", "B", "C", "D", "E", "F", "G",
   "H", "I", "J", "K", "L", "M", "N", "O",
   "P", "Q", "R", "S", "T", "U", "V", "W",
   "X", "Y", "Z", "[", "\\", "]", "^", "_",
   "`", "a", "b", "c", "d", "e", "f", "g",
   "h", "i", "j", "k", "l", "m", "n", "o",
   "p", "q", "r", "s", "t", "u", "v", "w",
   "x", "y", "z", "{", "|", "}", "~", ""
};

/*---------------------------------------------------------------------*/
/*    PUTC ...                                                         */
/*---------------------------------------------------------------------*/
#define PUTC( port, c, s ) \
  ( OUTPUT_PORT( port ).sysputc( (c), (s) ) )

/*---------------------------------------------------------------------*/
/*    PUTS ...                                                         */
/*---------------------------------------------------------------------*/
#define PUTS( port, str, stream ) \
  OUTPUT_PORT( port ).syswrite( str, 1, sizeof( str ) - 1, ostream )
  
/*---------------------------------------------------------------------*/
/*    PRINTF1 ...                                                      */
/*---------------------------------------------------------------------*/
#ifdef __GNUC__
#  define PRINTF1( port, bufsize, ostream, fmt, arg0 ) \
  if( OUTPUT_PORT( port ).kindof == KINDOF_FILE ) { \
    fprintf( ostream, fmt, arg0 ); \
  } else { \
    char __buf[ bufsize ]; \
    sprintf( __buf, fmt, arg0 ); \
    OUTPUT_PORT( port ).syswrite( __buf, 1, strlen( __buf ), ostream ); \
  }
#  else
#  define PRINTF1( port, bufsize, ostream, fmt, arg0 ) \
  if( OUTPUT_PORT( port ).kindof == KINDOF_FILE ) { \
    fprintf( ostream, fmt, arg0 ); \
  } else { \
    char *__buf = alloca( bufsize ); \
    sprintf( __buf, fmt, arg0 ); \
    OUTPUT_PORT( port ).syswrite( __buf, 1, strlen( __buf ), ostream ); \
  }
#endif

/*---------------------------------------------------------------------*/
/*    PRINTF2 ...                                                      */
/*---------------------------------------------------------------------*/
#ifdef __GNUC__
#  define PRINTF2( port, bufsize, stream, fmt, arg0, arg1 ) \
  if( OUTPUT_PORT( port ).kindof == KINDOF_FILE ) { \
    fprintf( stream, fmt, arg0, arg1 ); \
  } else { \
    char __buf[ bufsize ]; \
    sprintf( __buf, fmt, arg0, arg1 ); \
    OUTPUT_PORT( port ).syswrite( __buf, 1, strlen( __buf ), ostream ); \
  }
#  else
#  define PRINTF2( port, bufsize, stream, fmt, arg0, arg1 ) \
  if( OUTPUT_PORT( port ).kindof == KINDOF_FILE ) { \
    fprintf( stream, fmt, arg0, arg1 ); \
  } else { \
    char *__buf = (char *)alloca( bufsize ); \
    sprintf( __buf, fmt, arg0, arg1 ); \
    OUTPUT_PORT( port ).syswrite( __buf, 1, strlen( __buf ), ostream ); \
  }
#endif

/*---------------------------------------------------------------------*/
/*    We catch the `escape_char_found' variable from Clib/cstring.c    */
/*---------------------------------------------------------------------*/
extern int escape_char_found;

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_string ...                                               */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
display_string( obj_t o, obj_t port ) {
   long len = STRING_LENGTH( o );
   void *ostream = OUTPUT_PORT( port ).ostream;
   unsigned char *str = &STRING_REF( o, 0 );

   OUTPUT_PORT( port ).syswrite( str, 1, len, ostream );
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_string ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_string( obj_t o, bool_t r5rs_string, obj_t port ) {
   char *str = (char *)BSTRING_TO_STRING( o );
   long len  = STRING_LENGTH( o );
   void *ostream = OUTPUT_PORT( port ).ostream;

   if( r5rs_string && escape_char_found ) PUTC( port, '#', ostream );
   
   PUTC( port, '"', ostream );
   OUTPUT_PORT( port ).syswrite( str, 1, len, ostream );
   PUTC( port, '"', ostream );

   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_symbol ...                                               */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
display_symbol( obj_t o, obj_t port ) {
   return display_string( SYMBOL( o ).string, port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_keyword ...                                              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
display_keyword( obj_t o, obj_t port ) {
   return display_string( KEYWORD( o ).string, port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_fixnum ...                                               */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
display_fixnum( obj_t o, obj_t port ) {
   PRINTF1( port, 100, OUTPUT_PORT( port ).ostream, "%ld", CINT( o ) );
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_flonum ...                                               */
/*    -------------------------------------------------------------    */
/*    Many thanks to Raj Manandhar <raj@droid.msfc.nasa.gov> for       */
/*    providing this code.                                             */
/*---------------------------------------------------------------------*/
obj_t
display_flonum( obj_t o, obj_t port ) {
   return display_string( real_to_string( REAL( o ).real ), port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_char ...                                                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
display_char( obj_t o, obj_t port ) {
   PUTC( port, CCHAR( o ), OUTPUT_PORT( port ).ostream );
      
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_char ...                                                   */
/*---------------------------------------------------------------------*/
obj_t
write_char( obj_t o, obj_t port ) {
   int c = CCHAR( o );
   void *ostream = OUTPUT_PORT( port ).ostream;
   
   if( (c > 0) && (c < 128) && char_name[ c ][ 0 ] ) {
      char *name = char_name[ c ];
	 
      PUTC( port, '#', ostream );
      PUTC( port, '\\', ostream );
      OUTPUT_PORT( port ).syswrite( name, 1, strlen( name ), ostream );
   } else {
      PUTC( port, '#', ostream );
      PUTC( port, 'a', ostream );

      PRINTF1( port, 3, ostream, "%03d", (unsigned char)(c) );
   }
   
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    write_ucs2 ...                                                   */
/*---------------------------------------------------------------------*/
obj_t
write_ucs2( obj_t o, obj_t port ) {
   PRINTF1( port, 7, OUTPUT_PORT( port ).ostream, "#u%04x", CUCS2( o ) );
   return o;
}   

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_ucs2 ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
display_ucs2( obj_t o, obj_t port ) {
   ucs2_t ch = CUCS2( o );
   
   if( UCS2_ISOLATIN1P( ch ) ) {
      display_char( (obj_t)BCHAR( ch ), port );
      return o;
   } else
      return write_ucs2( o, port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_ucs2string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
display_ucs2string( obj_t o, obj_t port ) {
   int len  = UCS2_STRING_LENGTH( o );
   ucs2_t *ucs2 = BUCS2_STRING_TO_UCS2_STRING( o );
   void *ostream = OUTPUT_PORT( port ).ostream;
   int i;
   
   if( OUTPUT_STRING_PORTP( port ) ) {
      for( i = 0; i < len; i++ ) {
	 ucs2_t ch = ucs2[ i ];
	 
#if( UCS2_DISPLAYABLE )
#else
	 if( UCS2_ISOLATIN1P( ch ) )
	    PUTC( port, (char)ch, ostream );
#endif
      }
   } else {
      for( i = 0; i < len; i++ ) {
	 ucs2_t ch = ucs2[ i ];
	 
#if( UCS2_DISPLAYABLE )
#else
	 if( UCS2_ISOLATIN1P( ch ) )
	    PUTC( port, (char)ch, ostream );
#endif
      }
   }
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    write_utf8string ...                                             */
/*---------------------------------------------------------------------*/
obj_t
write_utf8string( obj_t o, obj_t port ) {
   char *str = (char *)BSTRING_TO_STRING( o );
   int  len = STRING_LENGTH( o );
   void *ostream = OUTPUT_PORT( port ).ostream;
   
   PUTS( port, "#u\"", ostream );
   OUTPUT_PORT( port ).syswrite( str, 1, len, ostream );
   PUTC( port, '"', ostream );

   return o;
}

/*---------------------------------------------------------------------*/
/*    ill_char_rep ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
ill_char_rep( unsigned char c ) {
   char aux[ 10 ];

   sprintf( aux, "#a%03d", c );

   return c_constant_string_to_string( aux );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    write_object ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_object( obj_t o, obj_t port ) {
   void *ostream = OUTPUT_PORT( port ).ostream;
   
   if( INTEGERP( o ) )
      return display_fixnum( o, port );
   
   if( CHARP( o ) )
      return display_char( o, port );

   if( UCS2P( o ) )
      return write_ucs2( o, port );

#if defined( TAG_STRING )
   if( STRINGP( o ) )
      return display_string( o, port );
#endif  
      
#if defined( TAG_REAL )
   if( REALP( o ) )
      return display_flonum( o, port );   
#endif

   switch( (long)o ) {
      case (long)BNIL :
	 PUTS( port, "()", ostream );
	 return o;

      case (long)BUNSPEC :
	 PUTS( port, "#unspecified", ostream );
	 return o;
    
      case (long)BFALSE :
	 PUTS( port, "#f", ostream );
	 return o;
    
      case (long)BTRUE :
	 PUTS( port, "#t", ostream );
	 return o;

      case (long)BEOF :
	 PUTS( port, "#eof-object", ostream );
	 return o;

      case (long)BOPTIONAL :
	 PUTS( port, "#!optional", ostream );
	 return o;

      case (long)BREST :
	 PUTS( port, "#!rest", ostream );
	 return o;

      case (long)BKEY :
	 PUTS( port, "#!key", ostream );
	 return o;
	 
      default:
	 if( CNSTP( o ) ) {
	    PRINTF1( port, 7, ostream, "#<%04x>", (int)CCNST( o ) );
	    return o;
	 }
                
	 if( !POINTERP( o ) ) {
	    PRINTF1( port, 16, ostream, "#<???:%08lx>", (unsigned long)o );
	    return o;
	 } else {
	    switch( TYPE( o ) ) {
#if( !defined( TAG_STRING ) )
	       case STRING_TYPE:
		  return display_string( o, port );
#endif

	       case UCS2_STRING_TYPE:
		  return display_ucs2string( o, port );
		     
	       case SYMBOL_TYPE:
		  return display_symbol( o, port );
		     
	       case KEYWORD_TYPE:
		  return display_keyword( o, port );

#if( !defined( TAG_REAL ) )
	       case REAL_TYPE:
		  return display_flonum( o, port );
#endif
                        
	       case PROCEDURE_TYPE:
		  PRINTF2( port, 100, ostream,
			   "#<procedure:%lx.%ld>",
			   VA_PROCEDUREP( o ) ?
			   (unsigned long)PROCEDURE_VA_ENTRY( o ) :
			   (unsigned long)PROCEDURE_ENTRY( o ),
			   (long)PROCEDURE( o ).arity );
		  return o;
        
	       case OUTPUT_PORT_TYPE:
		  PRINTF1( port, 20 + STRING_LENGTH( OUTPUT_PORT( o ).name ),
			   ostream,
			   "#<output_port:%s>",
			   BSTRING_TO_STRING( OUTPUT_PORT( o ).name ) );
		  return o;
                  
	       case OUTPUT_STRING_PORT_TYPE:
		  PUTS( port, "#<output_string_port>", ostream );
		  return o;
                  
	       case INPUT_PORT_TYPE:
		  PUTS( port, "#<input_port:", ostream );
		  write_object( INPUT_PORT( o ).name, port );
		  PRINTF1( port,
			   10,
			   ostream,
			   ".%ld>",
			   (long)INPUT_PORT( o ).bufsiz );
		  return o;
      
	       case BINARY_PORT_TYPE : 
		  PRINTF2( port, 40 + STRING_LENGTH( INPUT_PORT( o ).name ),
			   ostream,
			   "#<binary_%s_port:%s>",
			   BINARY_PORT_INP( o ) ? "input" : "output",
			   BSTRING_TO_STRING( BINARY_PORT( o ).name ) );
		  return o;

	       case ELONG_TYPE:
		  PRINTF1( port, 100, ostream, "#e%ld", BELONG_TO_LONG( o ) );
		  return o;

	       case LLONG_TYPE:
		  PUTS( port, "#l", ostream );
		  display_string( llong_to_string( BLLONG_TO_LLONG( o ), 10 ),
				  port );
		  return o;
                  
	       case FOREIGN_TYPE :
		  PUTS( port, "#<foreign:", ostream );
		  write_object( FOREIGN_ID( o ), port );
		  PRINTF1( port, 16, ostream,
			   ":%lx>", (long)FOREIGN_TO_COBJ( o ) );
		  return o;

	       case PROCESS_TYPE:
		  PUTS( port, "#<process:", ostream );
		  PRINTF1( port, 20, ostream, "%d>", PROCESS_PID( o ) );
		  return o;

	       case SOCKET_TYPE:
		  PRINTF2( port,
			   40 + (STRINGP( SOCKET( o ).hostname ) ?
			   STRING_LENGTH( SOCKET( o ).hostname ) :
			   sizeof( "localhost" )),
			   ostream,
			   "#<socket:%s.%d>",
			   STRINGP( SOCKET( o ).hostname ) ?
			   BSTRING_TO_STRING( SOCKET( o ).hostname ) :
			   "localhost",
			   SOCKET( o ).portnum );
		  return o;

	       case CUSTOM_TYPE:
		  CUSTOM_OUTPUT( o )( o, port );
		  return o;

	       case OPAQUE_TYPE:
		  PRINTF2( port, 40, ostream, "#<opaque:%ld:%08lx>", 
			   TYPE( o ),
			   (unsigned long)o );
		  return o;

	       case MUTEX_TYPE:
		  PUTS( port, "#<mutex>", ostream );
		  return o;

	       case CONDVAR_TYPE:
		  PUTS( port, "#<condvar>", ostream );
		  return o;

	       default :
		  PRINTF2( port, 40, ostream, "#<???:%ld:%08lx>",
			   TYPE( o ), (unsigned long)o );
		  return o;
	    }
	 }
   }
}


