/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>
#include <glib.h>
#include <gio/gio.h>
#include <gst/gst.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "gst-helper.h"
#include "metadata-defines.h"

static GdkPixbuf *
get_shot (const char *uri)
{
    GstElement *playbin, *audio_sink, *video_sink;
    GstStateChangeReturn state;
    GdkPixbuf *shot = NULL;
    int count = 0;

    playbin = gst_element_factory_make ("playbin", "playbin");
    audio_sink = gst_element_factory_make ("fakesink", "audiosink");
    video_sink = gst_element_factory_make ("fakesink", "videosink");

    g_object_set (playbin,
                  "uri", uri,
                  "audio-sink", audio_sink,
                  "video-sink", video_sink,
                  NULL);
    g_object_set (video_sink,
                  "sync", TRUE,
                  NULL);
    state = gst_element_set_state (playbin, GST_STATE_PAUSED);
    while (state == GST_STATE_CHANGE_ASYNC && count < 5) {
        g_print ("Waiting in loop %d\n", count);
        state = gst_element_get_state (playbin, NULL, 0, 1 * GST_SECOND);
        count++;
    }

    if (state != GST_STATE_CHANGE_FAILURE &&
        state != GST_STATE_CHANGE_ASYNC) {
        GstFormat format = GST_FORMAT_TIME;
        gint64 duration;

        if (gst_element_query_duration (playbin, &format, &duration)) {
            gint64 seekpos;
            GstBuffer *frame;

            if (duration / (3 * GST_SECOND) > 90) {
                seekpos = (rand () % (duration / (3 * GST_SECOND))) * GST_SECOND;
            } else {
                seekpos = (rand () % (duration / (GST_SECOND))) * GST_SECOND;
            }

            gst_element_seek_simple (playbin, GST_FORMAT_TIME,
                                     GST_SEEK_FLAG_FLUSH |
                                     GST_SEEK_FLAG_ACCURATE, seekpos);

            /* Wait for seek to complete */
            count = 0;
            state = gst_element_get_state (playbin, NULL, 0,
                                           0.2 * GST_SECOND);
	    while (state == GST_STATE_CHANGE_ASYNC && count < 3) {
		g_print ("Waiting in loop %d\n", count);
		state = gst_element_get_state (playbin, NULL, 0, 1 * GST_SECOND);
		count++;
	    }

            g_object_get (playbin,
                          "frame", &frame,
                          NULL);
            if (frame == NULL) {
                g_warning ("No frame for %s", uri);
                return NULL;
            }

            shot = convert_buffer_to_pixbuf (frame);
        }
    }

    gst_element_set_state (playbin, GST_STATE_NULL);
    g_object_unref (playbin);

    return shot;
}

static gboolean
is_interesting (GdkPixbuf *pixbuf)
{
    int width, height, r, rowstride;
    gboolean has_alpha;
    guint32 histogram[4][4][4] = {{{0,},},};
    guchar *pixels;
    int pxl_count = 0, count, i;

    width = gdk_pixbuf_get_width (pixbuf);
    height = gdk_pixbuf_get_height (pixbuf);
    rowstride = gdk_pixbuf_get_rowstride (pixbuf);
    has_alpha = gdk_pixbuf_get_has_alpha (pixbuf);

    pixels = gdk_pixbuf_get_pixels (pixbuf);
    for (r = 0; r < height; r++) {
        guchar *row = pixels + (r * rowstride);
        int c;

        for (c = 0; c < width; c++) {
            guchar r, g, b;

            r = row[0];
            g = row[1];
            b = row[2];

            histogram[r / 64][g / 64][b / 64]++;

            if (has_alpha) {
                row += 4;
            } else {
                row += 3;
            }

            pxl_count++;
        }
    }

    count = 0;
    for (i = 0; i < 4; i++) {
        int j;
        for (j = 0; j < 4; j++) {
            int k;

            for (k = 0; k < 4; k++) {
                /* Count how many bins have more than
                   1% of the pixels in the histogram */
                if (histogram[i][j][k] > pxl_count / 100) {
                    count++;
                }
            }
        }
    }

    /* Image is boring if there is only 1 bin with > 1% of pixels */
    return count > 1;
}

gboolean
bkl_task_video_thumbnail (GFile      *file,
                          GFileInfo  *info,
                          const char *mimetype,
                          GHashTable *metadata)
{
    gboolean success = TRUE;
    gchar *uri;
    GdkPixbuf *shot = NULL;
    gboolean interesting = FALSE;
    int count = 0;

    uri = g_file_get_uri (file);

    while (interesting == FALSE && count < 5) {
        shot = get_shot (uri);
        if (shot == NULL) {
            g_free (uri);
            return FALSE;
        }

	count++;
        interesting = is_interesting (shot);
    }

    if (shot) {
        char *filename, *fullname;
        guint w, h;
        GError *error = NULL;

        w = gdk_pixbuf_get_width (shot);
        h = gdk_pixbuf_get_height (shot);

        filename = g_compute_checksum_for_string (G_CHECKSUM_MD5, uri, -1);
        fullname = g_build_filename (g_get_home_dir (), ".bkl-thumbnails",
                                     filename, NULL);
        g_free (filename);

        if (gdk_pixbuf_save (shot, fullname, "png", &error, NULL) == FALSE) {
            g_warning ("Error writing file %s for %s: %s", fullname, uri,
                       error->message);
            g_error_free (error);

            success = FALSE;
        } else {
            char *thumb_uri;

            thumb_uri = g_filename_to_uri (fullname, NULL, &error);
            if (thumb_uri == NULL) {
                g_warning ("Error converting %s to uri: %s", fullname,
                           error->message);
                g_error_free (error);
                success = FALSE;
            } else {
                g_hash_table_insert (metadata, METADATA_EXTENDED_THUMBNAIL,
                                     thumb_uri);
            }

            g_hash_table_insert (metadata, METADATA_VIDEO_WIDTH,
                                 g_strdup_printf ("%u", w));
            g_hash_table_insert (metadata, METADATA_VIDEO_HEIGHT,
                                 g_strdup_printf ("%u", h));
        }

        g_free (fullname);
        g_object_unref (shot);
    } else {
        success = FALSE;
    }

    g_free (uri);

    return success;
}
