/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <libgupnp/gupnp-control-point.h>
#include <libgupnp/gupnp-device-info.h>

#include <bickley/bkl-db.h>
#include <bickley/bkl-entry.h>
#include <bickley/bkl-item-extended.h>

#include "bkl-indexer.h"
#include "bkl-source-upnp.h"
#include "bkl-finder-upnp.h"

typedef struct _BklSourceUPnP {
    BklSource source;

    BklFinder *finder;
    GUPnPDeviceInfo *info;
} BklSourceUPnP;

typedef struct _BklSourceUPnPClass {
    BklSourceClass parent_class;
} BklSourceUPnPClass;

G_DEFINE_TYPE (BklSourceUPnP, bkl_source_upnp, BKL_TYPE_SOURCE);

static void
bkl_source_upnp_finalize (GObject *object)
{
    BklSourceUPnP *upnp = (BklSourceUPnP *) object;
    BklSource *source = (BklSource *) object;

    if (source->db) {
        bkl_db_free (source->db);
        source->db = NULL;
    }

    if (upnp->finder) {
        bkl_finder_destroy (upnp->finder);
        upnp->finder = NULL;
    }

    G_OBJECT_CLASS (bkl_source_upnp_parent_class)->finalize (object);
}

static void
bkl_source_upnp_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_source_upnp_parent_class)->dispose (object);
}

static gboolean
bkl_source_upnp_do_work (BklSource *source)
{
    BklSourceUPnP *upnp = (BklSourceUPnP *) source;

    source->more_work = bkl_finder_lazy_dig (upnp->finder);
    if (source->more_work == FALSE) {
        bkl_source_completed (source);
    }

    return source->more_work;
}

static void
bkl_source_upnp_investigate_uris (BklSource *source,
                                  GPtrArray *files)
{
    /* BklSourceUPnP *upnp = (BklSourceUPnP *) source; */
}

static GPtrArray *
copy_string_array (GPtrArray *original)
{
    GPtrArray *clone;
    int i;

    if (original == NULL) {
        return NULL;
    }

    clone = g_ptr_array_sized_new (original->len);
    for (i = 0; i < original->len; i++) {
        g_ptr_array_add (clone, g_strdup (original->pdata[i]));
    }

    return clone;
}

static gboolean
bkl_source_upnp_add_item (BklSource  *source,
                          const char *uri,
                          BklItem    *item,
                          GError    **error)
{
    BklItem *old;

    old = bkl_db_get_item (source->db, uri, error);
    if (*error != NULL) {
        if ((*error)->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
            return FALSE;
        }

        g_error_free (*error);
        *error = NULL;
    }

    if (old == NULL) {
        bkl_db_add_item (source->db, uri, item, error);
        bkl_indexer_index_item (source, item, source->db->db, error);
        bkl_source_uri_added (source, uri);
    } else {
        BklItemExtended *ext = (BklItemExtended *) old;
        GPtrArray *tags;

        /* Copy the extended details from the existing item
           as these shouldn't change on update */
        bkl_item_extended_set_use_count
            ((BklItemExtended *) item, bkl_item_extended_get_use_count (ext));
        bkl_item_extended_set_last_used
            ((BklItemExtended *) item, bkl_item_extended_get_last_used (ext));
        bkl_item_extended_set_rating
            ((BklItemExtended *) item, bkl_item_extended_get_rating (ext));
        bkl_item_extended_set_pinned
            ((BklItemExtended *) item, bkl_item_extended_get_pinned (ext));

        tags = bkl_item_extended_get_tags (ext);
        bkl_item_extended_set_tags ((BklItemExtended *) item,
                                    copy_string_array (tags));

        bkl_db_replace_item (source->db, uri, item, error);
        bkl_indexer_update_item (source, item, old, source->db->db);

        g_object_unref (old);

        bkl_source_uri_changed (source, uri);
    }

    return TRUE;
}

static gboolean
kozo_init (BklSourceUPnP    *source)
{
    BklSource *s = (BklSource *) source;
    GError *error = NULL;
    const char *udn;
    char *name, *enc;

    udn = gupnp_device_info_get_udn (source->info);
    enc = g_compute_checksum_for_string (G_CHECKSUM_MD5, udn, -1);
    name = gupnp_device_info_get_friendly_name (source->info);

    s->db = bkl_db_get_for_name (enc, name, &error);
    g_free (enc);

    if (s->db == NULL) {
        g_warning ("Error getting Bickley database for %s: %s", udn,
                   error->message);
        g_free (name);
        return FALSE;
    }

    s->name = name;

    return TRUE;
}

static void
bkl_source_upnp_class_init (BklSourceUPnPClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    BklSourceClass *s_class = (BklSourceClass *) klass;

    o_class->finalize = bkl_source_upnp_finalize;
    o_class->dispose = bkl_source_upnp_dispose;

    s_class->do_work = bkl_source_upnp_do_work;
    s_class->investigate_files = bkl_source_upnp_investigate_uris;
    s_class->add_item = bkl_source_upnp_add_item;
}

static void
bkl_source_upnp_init (BklSourceUPnP *upnp)
{
}

BklSource *
bkl_source_upnp_new (GUPnPDeviceProxy *proxy)
{
    BklSourceUPnP *upnp;
    BklSource *source;

    upnp = g_object_new (BKL_TYPE_SOURCE_UPNP, NULL);
    source = (BklSource *) upnp;

    upnp->info = (GUPnPDeviceInfo *) proxy;

    if (kozo_init (upnp) == FALSE) {
        g_free (source->name);
        g_object_unref (upnp);
        return NULL;
    }

    g_print ("Created UPnP Source: %s\n", source->name);
    upnp->finder = bkl_finder_upnp_new (source, proxy);

    source->more_work = TRUE;

    return source;
}
