/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dbus/dbus-glib.h>

#include "bkl-source-manager.h"

struct _BklSourceManagerPrivate {
    GList *sources;
    DBusGConnection *connection;
};

enum {
    ADDED,
    REMOVED,
    LAST_SIGNAL
};

#define BKL_SOURCE_MANAGER_OBJECT_PATH "/org/moblin/Bickley/Sources/"
#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_SOURCE_MANAGER, BklSourceManagerPrivate))
G_DEFINE_TYPE (BklSourceManager, bkl_source_manager, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = {0, };

static gboolean bkl_source_manager_list_sources (BklSourceManager *manager,
                                                 char           ***OUT_sources,
                                                 GError          **error);
#include "bkl-source-manager-glue.h"

static char *
make_object_path (BklSource *source)
{
    char *path, *esc;

    esc = g_compute_checksum_for_string (G_CHECKSUM_MD5,
                                         kozo_db_get_filename (source->db->db),
                                         -1);

    path = g_strdup_printf ("%s%s", BKL_SOURCE_MANAGER_OBJECT_PATH, esc);
    g_free (esc);

    return path;
}

static gboolean
bkl_source_manager_list_sources (BklSourceManager *manager,
                                 char           ***OUT_sources,
                                 GError          **error)
{
    BklSourceManagerPrivate *priv = manager->priv;
    char **sources;
    int source_count, i;
    GList *s;

    source_count = g_list_length (priv->sources);

    /* + 1 for NULL termination */
    sources = g_new0 (char *, source_count + 1);

    for (s = priv->sources, i = 0; s; s = s->next, i++) {
        BklSource *source = s->data;
        char *path = make_object_path (source);

        sources[i] = path;
    }

    *OUT_sources = sources;
    return TRUE;
}

static void
bkl_source_manager_class_init (BklSourceManagerClass *klass)
{
    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                     &dbus_glib_bkl_source_manager_object_info);
    g_type_class_add_private (klass, sizeof (BklSourceManagerPrivate));

    signals[ADDED] = g_signal_new ("source-added",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                   g_cclosure_marshal_VOID__STRING,
                                   G_TYPE_NONE, 1, G_TYPE_STRING);
    signals[REMOVED] = g_signal_new ("source-removed",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
bkl_source_manager_init (BklSourceManager *self)
{
    BklSourceManagerPrivate *priv;
    GError *error = NULL;

    self->priv = GET_PRIVATE (self);
    priv = self->priv;

    priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (priv->connection == NULL) {
        g_warning ("Error getting bus: %s", error->message);
        g_error_free (error);

        return;
    }
}

void
bkl_source_manager_add_source (BklSourceManager *manager,
                               BklSource        *source)
{
    BklSourceManagerPrivate *priv = manager->priv;
    char *path;

    priv->sources = g_list_prepend (priv->sources, source);
    path = make_object_path (source);

    dbus_g_connection_register_g_object (priv->connection, path,
                                         G_OBJECT (source));
    g_signal_emit (manager, signals[ADDED], 0, path);
    g_free (path);
}

void
bkl_source_manager_remove_source (BklSourceManager *manager,
                                  BklSource        *source)
{
    BklSourceManagerPrivate *priv = manager->priv;
    char *path;

    priv->sources = g_list_remove (priv->sources, source);
    path = make_object_path (source);
    g_signal_emit (manager, signals[REMOVED], 0, path);
    g_free (path);
}
