/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef __KOZO_DB_H__
#define __KOZO_DB_H__

#include <glib.h>

#include "kozo-entry.h"

#define KOZO_FIELD_KEY -1

GQuark kozo_db_error_quark (void);

#define KOZO_DB_ERROR (kozo_db_error_quark ())

typedef enum {
        KOZO_DB_ERROR_BACKEND,
        KOZO_DB_ERROR_IO,
        KOZO_DB_ERROR_KEY_NOT_FOUND,
        KOZO_DB_ERROR_KEY_EXISTS,
	KOZO_DB_ERROR_VERSION_MISMATCH
} KozoDBError;

typedef struct _KozoDB KozoDB;

typedef gboolean (*KozoDBForeachFunc) (KozoDB     *db,
                                       const char *key,
                                       KozoEntry  *entry,
                                       gpointer    userdata);

/* All these functions are thread-safe */
KozoDB *kozo_db_get_for_path       (const char *filename,
                                    const char *db_name,
                                    guint       version,
                                    GError    **error);
KozoDB     *kozo_db_get            (const char *db_name,
                                    guint       version,
                                    GError    **error);
KozoDB     *kozo_db_get_for_name   (const char *filename,
                                    const char *db_name,
                                    guint       version,
                                    GError    **error);

KozoDB     *kozo_db_ref            (KozoDB     *db);
void        kozo_db_unref          (KozoDB     *db);

const char *kozo_db_get_filename   (KozoDB *db);
const char *kozo_db_get_name       (KozoDB     *db);
void        kozo_db_set_name       (KozoDB     *db,
                                    const char *db_name);

guint       kozo_db_get_version    (KozoDB     *db);

KozoEntry  *kozo_db_lookup         (KozoDB     *db,
                                    const char *key,
                                    GSList     *field_ids,
                                    GError    **error);

gboolean    kozo_db_set            (KozoDB     *db,
                                    const char *key,
                                    GSList     *fields,
                                    GError    **error);

gboolean    kozo_db_add            (KozoDB     *db,
                                    const char *key,
                                    GSList     *fields,
                                    GError    **error);

gboolean    kozo_db_remove         (KozoDB     *db,
                                    const char *key,
                                    GError    **error);

int         kozo_db_register_field (KozoDB     *db,
                                    const char *desc,
                                    GError    **error);

void        kozo_db_foreach        (KozoDB           *db,
                                    KozoDBForeachFunc func,
                                    gpointer          userdata);

gboolean    kozo_db_index_add_word (KozoDB     *db,
                                    const char *word,
                                    const char *uri,
                                    gboolean   *new_word,
                                    GError    **error);
GList *     kozo_db_index_lookup   (KozoDB     *db,
                                    const char *word,
                                    GError    **error);
gboolean kozo_db_index_remove_word (KozoDB     *db,
                                    const char *word,
                                    const char *uri,
                                    gboolean   *deleted,
                                    GError    **error);
GSequence *kozo_db_get_index_words (KozoDB *db);

/* Call this after modifying the DB to make sure all the changes are
 * propagated */
void        kozo_db_flush          (KozoDB     *db);

#endif /* __KOZO_DB_H__ */
