/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>

#include "bkl-entry.h"
#include "bkl-item-video.h"

enum {
    PROP_0,
    PROP_TITLE,
    PROP_DIRECTOR,
    PROP_SERIES_NAME,
    PROP_SEASON,
    PROP_EPISODE,
    PROP_YEAR,
    PROP_DURATION,
    PROP_WIDTH,
    PROP_HEIGHT,
    PROP_SERIES_IMAGE,
    PROP_DISC,
};

struct _BklItemVideoPrivate {
    char *title;
    char *director;
    char *series_name;
    char *series_image;

    guint season;
    guint episode;
    guint year;

    guint duration;

    guint width;
    guint height;

    guint disc;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_ITEM_VIDEO, BklItemVideoPrivate))
G_DEFINE_TYPE (BklItemVideo, bkl_item_video, BKL_TYPE_ITEM_EXTENDED);

static void
bkl_item_video_finalize (GObject *object)
{
    BklItemVideo *self = (BklItemVideo *) object;
    BklItemVideoPrivate *priv = self->priv;

    g_free (priv->title);
    g_free (priv->director);
    g_free (priv->series_name);
    g_free (priv->series_image);

    G_OBJECT_CLASS (bkl_item_video_parent_class)->finalize (object);
}

static void
bkl_item_video_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_item_video_parent_class)->dispose (object);
}

static void
bkl_item_video_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    switch (prop_id) {
    case PROP_TITLE:
        break;

    case PROP_DIRECTOR:
        break;

    case PROP_SERIES_NAME:
        break;

    case PROP_SEASON:
        break;

    case PROP_EPISODE:
        break;

    case PROP_YEAR:
        break;

    case PROP_DURATION:
        break;

    case PROP_WIDTH:
        break;

    case PROP_HEIGHT:
        break;

    case PROP_SERIES_IMAGE:
        break;

    case PROP_DISC:
        break;

    default:
        break;
    }
}

static void
bkl_item_video_get_property (GObject    *object,
                          guint       prop_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    switch (prop_id) {
    case PROP_TITLE:
        break;

    case PROP_DIRECTOR:
        break;

    case PROP_SERIES_NAME:
        break;

    case PROP_SEASON:
        break;

    case PROP_EPISODE:
        break;

    case PROP_YEAR:
        break;

    case PROP_DURATION:
        break;

    case PROP_WIDTH:
        break;

    case PROP_HEIGHT:
        break;

    case PROP_SERIES_IMAGE:
        break;

    case PROP_DISC:
        break;

    default:
        break;
    }
}

static void
bkl_item_video_class_init (BklItemVideoClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_item_video_dispose;
    o_class->finalize = bkl_item_video_finalize;
    o_class->set_property = bkl_item_video_set_property;
    o_class->get_property = bkl_item_video_get_property;

    g_type_class_add_private (klass, sizeof (BklItemVideoPrivate));

    g_object_class_install_property (o_class, PROP_TITLE,
                                     g_param_spec_string ("title", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_DIRECTOR,
                                     g_param_spec_string ("director", "", "",
                                                          "", G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SERIES_NAME,
                                     g_param_spec_string ("name", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SERIES_IMAGE,
                                     g_param_spec_string ("series-image", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SEASON,
                                     g_param_spec_uint ("season", "", "",
                                                        0, 100, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_EPISODE,
                                     g_param_spec_uint ("episode", "", "",
                                                        0, 100, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_YEAR,
                                     g_param_spec_uint ("year", "", "",
                                                        1900, 2100, 1900,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (o_class, PROP_DURATION,
                                     g_param_spec_uint ("duration", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (o_class, PROP_WIDTH,
                                     g_param_spec_uint ("width", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (o_class, PROP_HEIGHT,
                                     g_param_spec_uint ("height", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_DISC,
                                     g_param_spec_uint ("disc", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
}

static void
bkl_item_video_init (BklItemVideo *self)
{
    self->priv = GET_PRIVATE (self);

    bkl_item_set_item_type ((BklItem *) self, BKL_ITEM_TYPE_VIDEO);
}

BklItemVideo *
bkl_item_video_new (void)
{
    BklItemVideo *video;

    video = g_object_new (BKL_TYPE_ITEM_VIDEO, NULL);
    return video;
}

#define IS_EMPTY_STRING(s) ((s) == NULL || *(s) == '\0')
BklItemVideo *
bkl_item_video_new_from_field (KozoField *field)
{
    BklItemVideo *video;
    const char *data;
    guint number;

    video = g_object_new (BKL_TYPE_ITEM_VIDEO, NULL);

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_TITLE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_video_set_title (video, data);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_DIRECTOR);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_video_set_director (video, data);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_SERIES_NAME);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_video_set_series_name (video, data);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_SEASON);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_season (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_EPISODE);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_episode (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_YEAR);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_year (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_DURATION);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_duration (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_WIDTH);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_width (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_HEIGHT);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_height (video, number);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_SERIES_IMAGE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_video_set_series_image (video, data);
    }

    data = kozo_field_get_value_string (field, BKL_VIDEO_FIELD_DISC);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_video_set_disc (video, number);
    }

    return video;
}

GSList *
bkl_item_video_get_fields (BklItemVideo *item)
{
    BklItemVideoPrivate *priv = item->priv;
    GSList *fields = NULL;

    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->disc));
    fields = g_slist_prepend (fields, g_strdup (priv->series_image));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->height));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->width));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->duration));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->year));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->episode));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->season));
    fields = g_slist_prepend (fields, g_strdup (priv->series_name));
    fields = g_slist_prepend (fields, g_strdup (priv->director));
    fields = g_slist_prepend (fields, g_strdup (priv->title));

    return fields;
}

void
bkl_item_video_set_title (BklItemVideo *video,
                          const char   *title)
{
    g_free (video->priv->title);
    video->priv->title = g_strdup (title);
}

const char *
bkl_item_video_get_title (BklItemVideo *video)
{
    return video->priv->title;
}

void
bkl_item_video_set_director (BklItemVideo *video,
                             const char   *director)
{
    g_free (video->priv->director);
    video->priv->director = g_strdup (director);
}

const char *
bkl_item_video_get_director (BklItemVideo *video)
{
    return video->priv->director;
}

void
bkl_item_video_set_series_name (BklItemVideo *video,
                                const char   *series_name)
{
    g_free (video->priv->series_name);
    video->priv->series_name = g_strdup (series_name);
}

const char *
bkl_item_video_get_series_name (BklItemVideo *video)
{
    return video->priv->series_name;
}

void
bkl_item_video_set_season (BklItemVideo *video,
                           guint         season)
{
    video->priv->season = season;
}

guint
bkl_item_video_get_season (BklItemVideo *video)
{
    return video->priv->season;
}

void
bkl_item_video_set_episode (BklItemVideo *video,
                            guint         episode)
{
    video->priv->episode = episode;
}

guint
bkl_item_video_get_episode (BklItemVideo *video)
{
    return video->priv->episode;
}

void
bkl_item_video_set_year (BklItemVideo *video,
                         guint         year)
{
    video->priv->year = year;
}

guint
bkl_item_video_get_year (BklItemVideo *video)
{
    return video->priv->year;
}

void
bkl_item_video_set_duration (BklItemVideo *video,
                             guint         duration)
{
    video->priv->duration = duration;
}

guint
bkl_item_video_get_duration (BklItemVideo *video)
{
    return video->priv->duration;
}

void
bkl_item_video_set_width (BklItemVideo *video,
                          guint         width)
{
    video->priv->width = width;
}

guint
bkl_item_video_get_width (BklItemVideo *video)
{
    return video->priv->width;
}

void
bkl_item_video_set_height (BklItemVideo *video,
                           guint         height)
{
    video->priv->height = height;
}

guint
bkl_item_video_get_height (BklItemVideo *video)
{
    return video->priv->height;
}

void
bkl_item_video_set_series_image (BklItemVideo *video,
                                 const char   *series_image)
{
    g_free (video->priv->series_image);
    video->priv->series_image = g_strdup (series_image);
}

const char *
bkl_item_video_get_series_image (BklItemVideo *video)
{
    return video->priv->series_image;
}

void
bkl_item_video_set_disc (BklItemVideo *video,
                         guint         disc)
{
    video->priv->disc = disc;
}

guint
bkl_item_video_get_disc (BklItemVideo *video)
{
    return video->priv->disc;
}

char *
bkl_item_video_to_string (BklItemVideo *video)
{
    BklItemVideoPrivate *priv = video->priv;
    GString *contents;
    char *str, *extra;

    contents = g_string_new ("");
    if (priv->title) {
        g_string_append (contents, priv->title);
        g_string_append_c (contents, ' ');
    }

    if (priv->director) {
        g_string_append (contents, priv->director);
        g_string_append_c (contents, ' ');
    }

    if (priv->series_name) {
        g_string_append (contents, priv->series_name);
        g_string_append_c (contents, ' ');
    }

    if (priv->season > 0) {
        /* Search for season as Sx or S0x */
        extra = g_strdup_printf ("S%u", priv->season);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');

        extra = g_strdup_printf ("S%02u", priv->season);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');
    }

    if (priv->episode > 0) {
        /* Search for episode as Ex or E0x */
        extra = g_strdup_printf ("E%u", priv->episode);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');

        extra = g_strdup_printf ("E%02u", priv->episode);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');
    }

    if (priv->season > 0 && priv->episode > 0) {
        /* Search also for SxEx or S0xE0x */
        extra = g_strdup_printf ("S%uE%u", priv->season, priv->episode);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');

        extra = g_strdup_printf ("S%02uE%02u", priv->season, priv->episode);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');
    }

    if (priv->year > 0) {
        extra = g_strdup_printf ("%u", priv->year);
        g_string_append (contents, extra);
        g_free (extra);
        g_string_append_c (contents, ' ');
    }

    /* FIXME: Do we need the widthxheight? */

    str = contents->str;
    g_string_free (contents, FALSE);

    return str;
}
