/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "opendocument.h"
#include "usfm.h"
#include "bible.h"
#include "usfmtools.h"
#include "gwrappers.h"
#include "shell.h"
#include "utilities.h"
#include "directories.h"
#include <config.h>
#include "date_time_utils.h"
#include "projectutils.h"
#include "books.h"
#include "xmlutils.h"
#include "usfmtools.h"
#include "fonts.h"


/*
Some properties of the stylesheet are not implemented for certain reasons.

The "spancolumns" property is not implemented, because it is not so easy in
OpenOffice Writer to copy text divided into various sections. Switching from
one to two columns would divide the text into these sections.

The chapter number is not put at verse one, but just where it occurs. This is
because frames would be needed, and this makes copying text more difficult.
*/


OpenDocument::OpenDocument (const ustring& project, const ustring& filename, bool gui)
// Export a project to OpenDocument format.
: genconfig (0),
  projectconfig (""),
  usfm (projectconfig.stylesheet())
{
  // Check on basic markers.
  if (!usfm_basic_markers_present (usfm, gui)) return;
    
  // Progress information.  
  progresswindow = NULL;
  if (gui) progresswindow = new ProgressWindow ("Exporting to OpenDocument", true);

  // Prepare for inline text markers, notes and xrefs.
  usfm_inline_markers = new UsfmInlineMarkers (usfm);
  odtfootnote = new OdtFootnote (usfm);
  odtendnote = new OdtEndnote (usfm);
  odtxref = new OdtXref (usfm);
  
  // Retrieve and process template: page layout, font.
  get_template ();
  odt_create_page_master (workingdirectory);
  PrintingFonts printingfonts (project);
  if (!printingfonts.printing_families.empty()) 
    fontname = printingfonts.printing_families[0];   
  odt_set_font (workingdirectory, fontname);
  
  // Get the available books.
  books = project_get_books (project);

  // Progress
  if (progresswindow) progresswindow->set_iterate (0, 1, books.size());
    
  // Generate the cover.
  cover ();
    
  // Go through all the books.
  for (unsigned int i = 0; i < books.size(); i++) {
    
    // Progress.
    if (progresswindow) {
      progresswindow->iterate ();
      if (progresswindow->cancel) {
        return;
      }
    } else {
      gw_message (books_id_to_english (books[i]));
    }
    
    // Skip "Other Material".
    if (books_id_to_type (books[i]) == btOtherMaterial) 
      continue;

    // Signal new book to footnotes object.
    odtfootnote->new_book ();
    
    // Open the book.
    vector<ustring> book_lines;
    book_lines = project_retrieve_book (project, books[i]);
    
    // Format the lines of the book.
    format_general (book_lines);
  }
 
  // Produce the styles.
  vector <ustring> automaticstyles;
  vector <ustring> styles;
  generate_styles (automaticstyles, styles);
  
  // Store content.xml.
  odt_insert_content (workingdirectory, odtlines, automaticstyles);  
 
  // Store styles.xml.
  odt_insert_styles (workingdirectory, styles);  

  // Automatic .odt extension if needed.
  ustring filename_odt (filename);
  if (!g_str_has_suffix (filename.c_str(), ".odt")) filename_odt.append (".odt");
  
  // Compress the file.
  zip (filename_odt);
}


OpenDocument::~OpenDocument ()
{
  if (progresswindow) delete progresswindow;
  delete usfm_inline_markers;
  delete odtfootnote;
  delete odtendnote;
  delete odtxref;
}


void OpenDocument::get_template ()
{
  // Clear working directory.
  workingdirectory = gw_build_filename (directories_get_temp(), "opendocument");
  remove_directory (workingdirectory);
  create_directory (workingdirectory);
  // Copy template there.
  // Note: To create the template use zip -r template.odt *
  ustring command = "cd";
  command.append (shell_quote_space (workingdirectory));
  command.append ("; cp ");
  command.append (gw_build_filename (PACKAGE_DATA_DIR, "template.odt"));
  command.append (" .; unzip *; rm *.odt");
  system (command.c_str());
}


void OpenDocument::cover ()
// Generate cover page.
{
  // Store the lines containing the cover.  
  vector <ustring> lines;
  
  // Collect the lines: Look for the \cov section in the Other Material book. 
  for (unsigned int i = 0; i < books.size(); i++) {
    if (books_id_to_type (books[i]) == btOtherMaterial) 
      continue;
    vector <ustring> rawlines = project_retrieve_chapter (genconfig.project(), books[i], 0);
    bool within_cover_section = false;
    for (unsigned int i2 = 0; i2 < rawlines.size(); i2++) {
      ustring line = rawlines[i2];
      ustring marker = usfm_extract_marker (line);
      if (usfm.is_peripheral (marker))
        if (usfm.peripheral_get_subtype (marker) != ptCover)
          within_cover_section = false;
      if (within_cover_section) {
        lines.push_back (rawlines[i2]);
      }
      if (usfm.is_peripheral (marker)) {
        if (usfm.peripheral_get_subtype (marker) == ptCover)
          within_cover_section = true;
      }
    }
  }
  
  // Format cover.
  format_general (lines);  
}


void OpenDocument::format_general (vector <ustring>& lines)
// General formatter for USFM lines given.
{
  // Go through all the lines.
  odttextparagraph = NULL;
  for (unsigned int ln = 0; ln < lines.size(); ln++) {
    ustring line = lines[ln];
    // Take any elastics out, put the \b marker instead.
    replace_text (line, ELASTIC_MARKER, "b");
    // Change certain characters to xml entities.
    xml_handle_entities (line, NULL);
    // Deal with footnotes.
    odtfootnote->transform (line);
    // Deal with endnotes.
    odtendnote->transform (line);
    // Deal with crossreferences.
    odtxref->transform (line);
    // Deal with inline text.
    usfm_handle_inline_text (line, usfm_inline_markers, NULL, imOpenDocument, NULL);
    // Signal new line.
    if (odttextparagraph) odttextparagraph->newline ();
    // Get the style belonging to the marker.
    ustring marker = usfm_extract_marker (line);
    if (usfm.is_identifier (marker)) {
      // Handle some identifiers.
      IdentifierType identifiertype = usfm.identifier_get_subtype (marker);
      // New book.
      if (identifiertype == itBook) {
        if (odttextparagraph) delete odttextparagraph;
        odttextparagraph = new OdtTextParagraph (&odtlines, marker);
      }
    } else if (usfm.is_verse_number (marker)) {
      // Because of dealing with portions to include/exclude, handle verse first.
      // Get verse number. Handle combined verses too, e.g. 10-12b, etc.
      size_t position = line.find (" ");
      position = CLAMP (position, 0, line.length());
      ustring versenumber = line.substr (0, position);
      position++;
      line.erase (0, position);
      // There was a bug that the chapter number was missing from a book,
      // and that results in text being inserted without the opening xml code.
      // Solution: If no paragraph has been opened, open a default one.
      if (odttextparagraph == NULL) odttextparagraph = new OdtTextParagraph (&odtlines, "");
      // When the usfm is a verse number, then the number is put in the 
      // format specified by the stylesheet, but the remaining part of the
      // line inherits the formatting from the paragraph it is in.
      odttextparagraph->spannedtext (versenumber, marker);
      odttextparagraph->plaintext (line);
    } else if (usfm.is_starting_paragraph (marker)) {
      if (odttextparagraph) delete odttextparagraph;
      odttextparagraph = new OdtTextParagraph (&odtlines, marker);
      if (!line.empty()) odttextparagraph->plaintext(line);
    } else if (usfm.is_inline_text (marker)) {
      // Inline text, has been dealt with before (therefore should never occur here).
      if (odttextparagraph) odttextparagraph->spannedtext (line, marker);
    } else if (usfm.is_chapter_number (marker)) {
      // Close possible open paragraph.
      if (odttextparagraph) delete odttextparagraph;
      // Signal new chapter to footnotes object.
      odtfootnote->new_chapter ();      
      // Insert or prepare chapter text.
      odttextparagraph = new OdtTextParagraph (&odtlines, marker);
      odttextparagraph->plaintext (line);
    } else if (usfm.is_peripheral (marker)) {
    } else if (usfm.is_picture (marker)) {
    } else if (usfm.is_pagebreak (marker)) {
      if (odttextparagraph) delete odttextparagraph;
      odttextparagraph = new OdtTextParagraph (&odtlines, marker);
    } else {
      // Fallback for unknown marker or no marker.
      if (!odttextparagraph) odttextparagraph = new OdtTextParagraph (&odtlines, "");
      odttextparagraph->plaintext (line);
    }
  } 
  // Close possible last paragraph.
  if (odttextparagraph) delete odttextparagraph;
}


void OpenDocument::generate_styles (vector <ustring>& automaticstyles, vector <ustring>& styles)
{
  // Go through all the styles and generate the appropriate code.
  for (unsigned int i = 0; i < usfm.styles.size(); i++) {
    ustring marker = usfm.styles[i].marker;
    if (usfm.is_identifier (marker)) {
      switch (usfm.identifier_get_subtype (marker)) {
        case itBook:
          odt_paragraph_style (automaticstyles, marker, fontname, 12, 100, OFF, OFF, OFF, OFF, "", 0, 0, 0, 0, 0, false, usfm.userbool1());
          break;
        case itEncoding:
          break;
        case itComment:
          break;
        case itRunningHeader:
          break;
        case itLeftRunningHeader:
          break;
        case itRightRunningHeader:
          break;
        case itLongTOC:
          break;
        case itShortTOC:
          break;
      }
    }
    if (usfm.is_starting_paragraph (marker)) {
      odt_paragraph_style (automaticstyles, marker, fontname,
                           usfm.fontsize(), 100, usfm.italic(), usfm.bold(), usfm.underline(), 
                           usfm.smallcaps(), usfm.justification(),
                           usfm.spacebefore(), usfm.spaceafter(), usfm.leftmargin(),
                           usfm.rightmargin(), usfm.firstlineindent(), false, false);
    }
    if (usfm.is_inline_text (marker)) {
      odt_span_style (automaticstyles, marker, fontname, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), usfm.smallcaps(), usfm.superscript(), usfm.color());
    }
    if (usfm.is_chapter_number (marker)) {
      odt_paragraph_style (automaticstyles, marker, fontname,
                           usfm.fontsize(), 100, usfm.italic(), usfm.bold(), usfm.underline(), 
                           usfm.smallcaps(), usfm.justification(),
                           usfm.spacebefore(), usfm.spaceafter(), usfm.leftmargin(),
                           usfm.rightmargin(), usfm.firstlineindent(), false, false);
    }
    if (usfm.is_verse_number (marker)) {
      odt_span_style (automaticstyles, marker, fontname, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), usfm.smallcaps(), usfm.superscript(), usfm.color());
    }
    if (usfm.is_foot_endnote (marker)) {
      switch (usfm.foot_endnote_get_subtype (marker)) {
        case fentFootnote:
          break;
        case fentEndnote:
          break;
        case fentStandardContent:
        case fentContent:
        case fentContentWithEndmarker:
          odt_span_style (automaticstyles, marker, fontname, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), OFF, usfm.superscript(), usfm.color());
          break;
        case fentParagraph:
          break;
      }
    }
    if (usfm.is_xref (marker)) {
      switch (usfm.xref_get_subtype (marker)) {
        case ctCrossreference:
          break;
        case ctStandardContent:
        case ctContent:
        case ctContentWithEndmarker:
          odt_span_style (automaticstyles, marker, fontname, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), OFF, usfm.superscript(), usfm.color());
          break;
      }
    }
    if (usfm.is_peripheral (marker)) {
      switch (usfm.peripheral_get_subtype (marker)) {
        case ptPublication:
          break;
        case ptTableOfContents:
          break;
        case ptPreface:
          break;
        case ptIntroduction:
          break;
        case ptGlossary:
          break;
        case ptConcordance:
          break;
        case ptIndex:
          break;
        case ptMapIndex:
          break;
        case ptCover:
          break;
        case ptSpine:
          break;
      }
    }
    if (usfm.is_picture (marker)) {
    }
    if (usfm.is_pagebreak (marker)) {
      odt_paragraph_style (automaticstyles, marker, fontname, 12, 100, OFF, OFF, OFF, OFF, "", 0, 0, 0, 0, 0, false, true);
    }
  }
}


void OpenDocument::zip (const ustring filename)
{
  ustring command = "cd";
  command.append (shell_quote_space (workingdirectory));
  command.append ("; zip -r");
  command.append (shell_quote_space (filename));
  command.append ("*");
  system (command.c_str());
}
