/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "project.h"
#include "utilities.h"
#include "config.h"
#include <glib.h>
#include "style.h"
#include "xmlutils.h"
#include "gwrappers.h"
#include "directories.h"
#include "stylesheetutils.h"


Style::Style (const ustring& stylesheet, const ustring& style, bool write)
// Reads a style from the database.
{
  // Save variables.
  mystylesheet = stylesheet;
  marker = style;
  mywrite = write;
  // Set default values.
  name = "Marker";
  info = "Unified Standard Format Marker";
  fontsize = 12;
  fontpercentage = 100;
  italic = OFF;
  bold = OFF;
  underline = OFF;
  underline = OFF;
  superscript = false;
  justification = "left";
  spacebefore = 0;
  spaceafter = 0;
  leftmargin = 0;
  rightmargin = 0;
  firstlineindent = 0;
  spancolumns = false;
  type = stInlineText;
  subtype = 0;
  userbool1 = false;
  userbool2 = false;
  userbool3 = false;
  userint1 = 0;
  userint2 = 0;
  userint3 = 0;
  // Read values from the database.
  stylesheet_load_style (mystylesheet, * this);  
}


Style::~Style ()
{
  // Save style in database.
  if (mywrite) {
    stylesheet_save_style (mystylesheet, * this);
  }
}


void Style::import_it (const ustring& filename)
{
  // Read the style file.
  bool was_error = false;
  gchar *contents;
  gsize  length;
  GError *error;
  error = NULL;
  if (!g_file_get_contents (filename.c_str(), &contents, &length, &error)) {
    cerr << error->message << endl;
    g_error_free (error);
    was_error = true;
  }
  // Set up parser.
  GMarkupParseContext *context;
  GMarkupParser parser = {
    start_element_handler,
    end_element_handler,
    text_handler,
    passthrough_handler,
    error_handler
  };
  // Parse xml file.
  if (!was_error) {
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), gpointer (this), NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error) {
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      cerr << error->message << endl;
    }
  }
  if (!was_error)
    g_markup_parse_context_free (context);
  current_element.clear();
  current_value.clear();
}


void Style::export_it (const ustring& filename)
{
  // Save style in xml format. 
  vector <ustring> output;
  output.push_back (xml_tag (0, STYLE_BIBLEDIT_STYLE, false));
  output.push_back (xml_text_embed_in_tags (1, STYLE_NAME, name));
  output.push_back (xml_text_embed_in_tags (1, STYLE_DESCRIPTION, info));
  output.push_back (xml_text_embed_in_tags (1, STYLE_FONTSIZE, convert_to_string (fontsize)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_FONTPERCENTAGE, convert_to_string (fontpercentage)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_ITALIC, italic));
  output.push_back (xml_text_embed_in_tags (1, STYLE_BOLD, bold));
  output.push_back (xml_text_embed_in_tags (1, STYLE_UNDERLINE, underline));
  output.push_back (xml_text_embed_in_tags (1, STYLE_SMALL_CAPS, smallcaps));
  output.push_back (xml_text_embed_in_tags (1, STYLE_SUPERSCRIPT, convert_to_string (superscript)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_JUSTIFICATION, justification));
  output.push_back (xml_text_embed_in_tags (1, STYLE_SPACE_BEFORE, convert_to_string (spacebefore)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_SPACE_AFTER, convert_to_string (spaceafter)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_MARGIN_LEFT, convert_to_string (leftmargin)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_MARGIN_RIGHT, convert_to_string (rightmargin)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_INDENT_FIRSTLINE, convert_to_string (firstlineindent)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_COLUMNS_SPAN, convert_to_string (spancolumns)));
  ustring s;
  switch (type) {
    case stIdentifier:      { s = STYLE_TYPE_IDENTIFIER; break; }
    case stComment:         { s = STYLE_TYPE_COMMENT; break; }
    case stRunningHeader:   { s = STYLE_TYPE_RUNNING_HEADER; break; }
    case stStartsParagraph: { s = STYLE_TYPE_STARTS_PARAGRAPH; break; }
    case stInlineText:      { s = STYLE_TYPE_INLINE_TEXT; break; }
    case stChapterNumber:   { s = STYLE_TYPE_CHAPTER_NUMBER; break; }
    case stVerseNumber:     { s = STYLE_TYPE_VERSE_NUMBER; break; }
    case stFootEndNote:     { s = STYLE_TYPE_FOOT_END_NOTE; break; }
    case stCrossreference:  { s = STYLE_TYPE_CROSSREFERENCE; break; }
  }
  output.push_back (xml_text_embed_in_tags (1, STYLE_STYLE_TYPE, s));
  output.push_back (xml_text_embed_in_tags (1, STYLE_SUBTYPE, convert_to_string (subtype)));    
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_BOOL1, convert_to_string (userbool1)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_BOOL2, convert_to_string (userbool2)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_BOOL3, convert_to_string (userbool3)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_INT1, convert_to_string (userint1)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_INT2, convert_to_string (userint2)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_INT3, convert_to_string (userint3)));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_STRING1, userstring1));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_STRING2, userstring2));
  output.push_back (xml_text_embed_in_tags (1, STYLE_USER_STRING3, userstring3));
  output.push_back (xml_tag (0, STYLE_BIBLEDIT_STYLE, true));
  try
  {
    write_lines (filename, output);
  }
  catch (exception & ex)
  {
    cerr << ex.what () << endl;
  }
}


void Style::start_element_handler (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error)
{
  ustring element;
  element = element_name;
  ((Style *) user_data)->start_element_handler (element);
}


void Style::end_element_handler (GMarkupParseContext *context,
                                 const gchar         *element_name,
                                 gpointer             user_data,
                                 GError             **error)
{
  ustring element;
  element = element_name;
  ((Style *) user_data)->end_element_handler (element);
}


void Style::text_handler (GMarkupParseContext *context,
                          const gchar         *text,
                          gsize                text_len,
                          gpointer             user_data,
                          GError             **error)
{
  ustring utext;
  utext = text;
  ((Style *) user_data)->text_handler (utext);
}



void Style::passthrough_handler    (GMarkupParseContext *context,
                                    const gchar         *passthrough_text,
                                    gsize                text_len,
                                    gpointer             user_data,
                                    GError             **error)
{
}


void Style::error_handler          (GMarkupParseContext *context,
                                    GError              *error,
                                    gpointer             user_data)
{
  cerr << error->message << endl;
}


void Style::start_element_handler (const ustring& element_name)
// When we encounter a new element that starts data, this handler deals with that.
{
  current_element = element_name;
  current_value.clear();
}


void Style::end_element_handler (const ustring& element_name)
// When we encounter an element that ends data, this handler deals with that.
{
  current_value = trim (current_value);
  if        (current_element == STYLE_NAME) {
    name = current_value;
  } else if (current_element == STYLE_DESCRIPTION) {
    info = current_value;
  } else if (current_element == STYLE_FONTSIZE) {
    fontsize = convert_to_double (current_value);
  } else if (current_element == STYLE_FONTPERCENTAGE) {
    fontpercentage = convert_to_int (current_value);
  } else if (current_element == STYLE_ITALIC) {
    // Automatic conversion from previously used bool variables.
    // 0 = OFF; 1 = ON.
    if (current_value == "0") {
      italic = OFF;
    } else if (current_value == "1") {
      italic = ON;
    } else 
      italic = current_value;
  } else if (current_element == STYLE_BOLD) {
    // Automatic conversion from previously used bool variabeles.
    // 0 = OFF; 1 = ON.
    if (current_value == "0") {
      bold = OFF;
    } else if (current_value == "1") {
      bold = ON;
    } else 
      bold = current_value;
  } else if (current_element == STYLE_UNDERLINE) {
    underline = current_value;
  } else if (current_element == STYLE_SMALL_CAPS) {
    smallcaps = current_value;
  } else if (current_element == STYLE_SUPERSCRIPT) {
    superscript = convert_to_bool (current_value);
  } else if (current_element == STYLE_JUSTIFICATION) {
    justification = current_value;
  } else if (current_element == STYLE_SPACE_BEFORE) {
    spacebefore = convert_to_double (current_value);
  } else if (current_element == STYLE_SPACE_AFTER) {
    spaceafter = convert_to_double (current_value);
  } else if (current_element == STYLE_MARGIN_LEFT) {
    leftmargin = convert_to_double (current_value);
  } else if (current_element == STYLE_MARGIN_RIGHT) {
    rightmargin = convert_to_double (current_value);
  } else if (current_element == STYLE_INDENT_FIRSTLINE) {
    firstlineindent = convert_to_double (current_value);
  } else if (current_element == STYLE_COLUMNS_SPAN) {
    spancolumns = convert_to_bool (current_value);
  } else if (current_element == STYLE_STYLE_TYPE) {
    if (current_value == STYLE_TYPE_IDENTIFIER)
      type = stIdentifier;
    else if (current_value == STYLE_TYPE_COMMENT)
      type = stComment;
    else if (current_value == STYLE_TYPE_RUNNING_HEADER)
      type = stRunningHeader;
    else if (current_value == STYLE_TYPE_STARTS_PARAGRAPH)
      type = stStartsParagraph;
    else if (current_value == STYLE_TYPE_INLINE_TEXT)
      type = stInlineText;
    else if (current_value == STYLE_TYPE_CHAPTER_NUMBER)
      type = stChapterNumber;
    else if (current_value == STYLE_TYPE_VERSE_NUMBER)
      type = stVerseNumber;
    else if (current_value == STYLE_TYPE_FOOT_END_NOTE)
      type = stFootEndNote;
    else if (current_value == STYLE_TYPE_CROSSREFERENCE)
      type = stCrossreference;
  } else if (current_element == STYLE_SUBTYPE) {
    subtype = convert_to_int (current_value);
  } else if (current_element == STYLE_USER_BOOL1) {
    userbool1 = convert_to_bool (current_value);
  } else if (current_element == STYLE_USER_BOOL2) {
    userbool2 = convert_to_bool (current_value);
  } else if (current_element == STYLE_USER_BOOL3) {
    userbool3 = convert_to_bool (current_value);
  } else if (current_element == STYLE_USER_INT1) {
    userint1 = convert_to_int (current_value);
  } else if (current_element == STYLE_USER_INT2) {
    userint2 = convert_to_int (current_value);
  } else if (current_element == STYLE_USER_INT3) {
    userint3 = convert_to_int (current_value);
  } else if (current_element == STYLE_USER_STRING1) {
    userstring1 = current_value;
  } else if (current_element == STYLE_USER_STRING2) {
    userstring2 = current_value;
  } else if (current_element == STYLE_USER_STRING3) {
    userstring3 = current_value;
  }
  current_element.clear();
  current_value.clear();
}


void Style::text_handler (const ustring& text)
// When we encounter the text inside an element, this handler deals with that.
{
  current_value.append (text);
}
