/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "bookmetrics.h"
#include "gwrappers.h"
#include "shell.h"


BookMetrics::BookMetrics (const ustring & path)
/*
It reads the bookmetrics from the book in "path".
It uses a metrics file for higher speed, so that the book itself does not need
to be checked - only the metrics file. That gives us the information. 
It updates the metrics file when there was a change in the book.
*/
{
  // Get the name of the bookmetrics file.
  ustring metricsfilename;
  metricsfilename = gw_path_get_dirname (path);
  metricsfilename = gw_build_filename (metricsfilename, "book-metrics.xml");
  // If the metrics file does not exist, create it.
  if (!g_file_test (metricsfilename.c_str(), G_FILE_TEST_IS_REGULAR)) {
    create_metrics (path, metricsfilename);
  }
  // Parse the metrics file.
  parse_modified = 0;
  parse_metrics (metricsfilename);
  // If the book was updated, create a new metrics file and parse it again.
  if (parse_modified != file_get_modification_time (path)) {
    create_metrics (path, metricsfilename);
    parse_metrics (metricsfilename);
  }
}


BookMetrics::~BookMetrics ()
{
}


void BookMetrics::create_metrics (const ustring& bookfilename, const ustring& metricsfilename)
{
  // Delete any existing metrics file.
  unlink (metricsfilename.c_str());
  // Store modification time first.
  {
    ustring text = "<modified time=\"";
    text.append (convert_to_string ((int) file_get_modification_time (bookfilename)));
    text.append ("\">\n");
    text.append ("</modified>\n");
    WriteText wt (metricsfilename);
    wt.text (text);
  }
  // Let scripturechecks give us the book data.
  // Filter everything out, except the data we are interested in, 
  // like book, chapters and verses.
  ustring command;
  command = "sc-input-usfm --no-text";
  command.append (shell_quote_space (bookfilename));
  command.append (">>");
  command.append (shell_quote_space (metricsfilename));
  system (command.c_str());
}



void BookMetrics::parse_metrics (const ustring& filename)
{
  // Error and parse variables.
  GError *gerror = NULL;
  GMarkupParseContext *context = NULL;
  try {
    // Read the xml file.
    gchar *contents;
    gsize  length;
    if (!g_file_get_contents (filename.c_str(), &contents, &length, &gerror)) {
      throw runtime_error (gerror->message);
    }
    // If length is (about) zero, don't parse it, because it does not have data,
    // and parsing an empty file gives a segmentation error.
    if (length < 10)
      throw runtime_error ("File too short");  
    // Set up parser.
    GMarkupParser parser = {
      start_element_handler,
      end_element_handler,
      NULL,
      NULL,
      error_handler
    };
    // Parse xml file.
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), gpointer (this), NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      throw runtime_error (gerror->message);
    }
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      throw runtime_error (gerror->message);
    }
    g_markup_parse_context_free (context);
  }
  catch (exception & ex) {
    gw_critical (ex.what ());
    if (gerror)
      g_error_free (gerror);
    if (context)
      g_markup_parse_context_free (context);
  }
}


void BookMetrics::start_element_handler (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error)
{
  ((BookMetrics *) user_data)->start_element_handler (element_name, attribute_values);
}


void BookMetrics::end_element_handler (GMarkupParseContext *context,
                                 const gchar         *element_name,
                                 gpointer             user_data,
                                 GError             **error)
{
  ((BookMetrics *) user_data)->end_element_handler (element_name);
}


void BookMetrics::error_handler (GMarkupParseContext *context,
                                  GError              *error,
                                  gpointer             user_data)
{
  gw_critical (error->message);
}


void BookMetrics::start_element_handler (const gchar  *element_name,
                                          const gchar **attribute_values)
{
  // Get the element.
  ustring currentelement = element_name;
  // Deal with the elements, the ones that occur most first, for higher speed.
  if (currentelement == "verse") {
    parse_verses.push_back (attribute_values[0]);
  } else if (currentelement == "chapter") {
    parse_chapter = convert_to_int (attribute_values[0]);
  } else if (currentelement == "modified") {
    parse_modified = convert_to_int (attribute_values[0]);
  }
}


void BookMetrics::end_element_handler   (const gchar *element_name)
{
  // At the end of each chapte, store it, with the verses, in the map.
  ustring currentelement = element_name;
  if (currentelement == "chapter") {
    metrics[parse_chapter] = parse_verses;
    parse_verses.clear();
  }
}


vector<unsigned int> BookMetrics::get_chapters()
{
  vector<unsigned int> chapters;
  map <unsigned int, VectorUstring>::iterator iter;
  for (iter = metrics.begin(); iter != metrics.end(); iter++)
    chapters.push_back (iter->first);
  return chapters;
}


vector<ustring> BookMetrics::get_verses (unsigned int chapter)
{
  return metrics[chapter];
}
