/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "usfm.h"
#include "book.h"
#include "bible.h"
#include "usfmtools.h"
#include "dialogunicode.h"
#include "gwrappers.h"
#include "clean_usfm.h"
#include "shell.h"


Book::Book (const ustring & path, bool check_unicode, ustring encoding)
/*
When check_unicode is true we will check for conformity to Unicode UTF-8.
If it does not conform, we'll present the user with a list of possible 
conversions.
The conversion is done, the file written, and then loaded as usual.
*/
{
  // Store encoding. And do it right at the start of the object, so that it gets
  // preserved even if there was no conversion needed.
  from_encoding = encoding;

  if (check_unicode) {
    // See whether the file contents is proper Unicode
    ustring file_contents;
    bool unicode_valid = true;
    {
      gchar *s;
      if (!g_file_get_contents (path.c_str(), &s, NULL, NULL))
        unicode_valid = false;
      file_contents = s;
      g_free (s);
    }
    ustring unicode_file_contents;
    unicode_file_contents = file_contents;
    if (!unicode_file_contents.validate ())
      unicode_valid = false;
    if (!unicode_valid) {
      // Unicode is not valid: Let the user convert it to proper Unicode, and then proceed.
      // But if more than one book are imported, the user indicates with the first book
      // what encoding is wanted, and that is applied to all the following books.
      if (from_encoding.empty()) {
        UnicodeDialog unicodedialog (path);
        unicodedialog.run ();
        from_encoding = unicodedialog.encoding;
      }
      unicode_convert (path, from_encoding);
    }
  }
  load_execute (check_unicode, path);
}


Book::~Book ()
{
}


void Book::load_execute (bool gui, const ustring & path)
// Loads the new book.
// It looks at the book in file "path", and provides information on it.
{
  mypath = path;
  // Load, clean and split the lines.
  // The way to do it depends on whether we are loading a text file or a compressed file.
  ustring s;
  if (g_str_has_suffix (mypath.c_str(), ".gz")) {
    FILE *stream;
    s = "gunzip -c";
    s.append (shell_quote_space (mypath));
    stream = popen (s.c_str (), "r");
    char buf[10240];
    while (fgets (buf, sizeof (buf), stream)) {
      s = buf;
      cleaned_lines.push_back (trim (s));
    }
    pclose (stream);
  }
  else
  {
    // Plain file, just load it.
    ReadText rt (mypath, true);
    for (unsigned int i = 0; i < rt.lines.size (); i++) {
      s = rt.lines[i];
      cleaned_lines.push_back (trim (s));
    }
  }
  // Clean and split the lines according to the usfm standard.
  CleanUsfm cleanusfm (cleaned_lines);
  cleaned_lines.clear();
  cleaned_lines.assign (cleanusfm.lines.begin(), cleanusfm.lines.end());
  // Location for saving the backups of this book.
  backup_directory = gw_build_filename (g_path_get_dirname (mypath.c_str()), "backups");
  if (!g_file_test (backup_directory.c_str(), G_FILE_TEST_IS_DIR))
    create_directory (backup_directory);
}


void Book::get_chapter (unsigned int chapternumber, vector <ustring> &lines)
// This function puts chapter "chapternumber" in "lines".
{
  lines.clear ();
  unsigned int begin, end;
  get_chapter_boundaries (chapternumber, begin, end);
  for (unsigned int i = begin; i <= end; i++) {
    lines.push_back (cleaned_lines[i]);
  }
  previously_presented_chapter = chapternumber; // Used later to accept the chapter again.
}


void Book::set_chapter (vector < ustring > &lines)
{
  /*
     This function accepts the chapter in "lines", 
     and incorporates it again in the whole chapter,
     taking it as being the chapter number that was before presented.
   */
  // Get the two boundaries in the vector of strings, where the
  // chapter begins and where it ends.
  unsigned int begin, end;
  get_chapter_boundaries (previously_presented_chapter, begin, end);
  // Using an temporal vector of lines, start filling it with
  // 1. the set of lines before the current chapter starts.
  // 2. the current edited chapter.
  // 3. the lines following the current chapter.
  vector < ustring > temporal_lines;
  for (unsigned int i = 0; i < cleaned_lines.size (); i++) {
    if ((i < begin) || (i > end)) {
      temporal_lines.push_back (cleaned_lines[i]);
    } else {
      i = end;
      for (unsigned int i2 = 0; i2 < lines.size (); i2++) {
        temporal_lines.push_back (lines[i2]);
      }
    }
  }
  // Copy the temporal lines back to the book.
  cleaned_lines.clear ();
  for (unsigned int i = 0; i < temporal_lines.size (); i++) {
    cleaned_lines.push_back (temporal_lines[i]);
  }
}


bool Book::save ()
{
  // Saves the book, and returns true if all went well.
  bool all_went_well = false;
  // Only save when we have got a filename.
  if (!mypath.empty ()) {
    // Make backup of current file, if a directory is given where to put them.
//    if (!backup_directory.empty ())
    if (false) {
      // Temporarily disabled. Use date_time_utils.h for better processing without shell commands.
      // Get the date and time in the right format.
      // Shell shell0 ("date +%G-%m-%d-%H:%M-");
      // shell0.run ();
      ustring date;
      // date = trim (shell0.get_output ());
      // Get the filename to copy, and the path where to put it.
      ustring filename = g_path_get_basename (mypath.c_str());
      ustring s = date + filename;
      filename = gw_build_filename (backup_directory, s);
      // Carry out the copy command.
      ustring command = "cp" + shell_quote_space (mypath) + shell_quote_space (filename);
      system (command.c_str());
      // Compress the backup file. Force it to overwrite an existing file.
      // It is done asynchronously, so that the gui does not block during the time it takes.
      command = "gzip -f" + shell_quote_space (filename) + "&";
      system (command.c_str());
    }
    try
    {
      write_lines (mypath, cleaned_lines);
      all_went_well = true;
    }
    catch (exception & ex)
    {
      cerr << ex.what () << endl;
    }
  }
  return all_went_well;
}


void Book::get_chapter_boundaries (unsigned int chapter_number,
                                   unsigned int &begin, unsigned int &end)
// This function finds the boundaries of the chapter number requested.
{
  // Initialize the begin and end.
  begin = 0;
  end = 0;
  // Now go through all the text and start looking for the beginning and the end of the requested chapter.
  unsigned int CurrentChapter = 0;
  bool look_for_first_marker = true;
  for (unsigned int i = 0; i < cleaned_lines.size (); i++) {
    ustring s = cleaned_lines[i];
    s = usfm_extract_marker (s);
    if (CurrentChapter == 0) {
      // Allow for one-chapter books that have no \c 1 marker.
      if (usfm_is_verse (s))
        CurrentChapter = 1;
    }
    if (usfm_is_chapter (s))
      CurrentChapter = convert_to_int (number_in_string (cleaned_lines[i]));
    if (CurrentChapter == chapter_number) {
      end = i;
      if (look_for_first_marker) {
        begin = i;
        look_for_first_marker = false;
      }
    }
    // Speed up trick: break out when we have what we want.
    if (CurrentChapter > chapter_number)
      break;
  }
}


vector <ustring> Book::get_backups ()
{
  /*
     Gets all the backups that exist for this book.
   */
  // This holds the values.
  vector < ustring > backups;
  // Get all the names.
  ReadFiles
  rd (backup_directory, "", "");
  for (unsigned int i = 0; i < rd.files.size (); i++)
    {
      backups.push_back (rd.files[i]);
    }
  return backups;
}


ustring Book::get_backup_directory ()
{
  return backup_directory;
}


unsigned int Book::get_previously_presented_chapter ()
{
  return previously_presented_chapter;
}


unsigned int Book::get_id_index ()
{
  /*
     This function returns the index of the book in the IDs.
   */
  unsigned int end = 10;
  if (end > cleaned_lines.size ())
    end = cleaned_lines.size ();
  ustring marker;
  ustring line;
  for (unsigned int i = 0; i < end; i++) {
    line = cleaned_lines[i];
    marker = usfm_extract_marker (line);
    if (usfm_is_id (marker)) {
      try
      {
        ustring id = line.substr (0, 3);
        return paratext_id_to_index (id);
      }
      catch (exception & ex)
      {
        cerr << ex.what () << ": Bad ID in " << line << endl;
        return 0;
      }
    }
  }
  return 0;
}
