/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ProxyWatcherThread.h"
#include "ProxyWatcher.h"
#include <ace/config-lite.h>
#include <ace/Synch.h>
#include <stdexcept>

/*
This class is used to pass a reference to the run loop of the child thread
back to the parent thread.  The only reason we do it, is because it's
impossible to create a run loop in one thread and run it in another.
*/
class ProxyWatcherThread::ThreadSetup
{
public:
	ThreadSetup();
	
	~ThreadSetup();
	
	void setRunLoop(RefPtr<CFRunLoopRef> const& run_loop);
	
	RefPtr<CFRunLoopRef> getRunLoop();
private:
	ACE_Thread_Mutex m_mutex;
	ACE_Condition_Thread_Mutex m_condition;
	RefPtr<CFRunLoopRef> m_ptrRunLoop;
};


ProxyWatcherThread::ProxyWatcherThread()
:	m_ptrThreadSetup(new ThreadSetup)
{
	int const status = activate();
	if (status == -1) {
		throw std::runtime_error("Spawning ProxyWatcher thread failed");
	}
	m_ptrRunLoop = m_ptrThreadSetup->getRunLoop();
	m_ptrThreadSetup.reset(0); // we don't need it any more
}

ProxyWatcherThread::~ProxyWatcherThread()
{
	if (m_ptrRunLoop) {
		CFRunLoopStop(m_ptrRunLoop);
	}
	wait();
}

int
ProxyWatcherThread::svc()
{
	RefPtr<CFRunLoopRef> run_loop(CFRunLoopGetCurrent(), RETAIN);
	m_ptrThreadSetup->setRunLoop(run_loop);
	RefPtr<CFStringRef> client_name(CFSTR("BFilter ProxyWatcher"), RETAIN);
	ProxyWatcher watcher(client_name, run_loop);
	CFRunLoopRun();
	return 0;
}


/*==================== ProxyWatcherThread::ThreadSetup =====================*/

ProxyWatcherThread::ThreadSetup::ThreadSetup()
:	m_condition(m_mutex)
{
}

ProxyWatcherThread::ThreadSetup::~ThreadSetup()
{
}

void
ProxyWatcherThread::ThreadSetup::setRunLoop(RefPtr<CFRunLoopRef> const& run_loop)
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	m_ptrRunLoop = run_loop;
	m_condition.signal();
}

RefPtr<CFRunLoopRef>
ProxyWatcherThread::ThreadSetup::getRunLoop()
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	if (!m_ptrRunLoop) {
		m_condition.wait();
	}
	return m_ptrRunLoop;
}
