/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "LocationUpdater.h"
#include "AssertException.h"
#include "CFUtils.h"
#include "StringLess.h"
#include "OutputFunction.h"
#include <map>
#include <vector>
#include <utility>

using namespace std;
	
LocationUpdater::LocationUpdater()
:	NetworkPrefs(CFSTR("BFilter Installer")),
	m_bfPrefix(CFSTR("(BF) "))
{
}

LocationUpdater::~LocationUpdater()
{
}

void
LocationUpdater::updateLocations(int listen_port)
{
	// Duplicate location names are theoretically supported.
	typedef multimap<StringRef, StringRef, StringLess> Sets;
	Sets sets; // name => id
	
	DECLARE_INSERTER(
		SetInserter, StringPair, Sets,
		c.insert(std::make_pair(val.second, val.first))
	);
	SetInserter set_inserter(sets);
	enumSets(set_inserter);
	int const num_sets = sets.size();
	
	while (!sets.empty()) {
		Sets::iterator it = sets.begin();
		StringRef name = it->first;
		StringRef set_id = it->second;
		sets.erase(it);
		
		StringRef orig_name = CFUtils::removeStringPrefix(name, m_bfPrefix);
		if (orig_name == name) { // no prefix
			MutableStringRef prefixed_name(
				CFStringCreateMutableCopy(NULL, 0, m_bfPrefix),
				NORETAIN
			);
			CFStringAppend(prefixed_name, name);
			StringRef bf_set_id;
			Sets::iterator bf_it = sets.find(prefixed_name);
			if (bf_it != sets.end()) {
				bf_set_id = bf_it->second;
				sets.erase(bf_it);
			}
			handleLocationPair(
				set_id, name, bf_set_id, prefixed_name, listen_port
			);
		} else { // (BF) location
			StringRef orig_set_id;
			Sets::iterator orig_it = sets.find(orig_name);
			if (orig_it != sets.end()) {
				orig_set_id = orig_it->second;
				sets.erase(orig_it);
			}
			if (num_sets == 1) {
				// We have just one location, and it has a (BF) prefix.
				// Normally we would delete it, as it has no counterpart,
				// but not in this case.
			} else {
				handleLocationPair(
					orig_set_id, orig_name, set_id, name, listen_port
				);
			}
		}
	}
	
	switchCurrentSetIfRemoved();
}

void
LocationUpdater::handleLocationPair(
	StringRef const& orig_set_id, StringRef const& orig_set_name,
	StringRef const& bf_set_id, StringRef const& bf_set_name, int listen_port)
{
	if (bf_set_id) {
		deleteSet(bf_set_id);
	}
	if (orig_set_id) {
		createBfLocation(listen_port, orig_set_id, bf_set_name, bf_set_id);
	}
}

void
LocationUpdater::createBfLocation(
	int listen_port, StringRef const& orig_set_id,
	StringRef const& new_set_name, StringRef const& new_set_id_hint)
{
	StringRef new_set_id = duplicateSet(orig_set_id, new_set_name, new_set_id_hint);
	
	typedef vector<StringRef> ServicePaths;
	ServicePaths service_paths;
	
	DECLARE_INSERTER(Inserter, StringPair, ServicePaths, c.push_back(val.second));
	Inserter inserter(service_paths);
	enumServicesReferencedBySet(new_set_id, inserter);
	
	vector<StringRef>::iterator it = service_paths.begin();
	vector<StringRef>::iterator const end = service_paths.end();
	for (; it != end; ++it) {
		modifyServiceForBF(*it, listen_port);
	}
}

void
LocationUpdater::modifyServiceForBF(
	StringRef const& service_path, int listen_port)
{
	StringRef proxies_path = CFUtils::composePath(
		service_path,  kSCEntNetProxies, 0
	);
	
	DictRef orig_proxies(
		SCPreferencesPathGetValue(getRep(), proxies_path),
		RETAIN
	);
	if (!orig_proxies || CFGetTypeID(orig_proxies) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	MutableDictRef new_proxies = MutableDictRef(
		CFDictionaryCreateMutableCopy(NULL, 0, orig_proxies),
		NORETAIN
	);
	
	int const int_zero = 0;
	int const int_one = 1;
	NumberRef disabled(
		CFNumberCreate(NULL, kCFNumberIntType, &int_zero),
		NORETAIN
	);
	NumberRef enabled(
		CFNumberCreate(NULL, kCFNumberIntType, &int_one),
		NORETAIN
	);
	NumberRef http_port(
		CFNumberCreate(NULL, kCFNumberIntType, &listen_port),
		NORETAIN
	);
	
	CFDictionarySetValue(new_proxies, kSCPropNetProxiesHTTPProxy, CFSTR("127.0.0.1"));
	CFDictionarySetValue(new_proxies, kSCPropNetProxiesHTTPPort, http_port);
	CFDictionarySetValue(new_proxies, kSCPropNetProxiesHTTPEnable, enabled);
	CFDictionaryRemoveValue(new_proxies, kSCPropNetProxiesExceptionsList);
	CFDictionaryRemoveValue(new_proxies, CFSTR("ExcludeSimpleHostnames"));
	CFDictionarySetValue(new_proxies, CFSTR("ProxyAutoDiscoveryEnable"), disabled);
	CFDictionarySetValue(new_proxies, CFSTR("ProxyAutoConfigEnable"), disabled);
	

	if (!SCPreferencesPathSetValue(getRep(), proxies_path, new_proxies)) {
		throw AssertException(__FILE__, __LINE__);
	}
}
