/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ClientRequestDescriptor.h"
#include "DebugMessage.h"
#include "DebugConnection.h"
#include <limits>
#include <algorithm>
#include <cassert>

using namespace std;

namespace GtkGUI
{

ClientRequestDescriptor::ClientRequestDescriptor(
	Glib::ustring const& url,
	ListAppendFunctor const& list_append_functor,
	ListRemoveFunctor const& list_remove_functor,
	ListSelectFunctor const& list_select_functor)
:	m_url(url),
	m_listAppendFunctor(list_append_functor),
	m_listRemoveFunctor(list_remove_functor),
	m_listSelectFunctor(list_select_functor),
	m_requestListIter(),
	m_pFirstMessage(0),
	m_pLastMessage(0)
{
}

ClientRequestDescriptor::~ClientRequestDescriptor()
{
	if (m_requestListIter) {
		m_listRemoveFunctor(m_requestListIter);
	}
	
	assert(!m_pFirstMessage && !m_pLastMessage);
	// Messages hold an IntrusivePtr to ClientRequestDescriptor
}

void
ClientRequestDescriptor::registerMessage(DebugMessage* msg)
{
	if (!m_pFirstMessage) {
		msg->setPrev(0);
		msg->setNext(0);
		m_pFirstMessage = msg;
		m_pLastMessage = msg;
		
		assert(!m_requestListIter);
		m_requestListIter = m_listAppendFunctor(
			m_url,
			sigc::mem_fun(
				*this,
				&ClientRequestDescriptor::onListSelectionChange
			)
		);
		return;
	}

	msg->setPrev(m_pLastMessage);
	msg->setNext(0);
	m_pLastMessage->setNext(msg);
	m_pLastMessage = msg;
}

void
ClientRequestDescriptor::unregisterMessage(DebugMessage* msg)
{
	if (msg == m_pFirstMessage) {
		m_pFirstMessage = msg->getNext();
	} else {
		msg->getPrev()->setNext(msg->getNext());
	}
	if (msg == m_pLastMessage) {
		m_pLastMessage = msg->getPrev();
	} else {
		msg->getNext()->setPrev(msg->getPrev());
	}
	msg->setPrev(0);
	msg->setNext(0);
	
	if (!m_pFirstMessage && m_requestListIter) {
		m_listRemoveFunctor(m_requestListIter);
		m_requestListIter = Gtk::TreeIter();
	}
}

void
ClientRequestDescriptor::onMessageSelected()
{
	DebugMessage* msg = m_pFirstMessage;
	for (; msg; msg = msg->getNext()) {
		msg->select();
	}
	if (m_requestListIter) {
		m_listSelectFunctor(m_requestListIter);
	}
}

void
ClientRequestDescriptor::onListSelectionChange(bool selected)
{
	if (!m_pFirstMessage) {
		return;
	} else if (selected && m_pFirstMessage->isSelected()) {
		// This means we've been called as a result of onMessageSelected.
		// The messages are already selected, and we don't want
		// to scroll the layout when a user clicks on a message.
		return;
	}
	
	int top = std::numeric_limits<int>::max();
	int bottom = std::numeric_limits<int>::min();
	
	DebugMessage* msg = m_pFirstMessage;
	for (; msg; msg = msg->getNext()) {
		top = std::min(top, msg->getTop());
		bottom = std::max(bottom, msg->getBottom());
		if (selected) {
			msg->select();
		} else {
			msg->unselect();
		}
	}
	
	if (selected) {
		m_pFirstMessage->getConnection()->showMessagesInRange(top, bottom);
	}
}

} // namespace GtkGUI
