/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ObjectStorage.h"
#include "NonCopyable.h"
#include "RequestResolution.h"
#include "AbstractFileStorage.h"
#include "FileStorage.h"
#include "FileId.h"
#include "Item.h"
#include "IntrusivePtr.h"
#include "AbstractCommand.h"
#include "HttpRequestMetadata.h"
#include "BString.h"
#include "URI.h"
#include "HashValue.h"
#include "SHA1.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/Synch.h>
#include <ace/Singleton.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <vector>
#include <iterator> // for back_inserter()
#include <exception>
#include <assert.h>

namespace HttpCache
{

using namespace boost::multi_index;

class ObjectStorage::Impl : public ObjectStorage, public AbstractCommand
{
	DECLARE_NON_COPYABLE(Impl)
public:
	Impl();
	
	virtual ~Impl();
	
	virtual void setCacheSize(uint64_t size);
	
	virtual bool setCacheDir(std::string const& cache_dir);
	
	virtual RequestResolution handleRequest(
		ConstRequestPtr const& request, bool dont_create);
	
	void orphan();
private:
	enum {
		ITEMS_HIGH_WATERMARK = 2000, /**< Launch garbage collection when
		item count reaches this point. */
		ITEMS_LOW_WATERMARK = 1500 /**< Stop garbage collection when
		item count reaches this point. */
	};
	
	typedef ACE_Thread_Mutex Mutex;
	typedef SHA1 Hash;
	typedef std::map<Hash::Value, IntrusivePtr<AbstractItem> > Map;
	
	struct Entry
	{
		Hash::Value hash;
		IntrusivePtr<AbstractItem> item;
		
		Entry(Hash::Value const& h, IntrusivePtr<AbstractItem> const& it)
		: hash(h), item(it) {}
	};
	
	class HashTag;
	class SequenceTag;
	
	typedef multi_index_container<
		Entry,
		indexed_by<
			ordered_unique<
				tag<HashTag>,
				member<Entry, Hash::Value, &Entry::hash>
			>,
			sequenced<
				tag<SequenceTag>
			>
		>
	> Container;
	typedef Container::index<HashTag>::type HashIdx;
	typedef Container::index<SequenceTag>::type SequenceIdx;
	
	virtual void operator()();
	
	void garbageCollectST();
	
	static Hash::Value hashUrl(URI const& url);
	
	Mutex m_mutex;
	Container m_items;
	IntrusivePtr<AbstractFileStorage> m_ptrStorage;
	uint64_t m_cacheSize;
};


class ObjectStorage::ImplRef
{
	DECLARE_NON_COPYABLE(ImplRef)
public:
	ImplRef() : m_ptrImpl(new Impl) {}
	
	~ImplRef() { m_ptrImpl->orphan(); }
	
	Impl* get() const { return m_ptrImpl.get(); }
private:
	IntrusivePtr<ObjectStorage::Impl> m_ptrImpl;
};


ObjectStorage*
ObjectStorage::instance()
{
	return ACE_Singleton<ImplRef, ACE_Recursive_Thread_Mutex>::instance()->get();
}

ObjectStorage::Impl::Impl()
:	m_cacheSize(0)
{
}

ObjectStorage::Impl::~Impl()
{
}

void
ObjectStorage::Impl::setCacheSize(uint64_t const size)
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	
	m_cacheSize = size;
	if (AbstractFileStorage* storage = m_ptrStorage.get()) {
		storage->spaceManager()->setSpaceLimit(size);
	}
}

bool
ObjectStorage::Impl::setCacheDir(std::string const& cache_dir)
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	
	if (m_ptrStorage.get() || m_cacheSize == 0) {
		return false;
	}
	
	try {
		m_ptrStorage = FileStorage::create(
			cache_dir, m_cacheSize, IntrusivePtr<AbstractCommand>(this)
		);
	} catch (std::exception&) {
		return false;
	}
	
	Container new_items;
	
	std::vector<FileId> ids;
	m_ptrStorage->spaceManager()->listKeys(std::back_inserter(ids));
	
	std::vector<FileId>::const_iterator it(ids.begin());
	std::vector<FileId>::const_iterator const end(ids.end());
	
	// Note: FileId's are ordered by key ASC, version DESC.
	while (it != end && new_items.size() <= ITEMS_HIGH_WATERMARK) {
		bool file_ok = false;
		IntrusivePtr<AbstractItem> const item(
			Item::create(m_ptrStorage, *it, file_ok)
		);
		if (!file_ok) {
			m_ptrStorage->unlink(*it);
			++it;
			continue;
		}
		
		new_items.insert(Entry(it->getKey(), item));
		
		// Delete the remaining versions of this file.
		FileId::Key const& key(it->getKey());
		++it;
		for (; it != end && it->getKey() == key; ++it) {
			m_ptrStorage->unlink(*it);
		}
	}
	
	// Delete the remaing files (in case we've reached ITEMS_HIGH_WATERMARK)
	for (; it != end; ++it) {
		m_ptrStorage->unlink(*it);
	}
	
	m_items.swap(new_items);
	return true;
}

RequestResolution
ObjectStorage::Impl::handleRequest(
	ConstRequestPtr const& request, bool const dont_create)
{
	Hash::Value const hash(hashUrl(request->requestLine().getURI()));
	IntrusivePtr<AbstractItem> item;
	
	{
		ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
		
		if (!m_ptrStorage.get()) {
			return RequestResolution();
		}
		
		SequenceIdx& seq_idx = m_items.get<SequenceTag>();
		HashIdx& hash_idx = m_items.get<HashTag>();
		HashIdx::iterator it(hash_idx.lower_bound(hash));
		if (it != hash_idx.end() && it->hash == hash) {
			item = it->item;
			seq_idx.relocate(seq_idx.end(), m_items.project<SequenceTag>(it));
		} else {
			if (dont_create) {
				return RequestResolution();
			} else {
				if (m_items.size() >= ITEMS_HIGH_WATERMARK) {
					garbageCollectST();
					if (m_items.size() >= ITEMS_HIGH_WATERMARK) {
						// Too many items in use?
						return RequestResolution();
					}
				}
				bool file_ok = false;
				FileId const file_id(hash, 0);
				item = Item::create(m_ptrStorage, file_id, file_ok);
				hash_idx.insert(it, Entry(hash, item));
			}
		}
	} // guard scope
	
	assert(item.get());
	return item->handleRequest(request);
}

void
ObjectStorage::Impl::orphan()
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	
	// It's very important to do so.  Otherwise we get circular references
	// through LimitedSpaceManager which holds "gc_invoker" reference to us. 
	m_items.clear();
	m_ptrStorage.reset(0);
}

/**
 * \brief AbstractCommand action.
 *
 * This gets called by LimitedSpaceManager when we are out of space.
 */
void
ObjectStorage::Impl::operator()()
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	
	size_t todo = static_cast<size_t>(m_items.size() * 0.2);
	SequenceIdx& idx = m_items.get<SequenceTag>();
	SequenceIdx::iterator it(idx.begin());
	SequenceIdx::iterator const end(idx.end());
	while (todo && it != end) {
		if (it->item->prepareForGC()) {
			idx.erase(it++);
			--todo;
		} else {
			++it;
		}
	}
}

void
ObjectStorage::Impl::garbageCollectST()
{
	SequenceIdx& idx = m_items.get<SequenceTag>();
	
	{
		// Remove non-cached items only, starting from newer ones.
		SequenceIdx::iterator it(idx.end());
		while (m_items.size() > ITEMS_LOW_WATERMARK && it != idx.begin()) {
			SequenceIdx::iterator eff_it(it);
			--eff_it;
			if (eff_it->item->prepareForGC(AbstractItem::GC_ONLY_UNCACHED)) {
				idx.erase(eff_it);
			} else {
				--it;
			}
		}
	}
	
	{
		// If we are still not done, remove any items except those in use.
		SequenceIdx::iterator it(idx.begin());
		SequenceIdx::iterator const end(idx.end());
		while (m_items.size() > ITEMS_LOW_WATERMARK && it != end) {
			if (it->item->prepareForGC()) {
				idx.erase(it++);
			} else {
				++it;
			}
		}
	}
}

ObjectStorage::Impl::Hash::Value
ObjectStorage::Impl::hashUrl(URI const& url)
{
	BString const& str = url.toBString();
	Hash hash;
	hash.feed((unsigned char const*)str.begin(), str.size());
	Hash::Value hval;
	hash.finalize(hval);
	return hval;
}

} // namespace HttpCache
