/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ErrorFactory.h"
#include "ErrorDescriptor.h"
#include "ImmediateResponse.h"
#include "CraftedResponse.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "BString.h"
#include "Date.h"
#include "URI.h"
#include "StringUtils.h"
#include "HttpStatusLine.h"
#include "ArraySize.h"
#include "HtmlEscaper.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_netdb.h>
#include <ace/OS_NS_unistd.h>
#include <stddef.h>
#include <cstring>

using namespace std;

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errConnectionFailed(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const host, int const port,
	char const* const syserr)
{
	static char const* templ[] =
#	include "ERR_CONNECT_FAIL.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@HOST@")] = host;
	m[BString("@PORT@")] = StringUtils::fromNumber(port);
	m[BString("@SYSERR@")] = syserr;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errConnectionTimeout(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& host, int const port)
{
	static char const* templ[] =
#	include "ERR_CONNECT_TIMEOUT.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@HOST@")] = host;
	m[BString("@PORT@")] = StringUtils::fromNumber(port);
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_GATEWAY_TIMEOUT, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errHostNameResolution(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& host)
{
	static char const* templ[] =
#	include "ERR_HOST_NAME_RESOLUTION.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@HOST@")] = host;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errServerTimeout(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url)
{
	static char const* templ[] =
#	include "ERR_SERVER_TIMEOUT.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_GATEWAY_TIMEOUT, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errParsingRequest(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& problem)
{
	static char const* templ[] =
#	include "ERR_PARSING_REQUEST.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@PROBLEM@")] = problem;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_BAD_REQUEST, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errParsingResponse(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& problem)
{
	static char const* templ[] =
#	include "ERR_PARSING_RESPONSE.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@PROBLEM@")] = problem;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errUnsupportedProtocol(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& protocol)
{
	static char const* templ[] =
#	include "ERR_UNSUPPORTED_PROTOCOL.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@PROTOCOL@")] = protocol;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_NOT_IMPLEMENTED, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errUnsupportedMethod(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& method)
{
	static char const* templ[] =
#	include "ERR_UNSUPPORTED_METHOD.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@METHOD@")] = method;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_NOT_IMPLEMENTED, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errExpectationFailed(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& expectation)
{
	static char const* templ[] =
#	include "ERR_EXPECTATION_FAILED.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@EXPECTATION@")] = expectation;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_EXPECTATION_FAILED, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errGenericError(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& error, int status_code)
{
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			std::auto_ptr<ImmediateResponse>(
				getGenericError(url, error, status_code)
			)
		)
	);
}

std::auto_ptr<CraftedResponse>
ErrorFactory::getGenericError(
	URI const& url, std::string const& error, int status_code)
{
	static char const* templ[] =
#	include "ERR_GENERIC.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@ERROR@")] = error;
	
	return prepareCraftedResponse(status_code, templ, m);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errUrlForbidden(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url)
{
	static char const* templ[] =
#	include "ERR_URL_FORBIDDEN.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_FORBIDDEN, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errRemoteProxyConfig(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& problem)
{
	static char const* templ[] =
#	include "ERR_REMOTE_PROXY_CONFIG.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@PROBLEM@")] = problem;
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_INTERNAL_SERVER_ERROR, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errForwardingThroughProxy(
	ErrorCodes::Code const code, std::string const& debug_msg,
	URI const& url, std::string const& problem,
	std::string const& proxy_type, std::string const& proxy_host,
	int proxy_port)
{
	static char const* templ[] =
#	include "ERR_FORWARDING_THROUGH_PROXY.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url.toString();
	m[BString("@PROBLEM@")] = problem;
	m[BString("@PROXY_TYPE@")] = proxy_type;
	m[BString("@PROXY_HOST@")] = proxy_host;
	m[BString("@PROXY_PORT@")] = StringUtils::fromNumber(proxy_port);
	
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(
			code, debug_msg,
			prepareResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE, templ, m)
		)
	);
}

std::auto_ptr<ErrorDescriptor>
ErrorFactory::errDummyError(
	ErrorCodes::Code const code, std::string const& debug_msg)
{
	std::auto_ptr<ImmediateResponse> response(
		new CraftedResponse(HttpStatusLine::SC_INTERNAL_SERVER_ERROR)
	);
	return std::auto_ptr<ErrorDescriptor>(
		new ErrorDescriptor(code, debug_msg, response)
	);
}


void
ErrorFactory::prepareCommonMacros(MacroMap& m)
{
	m[BString("@DATE@")] = Date::formatCurrentTime().toStdString();
	m[BString("@HOSTNAME@")] = getFullHostName();
	m[BString("@VERSION@")] = BFILTER_VERSION;
}

void
ErrorFactory::prepareContent(
	CraftedResponse& response, char const* const templ[], MacroMap const& macros)
{
	BString const empty;
	SBOutStream strm;
	for (char const* str; (str = *templ); ++templ) {
		size_t len = strlen(str);
		if (len >= 2 && str[0] == '@' && str[len-1] == '@') {
			MacroMap::const_iterator fit =
				macros.find(BString(empty, str, str + len));
			if (fit != macros.end()) {
				strm << HtmlEscaper::escape(fit->second);
			}
		} else {
			strm.write(str, len);
		}
	}

	HttpHeadersCollection& headers = response.metadata().headers();
	headers.setHeader(BString("Content-Type"), BString("text/html"));
	headers.setHeader(BString("Cache-Control"), BString("no-cache, must-revalidate"));
	strm.swapData(response.body());
}

std::auto_ptr<ImmediateResponse>
ErrorFactory::prepareResponse(
	int const status_code, char const* const templ[],
	MacroMap const& macros)
{
	return std::auto_ptr<ImmediateResponse>(
		prepareCraftedResponse(status_code, templ, macros)
	);
}

std::auto_ptr<CraftedResponse>
ErrorFactory::prepareCraftedResponse(
	int const status_code, char const* const templ[],
	MacroMap const& macros)
{
	std::auto_ptr<CraftedResponse> response(
		new CraftedResponse(status_code)
	);
	prepareContent(*response, templ, macros);
	return response;
}

std::string
ErrorFactory::getFullHostName()
{
	char hname[MAXHOSTNAMELEN + 1];
	if (ACE_OS::hostname(hname, ARRAY_SIZE(hname)) < 0) {
		return string();
	}
	
	struct hostent h_ent, *h_entp;
	ACE_HOSTENT_DATA buf;
	int h_err = 0;
	h_entp = ACE_OS::gethostbyname_r(hname, &h_ent, buf, &h_err);
	if (!h_entp || h_err < 0) {
		return hname;
	}
	
	return std::string(h_entp->h_name);
}
