/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Log.h"
#include "StyledText.h"
#include <sigc++/sigc++.h>

using namespace std;

namespace GtkGUI
{

Log::Log()
:	m_numRecords(0),
	m_numVisibleViews(0),
	m_haveUnseenWarnings(false),
	m_visibilityReporter(sigc::mem_fun(*this, &Log::onViewVisibilityChange)),
	m_ptrBuffer(Gtk::TextBuffer::create()),
	m_ptrDefaultStyle(m_ptrBuffer->create_tag()),
	m_ptrSuccessStyle(m_ptrBuffer->create_tag()),
	m_ptrWarningStyle(m_ptrBuffer->create_tag()),
	m_ptrErrorStyle(m_ptrBuffer->create_tag())
{
	m_ptrSuccessStyle->property_foreground() = "#38893b";
	m_ptrSuccessStyle->property_weight() = PANGO_WEIGHT_BOLD;
	
	m_ptrWarningStyle->property_foreground() = "#f41d00";
	
	m_ptrErrorStyle->property_foreground() = "#ff0000";
	m_ptrErrorStyle->property_weight() = PANGO_WEIGHT_BOLD;
}

Log::~Log()
{
}

void
Log::appendRecord(GtkGUI::StyledText const& text)
{
	appendRecord(text.text(), text.style());
}

void
Log::appendRecord(Glib::ustring const& text)
{
	appendRecord(text, m_ptrDefaultStyle);
}

void
Log::appendRecord(Glib::ustring const& text, StylePtr const& style)
{
	++m_numRecords;
	appendText(text, style, m_ptrBuffer->size() != 0);
}

void
Log::appendToLastRecord(GtkGUI::StyledText const& text)
{
	appendToLastRecord(text.text(), text.style());
}

void
Log::appendToLastRecord(Glib::ustring const& text)
{
	appendToLastRecord(text, m_ptrDefaultStyle);
}

void
Log::appendToLastRecord(Glib::ustring const& text, StylePtr const& style)
{
	appendText(text, style, false);
}

void
Log::clear()
{
	m_ptrBuffer->set_text(Glib::ustring());
	m_numRecords = 0;
	
	if (m_haveUnseenWarnings) {
		m_haveUnseenWarnings = false;
		m_unseenWarningsSignal(false);
	}
}

void
Log::appendText(
	Glib::ustring const& text, StylePtr const& style,
	bool prepend_newline)
{
	if (prepend_newline) {
		char const nl = '\n';
		m_ptrBuffer->insert(m_ptrBuffer->end(), &nl, &nl + 1);
	}
	m_ptrBuffer->insert_with_tag(m_ptrBuffer->end(), text, style);
	
	if (style == m_ptrErrorStyle || style == m_ptrWarningStyle) {
		if (!m_haveUnseenWarnings && m_numVisibleViews == 0) {
			m_haveUnseenWarnings = true;
			m_unseenWarningsSignal(true);
		}
	}
}

void
Log::onViewVisibilityChange(bool visible)
{
	if (!visible) {
		--m_numVisibleViews;
	} else {
		if (m_numVisibleViews++ == 0) {
			if (m_haveUnseenWarnings) {
				m_haveUnseenWarnings = false;
				m_unseenWarningsSignal(false);
			}
		}
	}
}

} // namespace GtkGUI
