/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FileOps.h"
#include "FileVersion.h"
#include "AutoClosingSAP.h"
#include "OperationLog.h"
#include "Log.h"
#include "EncOps.h"
#include <ace/config-lite.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_IO.h>
#include <ace/FILE_Connector.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/OS_NS_sys_stat.h> // for stat(), fstat(), mkdir()
#include <ace/OS_NS_stdio.h> // for rename()
#include <ace/OS_NS_unistd.h> // for unlink()
#include <errno.h>

namespace GtkGUI
{

namespace FileOps
{

bool readFile(std::string const& file_path, std::string& target,
	FileVersion* version, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(file_path.c_str());
	ACE_FILE_Connector connector;
	
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, O_RDONLY) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Could not open "+EncOps::filenameToUtf8(file_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	std::string content;
	
	char buf[4096];
	ssize_t bytes_read = 0;
	while ((bytes_read = file.recv(buf, sizeof(buf))) > 0) {
		content.append(buf, bytes_read);
	}
	if (bytes_read != 0) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error reading "+EncOps::filenameToUtf8(file_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	target.swap(content);
	
	if (version) {
		ACE_stat st;
		if (ACE_OS::fstat(file.get_handle(), &st) == -1) {
			*version = FileVersion::undefined();
		} else {
			*version = FileVersion(st.st_mtime);
		}
	}
	
	return true;
}

bool writeFile(std::string const& file_path, std::string const& data,
	FileVersion* version, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(file_path.c_str());
	ACE_FILE_Connector connector;
	
	int const flags = O_WRONLY|O_CREAT|O_TRUNC;
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, flags) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Could not open "+EncOps::filenameToUtf8(file_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (file.send_n(data.c_str(), data.size()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error writing "+EncOps::filenameToUtf8(file_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (version) {
		ACE_stat st;
		if (ACE_OS::fstat(file.get_handle(), &st) == -1) {
			*version = FileVersion::undefined();
		} else {
			*version = FileVersion(st.st_mtime);
		}
	}
	
	return true;
}

bool renameFile(std::string const& from_path,
	std::string const& to_path, bool log_errors)
{
	if (ACE_OS::rename(from_path.c_str(), to_path.c_str()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error renaming "+EncOps::filenameToUtf8(from_path)+
				" to "+EncOps::filenameToUtf8(to_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool deleteFile(std::string const& file_path, bool log_errors)
{
	if (ACE_OS::unlink(file_path.c_str()) == -1 && errno != ENOENT) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error deleting "+EncOps::filenameToUtf8(file_path),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

FileVersion readFileVersion(std::string const& file_path)
{
	ACE_stat st;
	if (ACE_OS::stat(file_path.c_str(), &st) == -1) {
		return FileVersion::undefined();
	} else {
		return FileVersion(st.st_mtime);
	}
}

bool createDir(std::string const& dir_path, bool log_errors)
{
	bool ok = true;
	
	if (ACE_OS::mkdir(dir_path.c_str()) == -1) {
		if (errno != EEXIST) {
			ok = false;
		} else {
			ACE_stat st;
			if (ACE_OS::stat(dir_path.c_str(), &st) == -1) {
				ok = false;
			} else if (!(st.st_mode & S_IFDIR)) {
				ok = false;
			}
		}
	}
	
	if (!ok && log_errors) {
		Log* log = OperationLog::instance();
		log->appendRecord(
			"Could not create directory "+EncOps::filenameToUtf8(dir_path),
			log->getErrorStyle()
		);
	}
	
	return ok;
}

} // namespace FileOps

} // namespace GtkGUI
