/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConfigFile.h"
#include "FileVersion.h"
#include "ConfigErrorHandler.h"
#include "RefCountableSAP.h"
#include "IntrusivePtr.h"
#include "OperationLog.h"
#include "Log.h"
#include "GlobalState.h"
#include "Conf.h"
#include "WorkerThreadPoolExSingleton.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include "DnsResolver.h"
#include "FileOps.h"
#include <ace/config-lite.h>
#include <ace/SOCK_Acceptor.h>
#include <list>
#include <vector>
#include <sstream>
#include <stddef.h>
#include <time.h>

namespace GtkGUI
{

ConfigFile::ConfigFile(std::string const& file_path)
:	m_filePath(file_path)
{
}

ConfigFile::~ConfigFile()
{
}

bool
ConfigFile::loadAndForceApply()
{
	Log* const log = OperationLog::instance();
	
	log->appendRecord("Loading config ... ");
	size_t const num_records = log->getNumRecords();
	
	std::string text;
	FileVersion version;
	if (!FileOps::readFile(m_filePath, text, &version)) {
		return false;
	}
	
	Config new_config;
	ConfigFileStructure new_structure(m_fileStructure);
	ConfigErrorHandler eh("config");
	new_structure.load(text, new_config, eh);
	
	rebind(new_config);
	
	{
		m_fileStructure.swap(new_structure);
		GlobalState::WriteAccessor global_state;
		global_state->swapConfig(new_config);
		global_state->setConfigFileVersion(version);
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	return true;
}

bool
ConfigFile::applyAndSave(std::string const& text)
{
	Log* const log = OperationLog::instance();
	
	log->appendRecord("Applying config ... ");
	size_t const num_records = log->getNumRecords();
	
	Config new_config;
	ConfigFileStructure new_structure(m_fileStructure);
	ConfigErrorHandler eh("config");
	new_structure.load(text, new_config, eh);
	if (eh.numErrors() != 0) {
		return false;
	}
	
	if (!applySave(new_config, new_structure, text)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	return true;
}

bool
ConfigFile::applyAndSave(Config const& config)
{
	Log* const log = OperationLog::instance();
	
	log->appendRecord("Applying config ... ");
	size_t const num_records = log->getNumRecords();
	
	Config new_config(config);
	ConfigFileStructure new_structure(m_fileStructure);
	new_structure.updateWith(new_config);
	
	std::string text;
	
	{
		std::ostringstream strm;
		new_structure.toStream(strm);
		text = strm.str();
	}
	
	if (!applySave(new_config, new_structure, text)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	return true;
}

FileVersion
ConfigFile::readFileVersion() const
{
	return FileOps::readFileVersion(m_filePath);
}

bool
ConfigFile::applySave(Config& new_config,
	ConfigFileStructure& new_structure, std::string const& text)
{
	FileVersion version;
	if (!rebind(new_config)
	    || !FileOps::writeFile(m_filePath, text, &version)) {
		Config const config(GlobalState::ReadAccessor()->config());
		rebindWithMessage(config, "Re-applying the old config ... ");
		return false;
	}
	
	m_fileStructure.swap(new_structure);
	GlobalState::WriteAccessor global_state;
	global_state->swapConfig(new_config);
	global_state->setConfigFileVersion(version);
	return true;
}

bool
ConfigFile::rebind(Config const& config)
try {
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	Log* log = OperationLog::instance();
	
	WorkerThreadPoolExSingleton::instance()->removeAllAcceptors();
	
	bool ok = true;
	std::vector<AcceptorPtr> acceptors;
	std::list<SymbolicInetAddr> addrs(config.getListenAddresses());
	for (; !addrs.empty(); addrs.pop_front()) {
		SymbolicInetAddr const& addr = addrs.front();
		std::vector<InetAddr> resolved_addrs(DnsResolver::resolve(addr));
		if (resolved_addrs.empty()) {
			std::ostringstream err;
			err << "Could not resolve listen address \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addrs[0], true) == -1) {
			std::ostringstream err;
			err << "Could not bind to \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		log->appendRecord(
			"No addresses to listen on!",
			log->getErrorStyle()
		);
		return false;
	}
	
	std::vector<AcceptorPtr>::const_iterator it(acceptors.begin());
	std::vector<AcceptorPtr>::const_iterator const end(acceptors.end());
	for (; it != end; ++it) {
		try {
			WorkerThreadPoolExSingleton::instance()->addAcceptor(*it);
		} catch (...) {
			log->appendRecord(
				"Could not register a connection acceptor",
				log->getErrorStyle()
			);
			return false;
		}
	}
	
	return true;
} catch (...) {
	WorkerThreadPoolExSingleton::instance()->removeAllAcceptors();
	throw;
}

bool
ConfigFile::rebindWithMessage(Config const& config, char const* msg)
{
	Log* log = OperationLog::instance();
	log->appendRecord(msg);
	size_t num_records = log->getNumRecords();
	
	bool const res = rebind(config);
	
	if (res && num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	return res;
}

} // namespace GtkGUI

