/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Item.h"
#include "FileInstance.h"
#include "AbstractResponseReader.h"
#include "AbstractResponseWriter.h"
#include "AbstractResponseHandler.h"
#include "AbstractFileStorage.h"
#include "AbstractFileIO.h"
#include "MemoryReader.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpStatusLine.h"
#include "HttpVersion.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "RequestCacheControl.h"
#include "ResponseCacheControl.h"
#include "RequestResolution.h"
#include "ResponseResolution.h"
#include "RequestPtr.h"
#include "ErrorFactory.h"
#include "CraftedResponse.h"
#include "Date.h"
#include "BString.h"
#include "DataChunk.h"
#include "SplittableBuffer.h"
#include "StringUtils.h"
#include "InsensitiveEqual.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_stdio.h> // for ACE_OS::snprintf()
#include <ace/OS_NS_unistd.h>
#include <algorithm>
#include <list>
#include <sstream>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <assert.h>

namespace HttpCache
{

namespace detail {}
using namespace detail;

namespace detail
{

class ResponseHandler : public AbstractResponseHandler
{
public:
	ResponseHandler(IntrusivePtr<AccessorHelper> const& helper,
		ConstRequestPtr const& request,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		uint64_t body_length, uint32_t version, std::auto_ptr<FileInstance>& file);
	
	virtual ~ResponseHandler();
	
	virtual ConstRequestPtr const& getOutgoingRequest() const {
		return m_ptrRequest;
	}
	
	virtual ResponseResolution handleResponse(
		HttpResponseMetadata const& response_metadata);
private:
	IntrusivePtr<AccessorHelper> m_ptrHelper;
	ConstRequestPtr const m_ptrRequest;
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const m_ptrCachedMetadata;
	uint64_t const m_bodyLength;
	uint32_t const m_version;
	
	/**
	 * \brief File that contains a cached response, or null.
	 *
	 * We may write to it to update the metadata, but if we decide to
	 * cache a new version of a response, we will create another file.
	 * \note This member must go after m_ptrHelper, so that
	 *       the file is closed (and possibly deleted) before
	 *       unreferencing the helper.
	 */
	std::auto_ptr<FileInstance> m_ptrFile;
};


class CacheReader : public AbstractResponseReader
{
	DECLARE_NON_COPYABLE(CacheReader)
public:
	CacheReader(IntrusivePtr<AccessorHelper> const& helper,
		std::auto_ptr<HttpResponseMetadata>& metadata,
		std::auto_ptr<FileInstance>& file, CacheInfo const& cache_info,
		uint64_t body_length, uint32_t version);
	
	virtual ~CacheReader();
	
	virtual std::auto_ptr<HttpResponseMetadata> retrieveMetadata();
	
	virtual ssize_t readBodyData(void* buf, size_t len);
private:
	IntrusivePtr<AccessorHelper> m_ptrHelper;
	uint64_t m_bodyDataRemaining;
	ssize_t m_lastReadResult;
	std::auto_ptr<HttpResponseMetadata> m_ptrMetadata;
	CacheInfo m_cacheInfo;
	
	/**
	 * This auto_ptr can be null if an error occured.
	 * \note This member must go after m_ptrHelper, so that
	 *       the file is closed (and possibly deleted) before
	 *       unreferencing the helper.
	 */
	std::auto_ptr<FileInstance> m_ptrFile;
	
	uint32_t m_version;
};


class CacheWriter : public AbstractResponseWriter
{
public:
	CacheWriter(IntrusivePtr<detail::CacheWriterHelper> const& helper,
		HttpResponseMetadata const& response_metadata,
		std::auto_ptr<FileInstance>& file, uint32_t version);
	
	virtual ~CacheWriter();
	
	virtual ssize_t writeBodyData(void const* buf, size_t len);
	
	virtual void commit();
private:
	static char const m_sFileHeader[12];
	
	/**
	 * CacheWriterHelper is an interface implemented by Item.
	 */
	IntrusivePtr<detail::CacheWriterHelper> m_ptrHelper;
	
	/**
	 * This auto_ptr can be null if a write error occured.
	 * \note This member must go after m_ptrHelper, so that
	 *       the file is closed (and possibly deleted) before
	 *       unreferencing the helper.
	 */
	std::auto_ptr<FileInstance> m_ptrFile;
	
	SharedPtr<HttpResponseMetadata, ACE_MT_SYNCH> m_ptrResponseMetadata;
	std::string m_headers;
	CacheInfo m_cacheInfo;
	uint32_t m_version;
	bool m_isFileHeaderWritten : 1;
};


struct FileInfo
{
	uint64_t const body_length;
	std::auto_ptr<HttpResponseMetadata> metadata;
	
	FileInfo(std::auto_ptr<HttpResponseMetadata> metadata, uint64_t body_length)
	: body_length(body_length), metadata(metadata) {}
	
	off_t getBodyOffset() const { return 12; }
	
	off_t getHeadersOffset() const { return getBodyOffset() + body_length; }
};


class HeaderUpdater
{
public:
	HeaderUpdater(HttpHeadersCollection& target, Item::MetadataStatus& status)
	: m_rTarget(target), m_rStatus(status) {}
	
	void operator()(HttpHeader const& header);
private:
	HttpHeadersCollection& m_rTarget;
	Item::MetadataStatus& m_rStatus;
};


template<typename T>
static void writeNum(unsigned char* buf, T val)
{
	for (unsigned i = 0; i < sizeof(T); ++i) {
		buf[i] = val >> (i << 3);
	}
}

template<typename T>
static T readNum(unsigned char const* buf)
{
	T val = 0;
	for (unsigned i = 0; i < sizeof(T); ++i) {
		val |= T(buf[i]) << (i << 3);
	}
	return val;
}

inline void intrusive_ref(CacheWriterHelper& obj)
{
	obj.attachWriterMT();
}

inline void intrusive_unref(CacheWriterHelper& obj)
{
	obj.detachWriterMT();
}


/*============================ ResponseHandler ===========================*/

ResponseHandler::ResponseHandler(
	IntrusivePtr<AccessorHelper> const& helper,
	ConstRequestPtr const& request,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	uint64_t body_length, uint32_t version, std::auto_ptr<FileInstance>& file)
:	m_ptrHelper(helper),
	m_ptrRequest(request),
	m_ptrCachedMetadata(cached_metadata),
	m_bodyLength(body_length),
	m_version(version),
	m_ptrFile(file)
{
}

ResponseHandler::~ResponseHandler()
{
}

ResponseResolution
ResponseHandler::handleResponse(HttpResponseMetadata const& response_metadata)
{
	ResponseResolution resolution;
	
	if (m_ptrHelper.get()) {
		resolution = m_ptrHelper->handleResponseMT(
			m_ptrHelper, m_ptrRequest, m_ptrCachedMetadata,
			response_metadata, m_ptrFile, m_bodyLength, m_version
		);
		m_ptrHelper.reset(0);
	}
	
	return resolution;
}


/*============================= CacheReader =============================*/

CacheReader::CacheReader(
	IntrusivePtr<AccessorHelper> const& helper,
	std::auto_ptr<HttpResponseMetadata>& metadata,
	std::auto_ptr<FileInstance>& file, CacheInfo const& cache_info,
	uint64_t const body_length, uint32_t const version)
:	m_ptrHelper(helper),
	m_bodyDataRemaining(body_length),
	m_lastReadResult(0),
	m_ptrMetadata(metadata),
	m_cacheInfo(cache_info),
	m_ptrFile(file),
	m_version(version)
{
	assert(m_ptrFile.get());
	assert(m_ptrFile->fileIO().get());
}

CacheReader::~CacheReader()
{
}

std::auto_ptr<HttpResponseMetadata>
CacheReader::retrieveMetadata()
{
	return m_ptrMetadata;
}

ssize_t
CacheReader::readBodyData(void* buf, size_t len)
{
	if (!m_ptrFile.get()) {
		return m_lastReadResult;
	}
	size_t const chunk = std::min<uint64_t>(m_bodyDataRemaining, len);
	m_lastReadResult = m_ptrFile->fileIO()->read(buf, chunk);
	if (m_lastReadResult == -1) {
		m_ptrHelper->invalidateCacheEntryMT(m_ptrFile, m_version);
	} else {
		m_bodyDataRemaining -= m_lastReadResult;
	}
	
	if (m_lastReadResult <= 0) {
		m_ptrFile.reset();
		m_ptrHelper.reset(0);
	}
	
	return m_lastReadResult;
}


/*============================= CacheWriter =============================*/

char const CacheWriter::m_sFileHeader[12] = {'B', 'F', 1, 0 };

CacheWriter::CacheWriter(
	IntrusivePtr<CacheWriterHelper> const& helper,
	HttpResponseMetadata const& response_metadata,
	std::auto_ptr<FileInstance>& file, uint32_t const version)
:	m_ptrHelper(helper),
	m_ptrFile(file),
	m_ptrResponseMetadata(new HttpResponseMetadata(response_metadata)),
	m_cacheInfo(response_metadata, 0), // body_length is 0 for now
	m_version(version),
	m_isFileHeaderWritten(false)
{
	time_t date = static_cast<time_t>(-1);
	if (HttpHeader const* hdr = response_metadata.headers().getHeaderPtr(BString("Date"))) {
		date = Date::parse(hdr->getValue());
	}
	if (date == static_cast<time_t>(-1)) {
		date = time(0);
	}
	
	std::ostringstream strm;
	
	unsigned char buf[8];
	writeNum<int64_t>(buf, date);
	
	strm.write((char const*)buf, sizeof(buf));
	response_metadata.statusLine().toStream(strm);
	response_metadata.headers().toStream(strm);
	strm << "\r\n";
	strm.str().swap(m_headers);
}

CacheWriter::~CacheWriter()
{
}

ssize_t
CacheWriter::writeBodyData(void const* const buf, size_t const len)
{
	if (!m_ptrFile.get()) {
		return -1;
	}
	
	std::auto_ptr<FileInstance> file(m_ptrFile);
	IntrusivePtr<CacheWriterHelper> helper;
	m_ptrHelper.swap(helper);
	
	if (!m_isFileHeaderWritten) {
		if (file->fileIO()->write_n(m_sFileHeader, sizeof(m_sFileHeader)) != sizeof(m_sFileHeader)) {
			return -1;
		}
		m_isFileHeaderWritten = true;
	}
	ssize_t const written = file->fileIO()->write(buf, len);
	if (written > 0) {
		m_cacheInfo.body_length += written;
	} else {
		return written;
	}
	
	m_ptrHelper.swap(helper);
	m_ptrFile = file;
	
	return written;
}

void
CacheWriter::commit()
{
	if (!m_ptrFile.get()) {
		return;
	}
	
	std::auto_ptr<FileInstance> file(m_ptrFile);
	IntrusivePtr<CacheWriterHelper> helper;
	m_ptrHelper.swap(helper);
	
	if (!m_isFileHeaderWritten) {
		if (file->fileIO()->write_n(m_sFileHeader, sizeof(m_sFileHeader)) != sizeof(m_sFileHeader)) {
			return;
		}
		m_isFileHeaderWritten = true;
	}
	
	if (file->fileIO()->write_n(m_headers.c_str(), m_headers.size()) != m_headers.size()) {
		return;
	}
	
	if (file->fileIO()->seek(4, SEEK_SET) == -1) {
		return;
	}
	
	unsigned char body_length[8];
	writeNum<uint64_t>(body_length, m_cacheInfo.body_length);
	if (file->fileIO()->write_n(body_length, sizeof(body_length)) != sizeof(body_length)) {
		return;
	}
	
	helper->writeCommitMT(file, m_ptrResponseMetadata, m_cacheInfo, m_version);
}


/*=============================== CacheInfo ===============================*/

CacheInfo::CacheInfo()
:	body_length(0),
	last_modified(static_cast<time_t>(-1)),
	last_validated(static_cast<time_t>(-1)),
	freshness_lifetime(0),
	etag(),
	status_code(0),
	heuristic_expiration(false),
	must_revalidate(false)
{
}

CacheInfo::CacheInfo(HttpResponseMetadata const& metadata, uint64_t const body_len)
:	body_length(body_len),
	last_modified(static_cast<time_t>(-1)),
	last_validated(static_cast<time_t>(-1)),
	freshness_lifetime(0),
	etag(),
	status_code(metadata.statusLine().getCode()),
	heuristic_expiration(false),
	must_revalidate(false)
{
	HttpHeadersCollection const& headers = metadata.headers();
	time_t const cur_time = time(0);
	
	if (HttpHeader const* hdr = headers.getHeaderPtr(BString("ETag"))) {
		ETag(*hdr).swap(etag);
	}
	if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Last-Modified"))) {
		last_modified = Date::parse(hdr->getValue());
	}
	int age = 0;
	if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Age"))) {
		age = StringUtils::toNumber<int>(hdr->getValue(), 0);
	}
	time_t date = static_cast<time_t>(-1);
	if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Date"))) {
		date = Date::parse(hdr->getValue());
	}
	if (date != static_cast<time_t>(-1)) {
		last_validated = std::min(date, cur_time) - age;
	}
	
	ResponseCacheControl const cc(metadata);
	must_revalidate = cc.must_revalidate;
	if (cc.s_maxage_set) {
		freshness_lifetime = std::max<int>(cc.s_maxage, 0);
	} else if (cc.max_age_set) {
		freshness_lifetime = std::max<int>(cc.max_age, 0);
	} else if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Expires"))) {
		time_t const expires = Date::parse(hdr->getValue());
		if (expires != static_cast<time_t>(-1) && expires > cur_time) {
			freshness_lifetime = expires - cur_time;
		}
	} else {
		heuristic_expiration = true;
		freshness_lifetime = 24*60*60;
	}
}


/*============================== HeaderUpdater ============================*/

void
HeaderUpdater::operator()(HttpHeader const& header)
{
	InsensitiveEqual const ieq;
	
	if (ieq(header.getName(), BString("ETag"))) {
		// If we used a strong validation, then ETag is bound to be the same.
		// Otherwise, our stored body is not guaranteed to match the new ETag.
		return;
	}
	
	if (HttpHeader const* hdr = m_rTarget.getHeaderPtr(header.getName())) {
		if (*hdr == header) {
			// The old and the new headers are the same.
			return;
		}
	}
	
	m_rTarget.setHeader(header);
	
	if (ieq(header.getName(), BString("Date"))) {
		if (m_rStatus == Item::METADATA_UNMODIFIED) {
			m_rStatus = Item::METADATA_ONLY_DATE_MODIFIED;
		}
		return;
	}
	
	m_rStatus = Item::METADATA_MODIFIED;
}

} // namespace detail


/*================================= Item ================================*/

Item::Item(
	IntrusivePtr<AbstractFileStorage> const& storage,
	FileId const& file_id, bool& file_ok)
:	m_ptrStorage(storage),
	m_fileKey(file_id.getKey()),
	m_numWriters(0),
	m_numRefs(0),
	m_cachedMetadataStatus(METADATA_UNMODIFIED),
	m_curVersion(file_id.getVersion()),
	m_nextVersion(file_id.getVersion() + 1),
	m_readyForGC(false)
{
	file_ok = false;
	
	std::auto_ptr<AbstractFileIO> file_io(
		m_ptrStorage->openExisting(file_id, AbstractFileStorage::RDONLY)
	);
	if (!file_io.get()) {
		return;
	}
	
	std::auto_ptr<FileInfo> const file_info(loadFileInfoMT(*file_io));
	if (!file_info.get()) {
		return;
	}
	
	std::auto_ptr<FileInstance> file(
		new FileInstance(m_ptrStorage, file_id, std::auto_ptr<AbstractFileIO>(), true)
	);
	
	m_cacheInfo = CacheInfo(*file_info->metadata, file_info->body_length);
	m_ptrCurFile = file;
	m_ptrCachedMetadata.reset(file_info->metadata.release());
	file_ok = true;
}

Item::~Item()
{
	if (m_ptrCurFile.get()) {
		if (!m_readyForGC) {
			updateCachedMetadataST();
			if (m_ptrCurFile.get()) {
				m_ptrCurFile->unlinkOnNoRefs(false);
			}
		}
	}
}

void
Item::ref() const
{
	++m_numRefs;
}

void
Item::unref() const
{
	if (--m_numRefs == 0) {
		delete this;
	}
}

IntrusivePtr<AbstractItem>
Item::create(
	IntrusivePtr<AbstractFileStorage> const& storage,
	FileId const& file_id, bool& file_ok)
{
	return IntrusivePtr<AbstractItem>(new Item(storage, file_id, file_ok));
}

std::auto_ptr<FileInstance>
Item::openCurFileST(AbstractFileStorage::Mode const mode)
{
	if (!m_ptrCurFile.get()) {
		return std::auto_ptr<FileInstance>();
	}
	
	FileId const file_id(m_fileKey, m_curVersion);
	return openFileMT(*m_ptrCurFile, file_id, mode);
}

std::auto_ptr<FileInstance>
Item::openFileMT(FileInstance const& prototype,
	FileId const& file_id, AbstractFileStorage::Mode const mode)
{
	std::auto_ptr<AbstractFileIO> file_io(m_ptrStorage->openExisting(file_id, mode));
	if (!file_io.get()) {
		return std::auto_ptr<FileInstance>();
	}
	
	return std::auto_ptr<FileInstance>(new FileInstance(prototype, file_io));
}

void
Item::invalidateCurrentCacheEntryST()
{
	m_cacheInfo = CacheInfo();
	m_ptrCurFile.reset();
	m_ptrCachedMetadata.reset(0);
	m_cachedMetadataStatus = METADATA_UNMODIFIED;
}

void
Item::invalidateCacheEntryST(std::auto_ptr<FileInstance> file, uint32_t const version)
{
	if (version == m_curVersion) {
		invalidateCurrentCacheEntryST();
	}
	//file.reset(); // not necessary because auto_ptr is passed by value
}

void
Item::invalidateCacheEntryMT(
	std::auto_ptr<FileInstance> file, uint32_t const version)
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	invalidateCacheEntryST(file, version);
}

RequestResolution
Item::handleRequest(ConstRequestPtr const& request)
{
	assert(request.get());
	
	RequestCacheControl const request_cc(*request);
	
	RequestResolution res(doHandleRequest(request, request_cc));
	if (request_cc.only_if_cached && !res.reader.get()) {
		return RequestResolution(createGatewayTimeoutResponseMT(*request));
	}
	return res;
}

RequestResolution
Item::doHandleRequest(
	ConstRequestPtr const& request, RequestCacheControl const& request_cc)
{
	HttpHeadersCollection const& headers = request->headers();
	
	if (request_cc.no_cache || request_cc.no_store) {
		// FIXME: Cache-Control: no-cache in request only forbids answering
		// with a cached response, it doesn't forbid caching the response
		// we get from upstream.
		return RequestResolution();
	}
	if (request->requestLine().getMethod() != BString("GET")) {
		// Even though it should be possible to cache HEAD and
		// other methods, but what will happen if a cached response
		// to HEAD will get served to a GET request?
		return RequestResolution();
	}
	if (headers.hasHeader(BString("Authorization"))) {
		return RequestResolution();
	}
	if (headers.hasHeader(BString("Range"))) {
		return RequestResolution();
	}
	
	IntrusivePtr<detail::AccessorHelper> helper;
	std::auto_ptr<FileInstance> file_template;
	std::auto_ptr<FileInstance> file;
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> cached_metadata;
	CacheInfo cache_info;
	uint32_t file_version;
	
	{ // guard scope
		ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
		
		helper.reset(this);
		
		// If we have a file, we need metadata as well.
		if (m_ptrCurFile.get() && !m_ptrCachedMetadata.get()) {
			file = openCurFileST(AbstractFileStorage::RDONLY);
			if (!file.get()) {
				invalidateCurrentCacheEntryST();
			} else {
				std::auto_ptr<FileInfo> file_info(loadFileInfoMT(*file->fileIO()));
				if (!file_info.get()) {
					invalidateCurrentCacheEntryST();
				} else {
					m_ptrCachedMetadata.reset(file_info->metadata.release());
				}
			}
		}
		
		if (m_ptrCurFile.get()) {
			file_template.reset(
				new FileInstance(*m_ptrCurFile, std::auto_ptr<AbstractFileIO>())
			);
		}
		
		cached_metadata = m_ptrCachedMetadata;
		cache_info = m_cacheInfo;
		file_version = m_curVersion;
		
	} // guard scope
	
	if (!file_template.get() || !cached_metadata.get()) {
		// Maybe we could cache server's response...
		return RequestResolution(createResponseHandlerMT(
			helper, request, cached_metadata, std::auto_ptr<FileInstance>(), 0, file_version
		));
	}
	
	assert(file_template.get());
	assert(cached_metadata.get());
	
	HttpHeader const* const ims_hdr = headers.getHeaderPtr(BString("If-Modified-Since"));
	HttpHeader const* const ius_hdr = headers.getHeaderPtr(BString("If-Unmodified-Since"));
	HttpHeader const* const im_hdr = headers.getHeaderPtr(BString("If-Match"));
	HttpHeader const* const inm_hdr = headers.getHeaderPtr(BString("If-None-Match"));
	HttpHeader const* const ir_hdr = headers.getHeaderPtr(BString("If-Range"));
	bool const have_client_validation = (ims_hdr || ius_hdr || im_hdr || inm_hdr || ir_hdr);
	bool const client_accepts_age = clientAcceptsAgeMT(request_cc, cache_info);
	
	if (!have_client_validation) {
		if (!client_accepts_age) {
			ConstRequestPtr const new_request(addProxyValidationMT(*request, cache_info));
			if (new_request && !file.get()) {
				// new_request contains validation headers added by us.
				FileId const file_id(m_fileKey, file_version);
				file = openFileMT(*file_template, file_id, AbstractFileStorage::RDONLY);
			}
			if (new_request && file.get()) {
				// If the server responds with 304 Not Modified, response
				// body will be read from this file.
				return RequestResolution(createResponseHandlerMT(
					helper, new_request, cached_metadata, file,
					cache_info.body_length, file_version
				));
			} else {
				// We still have an opportunity to cache the response.
				file.reset();
				return RequestResolution(createResponseHandlerMT(
					helper, request, cached_metadata, file, 0, file_version
				));
			}
		} // otherwise we may respond with a cached response
	
	} else { // have client validation
		
		if (!client_accepts_age) {
			// We still have the opportunity to cache the response,
			// unless the server responds with 304 Not Modified.
			file.reset();
			return RequestResolution(createResponseHandlerMT(
				helper, request, cached_metadata,
				file, cache_info.body_length, file_version
			));
		}
		
		enum ValidationResolution {
			UNDEFINED, UPSTREAM, NOT_MODIFIED, PRECONDITION_FAILED
		};	
		ValidationResolution resolution = UNDEFINED;
			
		if (ims_hdr && resolution != UPSTREAM) {
			time_t const time = Date::parse(ims_hdr->getValue());
			if (time == static_cast<time_t>(-1)) {
				resolution = UPSTREAM;
			} else if (cache_info.last_modified <= time) {
				if (cache_info.status_code == 200) {
					if (resolution == UNDEFINED) {
						resolution = NOT_MODIFIED;
					}
				}
			}
		}
		
		if (ius_hdr && resolution != UPSTREAM) {
			time_t const time = Date::parse(ius_hdr->getValue());
			if (time == static_cast<time_t>(-1)) {
				resolution = UPSTREAM;
			} else if (cache_info.last_modified <= time) {
				resolution = UPSTREAM;
			} else {
				if (resolution == UNDEFINED) {
					resolution = PRECONDITION_FAILED;
				}
			}
		}
		
		if (im_hdr) {
			resolution = UPSTREAM;
		}
		
		if (inm_hdr && resolution != UPSTREAM) {
			if (!cache_info.etag.isValid()) {
				resolution = UPSTREAM;
			} else {
				// FIXME: HttpHeaderStructure might not be able to parse etags.
#if 0
				BString const& etag = cache_info.etag.toBString();
				HttpHeaderStructure const structure(*inm_hdr);
				bool match_found = false;
				typedef std::list<HttpHeaderElement> ElementList;
				ElementList::const_iterator it(structure.elements().begin());
				ElementList::const_iterator const end(structure.elements().end());
				for (; it != end; ++it) {
					if (it->getName() == etag || it->getName() == BString("*")) {
						match_found = true;
						break;
					}
				}
				if (!match_found) {
					resolution = UPSTREAM;
				}
#else
				resolution = UPSTREAM;
#endif
			}
		}
			
		if (ir_hdr) {
			resolution = UPSTREAM;
		}
			
		assert(client_accepts_age); // see above
			
		switch (resolution) {
		case UNDEFINED: // this means we can use a cached response
			break;
		case UPSTREAM: // just forward the request to the upstream server
			file.reset();
			return RequestResolution(createResponseHandlerMT(
				helper, request, cached_metadata,
				file, cache_info.body_length, file_version
			));
		case NOT_MODIFIED:
			return RequestResolution(createNotModifiedResponseMT(cache_info));
		case PRECONDITION_FAILED:
			return RequestResolution(createPreconditionFailedResponseMT());
		}
	}
	
	// OK, now we may just respond with a cached response.
	
	std::auto_ptr<AbstractResponseReader> reader;
	
	if (!file.get()) {
		FileId const file_id(m_fileKey, file_version);
		file = openFileMT(*file_template, file_id, AbstractFileStorage::RDONLY);
	}
	
	if (file.get()) {
		assert(file->fileIO().get());
		reader = createCacheReaderMT(
			helper, cache_info, cached_metadata, file, file_version
		);
		if (!reader.get()) {
			// Damaged file?
			invalidateCacheEntryMT(file, file_version);
			assert(!file.get());
		}
	}
	
	if (reader.get()) {
		return RequestResolution(reader);
	} else {
		// We couldn't respond with a cached response, but maybe we could
		// cache a new one.
		assert(!file.get());
		return RequestResolution(createResponseHandlerMT(
			helper, request, cached_metadata, file, cache_info.body_length, file_version
		));
	}
}

bool
Item::prepareForGC(GCFlags const flags)
{
	// We assume our owner holds exactly one reference to us.
	uint32_t const nrefs = (m_numRefs += 0);
	if (nrefs != 1) {
		// This means CacheReader, CacheWriter or ResponseHandler
		// reference us.
		return false;
	}
	
	// We assume we are protected by our owner's mutex.
	if (m_ptrCurFile.get() && flags == GC_ONLY_UNCACHED) {
		return false;
	}
	
	m_readyForGC = true;
	return true;
}

bool
Item::clientAcceptsAgeMT(
	RequestCacheControl const& cc, detail::CacheInfo const& inf)
{
	int const age = time(0) - inf.last_validated;
	int const max_age = (inf.must_revalidate && inf.heuristic_expiration)
		? 0 : inf.freshness_lifetime;
	if (cc.max_stale_set && !inf.must_revalidate) {
		if (!cc.max_stale_has_value) {
			return true;
		} else if (age < max_age + cc.max_stale) {
			return true;
		}
	}
	if (cc.max_age_set && age >= cc.max_age) {
		return false;
	}
	if (cc.min_fresh_set && age + cc.min_fresh < max_age) {
		return true;
	}
	return (age < max_age);
}

/**
 * \brief If possible, add validating headers based on \p cache_info.
 * \return The new request with validating headers added, or null request ptr,
 *         if adding headers was impossible.
 */
ConstRequestPtr
Item::addProxyValidationMT(
	HttpRequestMetadata const& request, detail::CacheInfo const& cache_info)
{
	std::auto_ptr<HttpRequestMetadata> new_request;
	if (cache_info.etag.isValid()) {
		new_request.reset(new HttpRequestMetadata(request));
		new_request->headers().setHeader(
			BString("If-None-Match"), cache_info.etag.toBString()
		);
	}
	if (cache_info.last_modified != static_cast<time_t>(-1)) {
		if (!new_request.get()) {
			new_request.reset(new HttpRequestMetadata(request));
		}
		new_request->headers().setHeader(
			BString("If-Modified-Since"),
			Date::format(cache_info.last_modified)
		);
	}
	
	return ConstRequestPtr(new_request.release());
}

void
Item::updateCachedMetadataST()
{
	if (m_cachedMetadataStatus == METADATA_UNMODIFIED) {
		return;
	}
	
	std::auto_ptr<FileInstance> file(openCurFileST(AbstractFileStorage::WRONLY));
	if (!file.get()) {
		invalidateCacheEntryST(file, m_curVersion);
		return;
	}
	
	time_t date = static_cast<time_t>(-1);
	if (HttpHeader const* hdr = m_ptrCachedMetadata->headers().getHeaderPtr(BString("Date"))) {
		date = Date::parse(hdr->getValue());
	}
	unsigned char date_buf[8];
	writeNum<int64_t>(date_buf, date);
	
	if (file->fileIO()->seek(12 + m_cacheInfo.body_length, SEEK_SET) == -1) {
		invalidateCacheEntryST(file, m_curVersion);
		return;
	}
	
	if (file->fileIO()->write_n(date_buf, 8) != 8) {
		invalidateCacheEntryST(file, m_curVersion);
		return;
	}
	
	if (m_cachedMetadataStatus == METADATA_ONLY_DATE_MODIFIED) {
		m_cachedMetadataStatus = METADATA_UNMODIFIED;
		return;
	}
	
	std::ostringstream strm;
	m_ptrCachedMetadata->statusLine().toStream(strm);
	m_ptrCachedMetadata->headers().toStream(strm);
	strm << "\r\n";
	std::string const& headers = strm.str();
	if (file->fileIO()->write_n(headers.c_str(), headers.size()) != headers.size()) {
		invalidateCacheEntryST(file, m_curVersion);
		return;
	}
	
	if (file->fileIO()->truncate(file->fileIO()->tell()) == -1) {
		invalidateCacheEntryST(file, m_curVersion);
		return;
	}
	
	m_cachedMetadataStatus = METADATA_UNMODIFIED;
}

std::auto_ptr<AbstractResponseReader>
Item::createNotModifiedResponseMT(detail::CacheInfo const& cache_info)
{
	HttpStatusLine const sline(
		HttpVersion::HTTP_1_1, HttpStatusLine::SC_NOT_MODIFIED
	);
	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(sline)
	);
	metadata->headers().setHeader(BString("Date"), Date::formatCurrentTime());
	metadata->setBodyStatus(HttpResponseMetadata::BODY_FORBIDDEN);
	
	if (cache_info.etag.isValid()) {
		metadata->headers().setHeader(
			BString("ETag"), cache_info.etag.toBString()
		);
	}
	
	return std::auto_ptr<AbstractResponseReader>(
		new MemoryReader(metadata, BString())
	);
}

std::auto_ptr<AbstractResponseReader>
Item::createPreconditionFailedResponseMT()
{
	HttpStatusLine const sline(
		HttpVersion::HTTP_1_1, HttpStatusLine::SC_PRECONDITION_FAILED
	);
	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(sline)
	);
	metadata->headers().setHeader(BString("Date"), Date::formatCurrentTime());
	BString const body;
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_KNOWN);
	metadata->setBodySize(body.size());
	return std::auto_ptr<AbstractResponseReader>(
		new MemoryReader(metadata, body)
	);
}

std::auto_ptr<AbstractResponseReader>
Item::createGatewayTimeoutResponseMT(HttpRequestMetadata const& request)
{
	std::auto_ptr<CraftedResponse> resp(
		ErrorFactory::getGenericError(
			request.requestLine().getURI(),
			"Resource not in cache, but user agent only accepts cached responses.",
			HttpStatusLine::SC_GATEWAY_TIMEOUT
		)
	);
	
	assert(resp.get());
	
	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(resp->metadata())
	);
	BString const body(resp->body().toBString());
	
	return std::auto_ptr<AbstractResponseReader>(
		new MemoryReader(metadata, body)
	);
}

std::auto_ptr<AbstractResponseReader>
Item::createCacheEntryBrokenResponseMT(HttpRequestMetadata const& request)
{
	std::auto_ptr<CraftedResponse> resp(
		ErrorFactory::getGenericError(
			request.requestLine().getURI(),
			"Cached response is damaged.",
			HttpStatusLine::SC_INTERNAL_SERVER_ERROR
		)
	);
	
	assert(resp.get());
	
	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(resp->metadata())
	);
	BString const body(resp->body().toBString());
	
	return std::auto_ptr<AbstractResponseReader>(
		new MemoryReader(metadata, body)
	);
}

std::auto_ptr<AbstractResponseReader>
Item::createCacheReaderMT(
	IntrusivePtr<AccessorHelper> const& helper,
	detail::CacheInfo const& cache_info,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	std::auto_ptr<FileInstance> file, uint32_t const version)
{
	assert(file.get());
	assert(file->fileIO().get());
	assert(cached_metadata.get());

	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(*cached_metadata)
	);
	
	prepareMetadataForOutputMT(*metadata, cache_info.body_length, cache_info);
	
	if (file->fileIO()->seek(12, SEEK_SET) == -1) {
		return std::auto_ptr<AbstractResponseReader>();
	}
	return std::auto_ptr<AbstractResponseReader>(
		new detail::CacheReader(
			helper, metadata, file, cache_info,
			cache_info.body_length, version
		)
	);
}

std::auto_ptr<AbstractResponseReader>
Item::createCacheReaderMT(
	IntrusivePtr<AccessorHelper> const& helper,
	std::auto_ptr<FileInstance> file,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	HttpResponseMetadata const& not_modified_response,
	uint64_t const body_length, uint32_t const version)
{	
	assert(file.get());
	assert(file->fileIO().get());

	std::auto_ptr<HttpResponseMetadata> metadata(
		new HttpResponseMetadata(*cached_metadata)
	);
	
	MetadataStatus const metadata_status = updateMetadataMT(
		*metadata, not_modified_response
	);
	CacheInfo const cache_info(*metadata, body_length);
	
	if (metadata_status != METADATA_UNMODIFIED) {
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> new_metadata(
			new HttpResponseMetadata(*metadata)
		);
		
		ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);

		m_cacheInfo = cache_info;
		m_ptrCachedMetadata.swap(new_metadata);
		m_cachedMetadataStatus = metadata_status;
	}
	
	prepareMetadataForOutputMT(*metadata, body_length, cache_info);
	
	if (file->fileIO()->seek(12, SEEK_SET) == -1) {
		return std::auto_ptr<AbstractResponseReader>();
	}
	
	return std::auto_ptr<AbstractResponseReader>(
		new detail::CacheReader(
			helper, metadata, file, cache_info,
			body_length, version
		)
	);
}

void
Item::prepareMetadataForOutputMT(
	HttpResponseMetadata& metadata,
	uint64_t const body_length, detail::CacheInfo const& cache_info)
{
	metadata.setBodyStatus(HttpResponseMetadata::BODY_SIZE_KNOWN);
	metadata.setBodySize(body_length);
	
	time_t const cur_time = time(0);
	metadata.headers().setHeader(BString("Date"), Date::format(cur_time));
	char buf[15];
	ACE_OS::snprintf(buf, sizeof(buf), "%d", int(cur_time - cache_info.last_validated));
	metadata.headers().setHeader(BString("Age"), BString(buf, strlen(buf)));
}

std::auto_ptr<detail::FileInfo>
Item::loadFileInfoMT(AbstractFileIO& file_io)
{
	/*
	File format:
	+--------------+--------------+------------+---------------------+
	| "BF" CHAR(2) | version BYTE | flags BYTE | body_length U64(LE) |
	+--------------+--------------+------------+---------------------+
	version: always 1
	flags: none are defined, so always 0.
	
	+------+--------------+---------------------+
	| body | date S64(LE) | headers (until EOF) |
	+------+--------------+---------------------+
	date: the parsed value (unix timestamp) of the Date header.
	headers: the string representation of HTTP response headers, including
	the status line and the trailing empty line.  There may be a Date header
	among them, but it's going to be ignored in favour of the parsed date value.
	This separation of the date value is done to optimize a typical case
	where the only modification necessary is an updated Date value.
	*/
	
	assert(file_io.seek(0, SEEK_SET) == 0);
	
	unsigned char header[12];
	if (file_io.read_n(header, sizeof(header)) != sizeof(header)) {
		return std::auto_ptr<FileInfo>();
	}
	if (header[0] != 'B' || header[1] != 'F' || header[2] != 1) {
		return std::auto_ptr<FileInfo>();
	}
	
	uint64_t const body_length = readNum<uint64_t>(header + 4);
	
	off_t const end_pos = file_io.seek(0, SEEK_END);
	if (end_pos == -1) {
		return std::auto_ptr<FileInfo>();
	}
	
	if (file_io.seek(sizeof(header) + body_length, SEEK_SET) == -1) {
		return std::auto_ptr<FileInfo>();
	}
	
	size_t const hdr_len = end_pos - sizeof(header) - body_length;
	std::auto_ptr<DataChunk> hdr_data(DataChunk::create(hdr_len));
	if (file_io.read_n(hdr_data->getDataAddr(), hdr_len) != hdr_len) {
		return std::auto_ptr<FileInfo>();
	}
	
	time_t const date = readNum<int64_t>((unsigned char const*)hdr_data->getDataAddr());
	
	std::auto_ptr<HttpResponseMetadata> metadata(
		parseResponseMetadataMT(BString(hdr_data, 8))
	);
	if (!metadata.get()) {
		return std::auto_ptr<FileInfo>();
	}
	metadata->headers().setHeader(BString("Date"), Date::format(date));
	
	return std::auto_ptr<FileInfo>(new FileInfo(metadata, body_length));
}

std::auto_ptr<HttpResponseMetadata>
Item::parseResponseMetadataMT(BString const& data)
{
	char const* p = data.begin();
	char const* const end = data.end();
	
	BString const tail("\r\n\r\n");
	if (!StringUtils::endsWith(p, end, tail.begin(), tail.end())) {
		return std::auto_ptr<HttpResponseMetadata>();
	}
	
	for (; p != end; ++p) {
		if (*p == ' ' || *p == '\t') {
			break;
		} else if (*p == '\n') {
			return std::auto_ptr<HttpResponseMetadata>();
		}
	}
	// HTTP/1.1 200 OK
	//         ^-- p points here
	
	HttpVersion http_vers(1, 0);
	if (!http_vers.parse(data.begin(), p)) {
		return std::auto_ptr<HttpResponseMetadata>();
	}
	
	for (; p != end; ++p) {
		if (*p != ' ' && *p != '\t') {
			break;
		}
	}
	// HTTP/1.1 200 OK
	//          ^-- p points here
	
	int status_code = 0;
	for (; p != end && isdigit(*p); ++p) {
		status_code = (status_code << 3) + (status_code << 1) + (*p - '0');
	}
	// HTTP/1.1 200 OK
	//             ^-- p points here
	char const* const status_code_end = p;
	
	if (status_code < 100 || p == end || !isspace(*p)) {
		return std::auto_ptr<HttpResponseMetadata>();
	}
	
	for (; p != end && *p != '\n'; ++p) {
		// skip the rest of the line
	}
	if (p == end) {
		return std::auto_ptr<HttpResponseMetadata>();
	}
	
	char const* status_msg_begin = StringUtils::ltrim(status_code_end, p);
	char const* status_msg_end = StringUtils::rtrim(status_msg_begin, p);
	BString const status_msg(data, status_msg_begin, status_msg_end);
	HttpStatusLine const sline(http_vers, status_code, status_msg);
	std::auto_ptr<HttpResponseMetadata> metadata(new HttpResponseMetadata(sline));
	
	++p;
	// Header: value
	// ^-- p points here
	
	for (; p != end; ++p) { // loop through headers
		char const* const hdr_name_begin = p;
		for (; p != end; ++p) {
			if (*p == ':' || *p == '\n') {
				break;
			}
		}
		if (p == end) {
			break;
		}
		if (*p == '\n') {
			continue;
		}
		// Header: value
		//       ^-- p points here
		
		BString const header_name(data, hdr_name_begin, p);
		char const* const hdr_after_colon = p + 1;
		
		for (; p != end && *p != '\n'; ++p) {
			// skip the rest of the line
		}
		// Because the cache file contains pre-processed headers,
		// we don't need to care about multiline ones.
		
		char const* const hdr_value_begin = StringUtils::ltrim(hdr_after_colon, p);
		char const* const hdr_value_end = StringUtils::rtrim(hdr_value_begin, p);
		BString const header_value(data, hdr_value_begin, hdr_value_end);
		metadata->headers().addHeader(header_name, header_value);
	}
	
	return metadata;
}

Item::MetadataStatus
Item::updateMetadataMT(
	HttpResponseMetadata& metadata, HttpResponseMetadata const& not_modified_metadata)
{
	// See RFC 2616, section 13.5 (in particular, 13.5.3)
	
	MetadataStatus status = METADATA_UNMODIFIED;
	
	if (HttpHeader* hdr = metadata.headers().getHeaderPtr(BString("Warning"))) {
		// Remove any Warning headers with 1xx code.
		std::list<BString> values(hdr->getValues());
		std::list<BString>::iterator it(values.begin());
		std::list<BString>::iterator const end(values.end());
		while (it != end) {
			BString const& value = *it;
			if (value.size() < 3 || value[0] == '1') {
				values.erase(it++);
				status = METADATA_MODIFIED;
			} else {
				++it;
			}
		}
		if (status == METADATA_MODIFIED) {
			hdr->setValues(values);
		}
	}
	
	not_modified_metadata.headers().foreach(
		HeaderUpdater(metadata.headers(), status)
	);
	
	return status;
}

std::auto_ptr<AbstractResponseHandler>
Item::createResponseHandlerMT(
	IntrusivePtr<detail::AccessorHelper> const& helper,
	ConstRequestPtr const& request,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	std::auto_ptr<FileInstance> file, uint64_t const body_length, uint32_t const version)
{	
	return std::auto_ptr<AbstractResponseHandler>(
		new ResponseHandler(
			helper, request, cached_metadata, body_length, version, file
		)
	);
}

ResponseResolution
Item::handleResponseMT(
	IntrusivePtr<detail::AccessorHelper> const& helper,
	ConstRequestPtr const& request,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	HttpResponseMetadata const& response_metadata,
	std::auto_ptr<FileInstance> file, uint64_t body_length, uint32_t const version)
{
	if (response_metadata.statusLine().getCode() == 304) {
		return handle304ResponseMT(
			helper, request, cached_metadata,
			response_metadata, file, body_length, version
		);
	} else {
		if (mayBeCachedMT(response_metadata)) {
			return ResponseResolution(
				createCacheWriterMT(response_metadata)
			);
		} else {
			invalidateCacheEntryMT(file, version);
			return ResponseResolution();
		}
	}
}

ResponseResolution
Item::handle304ResponseMT(
	IntrusivePtr<detail::AccessorHelper> const& helper,
	ConstRequestPtr const& request,
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
	HttpResponseMetadata const& response_metadata,
	std::auto_ptr<FileInstance> file, uint64_t const body_length, uint32_t const version)
{
	if (file.get()) {
		// Validation by us.
		std::auto_ptr<AbstractResponseReader> reader(
			createCacheReaderMT(
				helper, file, cached_metadata, response_metadata, body_length, version
			)
		);
		if (!reader.get()) {
			invalidateCacheEntryMT(file, version);
			reader = createCacheEntryBrokenResponseMT(*request);
		}
		return ResponseResolution(reader);
	} else {
		// Validation by client.
		if (varyOkMT(response_metadata.headers())) {
			CacheInfo cache_info(response_metadata, body_length);
			ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
			if (m_ptrCurFile.get()) {
				if (responseWeakIdentityMT(m_cacheInfo, cache_info)) {
					// TODO: update cache entry
				} else {
					invalidateCacheEntryST(file, version);
				}
			}
		}
		return ResponseResolution();
	}
}

bool
Item::responseWeakIdentityMT(
	detail::CacheInfo const& r1, detail::CacheInfo const& r2)
{
	if (r1.last_modified != r2.last_modified) {
		return false;
	} else if (r1.etag.isNull() && r2.etag.isNull()) {
		return true;
	} else {
		return r1.etag.weakIdentity(r2.etag);
	}
}

bool
Item::varyOkMT(HttpHeadersCollection const& headers)
{
	HttpHeader const* hdr = headers.getHeaderPtr(BString("Vary"));
	if (!hdr) {
		return true;
	}
	
	// Normally we don't cache content-negotiated responses,
	// but we handle a common case of Vary: Accept-Encoding.
	// Actually, we don't need to do anything special about it,
	// as we always set Accept-Encoding: gzip,deflate.
	
	InsensitiveEqual ieq;
	return ieq(hdr->getValue(), BString("Accept-Encoding"));
}

bool
Item::mayBeCachedMT(HttpResponseMetadata const& response_metadata)
{
	if (response_metadata.statusLine().getHttpVersion() < HttpVersion::HTTP_1_0) {
		return false;
	}
	
	if (response_metadata.getBodyStatus() == HttpResponseMetadata::BODY_FORBIDDEN) {
		return false;
	}
	
	HttpHeadersCollection const& headers = response_metadata.headers();
	ResponseCacheControl const cc(response_metadata);
	
	if (cc.no_cache || cc.no_store || cc.priv) {
		return false;
	}
	if (cc.publ || cc.max_age_set || cc.s_maxage_set) {
		return true;
	}
	
	// See RFC2616, section 13.4
	switch (response_metadata.statusLine().getCode()) {
	case 200:
	case 203:
	case 300:
	case 301:
	case 410:
		break;
	default:
		return false;
	}
	
	if (!varyOkMT(headers)) {
		return false;
	}
	
	time_t expires = static_cast<time_t>(-1);
	if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Expires"))) {
		expires = Date::parse(hdr->getValue());
		if (expires == static_cast<time_t>(-1) || expires <= time(0)) {
			return false;
		}
	}
	
	if (expires == static_cast<time_t>(-1) && !cc.max_age_set && !cc.s_maxage_set) {
		bool valid_etag = false;
		bool valid_last_modified = false;
		if (HttpHeader const* hdr = headers.getHeaderPtr(BString("ETag"))) {
			valid_etag = ETag(*hdr).isValid();
		}
		if (HttpHeader const* hdr = headers.getHeaderPtr(BString("Last-Modified"))) {
			time_t const date = Date::parse(hdr->getValue());
			valid_last_modified = (date != static_cast<time_t>(-1));
		}
		if (!valid_etag && !valid_last_modified) {
			return false;
		}
	}
	
	// TODO: we should not cache responses with undefined body length,
	// that is: no Content-Length and no Transfer-Encoding: chunked.
	// Unfortunately, Transfer-Encoding: chunked is removed while
	// parsing such a response.
	
	return true;
}

std::auto_ptr<AbstractResponseWriter>
Item::createCacheWriterMT(HttpResponseMetadata const& response_metadata)
{
	if ((m_numWriters += 0) > 0) {
		// No point caching several instances of the same resource.
		return std::auto_ptr<AbstractResponseWriter>();
	}
	
	uint32_t const version = m_nextVersion++;
	
	FileId const file_id(m_fileKey, version);
	std::auto_ptr<AbstractFileIO> file_io(
		m_ptrStorage->createNew(file_id, AbstractFileStorage::WRONLY)
	);
	if (!file_io.get()) {
		return std::auto_ptr<AbstractResponseWriter>();
	} 
	
	std::auto_ptr<FileInstance> file(
		new FileInstance(m_ptrStorage, file_id, file_io, true)
	);
	
	return std::auto_ptr<AbstractResponseWriter>(
		new CacheWriter(
			IntrusivePtr<CacheWriterHelper>(this),
			response_metadata, file, version
		)
	);
}

void
Item::writeCommitMT(
	std::auto_ptr<FileInstance> file,
	SharedPtr<HttpResponseMetadata, ACE_MT_SYNCH> const& metadata,
	detail::CacheInfo const& cache_info, uint32_t const version)
{
	assert(file.get());
	assert(file->fileIO().get());
	
	file->fileIO().reset(); // no need to keep the file open
	
	ACE_Guard<ACE_Thread_Mutex> guard(m_mutex);
	
	m_cacheInfo = cache_info;
	m_ptrCachedMetadata = metadata;
	m_cachedMetadataStatus = METADATA_UNMODIFIED;
	m_curVersion = version;
	m_ptrCurFile = file;
}

void
Item::attachWriterMT()
{
	ref();
	++m_numWriters;
}

void
Item::detachWriterMT()
{
	--m_numWriters;
	unref();
}

} // namespace HttpCache
