/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestLogWindow.h"
#include "RequestLogHandler.h"
#include "RequestLog.h"
#include "Application.h"
#include "AutoScrollingWindow.h"
#include "CompiledImages.h"
#include <gtk/gtkversion.h>
#include <gtkmm/base.h>
#include <gtkmm/box.h>
#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/alignment.h>
#include <gtkmm/togglebutton.h>
#include <gtkmm/liststore.h>
#include <gtkmm/treeview.h>
#include <gtkmm/treeiter.h>
#include <gtkmm/treepath.h>
#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/treeviewcolumn.h>
#include <gtkmm/cellrenderertext.h>
#include <gtkmm/cellrendererpixbuf.h>
#include <gtkmm/accelkey.h>
#include <gtkmm/stock.h>
#include <gtkmm/image.h>
#include <gdkmm/color.h>
#include <gdkmm/pixbuf.h>
#include <cassert>

using namespace std;

namespace GtkGUI
{

class RequestLogWindow::LegendModel : public Gtk::ListStore
{
public:
	struct Columns : public Gtk::TreeModelColumnRecord
	{
		Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > IMAGE;
		Gtk::TreeModelColumn<Glib::ustring> TEXT;
		Gtk::TreeModelColumn<Gdk::Color> BACKGROUND;
		
		Columns() { add(IMAGE); add(TEXT); add(BACKGROUND); }
	};
	
	LegendModel();
	
	virtual ~LegendModel();
	
	Columns const& getCols() const { return m_cols; }
private:
	void fillModel();
	
	Columns m_cols;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconAd;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconCache;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconCancel;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconError;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconOK;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconRedirect;
};


class RequestLogWindow::LogView : public Gtk::TreeView
{
private:
	virtual bool on_button_press_event(GdkEventButton* evt);
};


RequestLogWindow* RequestLogWindow::m_spInstance = 0;

RequestLogWindow::RequestLogWindow()
:	m_pTreeView(0),
	m_pLegendButton(0)
{
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	set_title("BFilter Request Log");
	
	Gtk::VBox* top_vbox = manage(new Gtk::VBox);
	add(*top_vbox);
	
	Gtk::Alignment* list_align = manage(new Gtk::Alignment);
	top_vbox->pack_start(*list_align);
	list_align->property_top_padding() = 5;
	list_align->property_left_padding() = 5;
	list_align->property_right_padding() = 5;
	
	AutoScrollingWindow* scrolled_window = manage(new AutoScrollingWindow);
	list_align->add(*scrolled_window);
	scrolled_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	scrolled_window->set_shadow_type(Gtk::SHADOW_IN);
	scrolled_window->set_size_request(550, 300);
	
	m_pTreeView = manage(new LogView);
	scrolled_window->add(*m_pTreeView);
	
	Gtk::HBox* hbox1 = manage(new Gtk::HBox);
	top_vbox->pack_start(*hbox1, Gtk::PACK_SHRINK);
	hbox1->set_border_width(5);
	
	Gtk::Alignment* clearlog_align = manage(new Gtk::Alignment(0.0, 0.5, 0.0, 0.0));
	hbox1->pack_start(*clearlog_align);
	
	Gtk::HButtonBox* clearlog_bbox = manage(new Gtk::HButtonBox);
	clearlog_align->add(*clearlog_bbox);
	
	Gtk::Button* clearlog_btn = manage(new Gtk::Button(Gtk::Stock::CLEAR));
	clearlog_bbox->pack_start(*clearlog_btn, Gtk::PACK_SHRINK);
	clearlog_btn->signal_clicked().connect(sigc::mem_fun(*this, &RequestLogWindow::onClear));
	
	Gtk::HButtonBox* close_bbox = manage(new Gtk::HButtonBox);
	hbox1->pack_start(*close_bbox, Gtk::PACK_SHRINK);
	
	Gtk::Button* close_btn = manage(new Gtk::Button(Gtk::Stock::CLOSE));
	close_bbox->pack_start(*close_btn, Gtk::PACK_SHRINK);
	close_btn->signal_clicked().connect(sigc::mem_fun(*this, &RequestLogWindow::hide));
	
	Gtk::Alignment* legend_align = manage(new Gtk::Alignment(1.0, 0.5, 0.0, 0.0));
	hbox1->pack_start(*legend_align);
	
	Gtk::HButtonBox* legend_bbox = manage(new Gtk::HButtonBox);
	legend_align->add(*legend_bbox);
	
	m_pLegendButton = manage(new Gtk::ToggleButton("Legend"));
	legend_bbox->pack_start(*m_pLegendButton, Gtk::PACK_SHRINK);
#if GTK_VERSION_GE(2, 6) && 0
	m_pLegendButton->set_image(*manage(new Gtk::Image(Gtk::Stock::HELP, Gtk::ICON_SIZE_BUTTON)));
#endif
	m_pLegendButton->signal_toggled().connect(sigc::mem_fun(*this, &RequestLogWindow::onLegend));
	
	Application::instance()->getRequestLogHandler().attachView(m_pTreeView);
	
	show_all_children();
}

RequestLogWindow::~RequestLogWindow()
{
	assert(m_spInstance);
	m_spInstance = 0;
	
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
RequestLogWindow::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new RequestLogWindow)->show();
	}
}

void
RequestLogWindow::on_hide()
{
	delete this;
}

bool
RequestLogWindow::on_key_press_event(GdkEventKey* evt)
{
	bool const is_ctrl_c = (evt->hardware_keycode == 54 && (evt->state & GDK_CONTROL_MASK));
	bool const event_handled = Gtk::Window::on_key_press_event(evt);
	// If event has been handled, it means cell editing was in progress.
	if (is_ctrl_c && !event_handled) {
		Application::instance()->getRequestLogHandler().copyToClipboard();
		return true;
	}
	return event_handled;
}

void
RequestLogWindow::onClear()
{
	Application::instance()->getRequestLogHandler().clearLog();
}

void
RequestLogWindow::onLegend()
{
	if (m_pLegendButton->get_active()) {
		showLegend();
	} else {
		Application::instance()->getRequestLogHandler().attachView(m_pTreeView);
	}
}

void
RequestLogWindow::showLegend()
{
	Glib::RefPtr<LegendModel> model(new LegendModel);
	m_pTreeView->set_model(model);
	m_pTreeView->remove_all_columns();
	m_pTreeView->set_headers_visible(false);
	m_pTreeView->get_selection()->set_mode(Gtk::SELECTION_NONE);
	
	LegendModel::Columns const& cols = model->getCols();
	int ncols = 0;
	
	Gtk::CellRendererPixbuf* img_renderer = Gtk::manage(new Gtk::CellRendererPixbuf);
	ncols = m_pTreeView->append_column(Glib::ustring(), *img_renderer);
	Gtk::TreeViewColumn* img_col = m_pTreeView->get_column(ncols-1);
	img_col->add_attribute(img_renderer->property_pixbuf(), cols.IMAGE);
	img_col->add_attribute(img_renderer->property_cell_background_gdk(), cols.BACKGROUND);
	
	Gtk::CellRendererText* text_renderer = Gtk::manage(new Gtk::CellRendererText);
	ncols = m_pTreeView->append_column(Glib::ustring(), *text_renderer);
	Gtk::TreeViewColumn* text_col = m_pTreeView->get_column(ncols-1);
	text_col->add_attribute(text_renderer->property_markup(), cols.TEXT);
	text_col->add_attribute(text_renderer->property_cell_background_gdk(), cols.BACKGROUND);
}


/*=================== RequestLogWindow::LegendModel ======================*/

RequestLogWindow::LegendModel::LegendModel()
:	m_ptrIconAd(CompiledImages::resp_ad_png.getPixbuf()),
	m_ptrIconCache(CompiledImages::resp_cache_png.getPixbuf()),
	m_ptrIconCancel(CompiledImages::resp_cancel_png.getPixbuf()),
	m_ptrIconError(CompiledImages::resp_error_png.getPixbuf()),
	m_ptrIconOK(CompiledImages::resp_ok_png.getPixbuf()),
	m_ptrIconRedirect(CompiledImages::resp_redirect_png.getPixbuf())
{
	set_column_types(m_cols);
	fillModel();
}

RequestLogWindow::LegendModel::~LegendModel()
{
}

void
RequestLogWindow::LegendModel::fillModel()
{
	typedef RequestLogHandler RLH;
	
	Gtk::TreeRow row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.TEXT] = "<span>Normal request</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::SUBST_REQUEST);
	row[m_cols.TEXT] = "<span>Ad subsitution request</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::ANALYZE_REQUEST);
	row[m_cols.TEXT] = "<span>Analyze request (response is to be analyzed)</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::INTERNAL_REQUEST);
	row[m_cols.TEXT] = "<span>External script fetch during HTML processing</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconOK;
	row[m_cols.TEXT] = "<span>Normal response</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconError;
	row[m_cols.TEXT] = "<span>Error response</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconCancel;
	row[m_cols.TEXT] = "<span>Request cancelled (client has disconnected)</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconRedirect;
	row[m_cols.TEXT] = "<span>Redirect response</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconAd;
	row[m_cols.TEXT] = "<span>Ad substitution response</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.IMAGE] = m_ptrIconCache;
	row[m_cols.TEXT] = "<span>\"Not Modified\" response. Data will be taken from cache.</span>";
	
	append();
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.TEXT] = "<span color=\"#5724ff\">Hint #1:  Double-clicking a URL copies it to clipboard.</span>";
	
	row = *append();
	row[m_cols.BACKGROUND] = RLH::getColorFor(RequestLog::NORMAL_REQUEST);
	row[m_cols.TEXT] = "<span color=\"#5724ff\">Hint #2:  Ctrl+C copies the whole list.</span>";
}


/*===================== RequestLogWindow::LogView ========================*/

bool
RequestLogWindow::LogView::on_button_press_event(GdkEventButton* evt)
{
	if (!(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS)) {
		return false;
	}
	
	Gtk::TreePath path;
	Gtk::TreeViewColumn* column = 0;
	int cell_x = 0;
	int cell_y = 0;
	if (!get_path_at_pos(int(evt->x), int(evt->y), path, column, cell_x, cell_y)) {
		return false;
	}
	
	Application::instance()->getRequestLogHandler().copyUrlAtRow(path);
	
	return true; // event has been handled
}

} // namespace GtkGUI
