/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef GTKGUI_CONFIGFILE_H_
#define GTKGUI_CONFIGFILE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "ConfigFileStructure.h"
#include <string>

class Config;
class FileVersion;

namespace GtkGUI
{

/**
 * \brief Stores a represenation of the "config" file structure.
 *
 * Provides high-level load-apply-save operations.  When a file is both
 * loaded and saved using this class, its structure (order of keys, comments)
 * is preserved.
 */
class ConfigFile
{
	DECLARE_NON_COPYABLE(ConfigFile) // can be easily made copyable if necessary
public:
	ConfigFile(std::string const& file_path);
	
	~ConfigFile();
	
	/**
	 * \brief Loads from disk, parses and applies the new config.
	 *
	 * Returns true on success and false on failure.  Note that success
	 * doesn't imply the absence of errors or warnings.  This function
	 * ignores parsing errors and inability to bind to listen address(es).
	 * Progress and errors / warnings will be reported through OperationLog.
	 */
	bool loadAndForceApply();
	
	/**
	 * \brief Parses, applies and saves the new config.
	 *
	 * Returns true on success and false on failure.  Note that success
	 * doesn't imply the absence of errors or warnings.  Only serious
	 * errors will cause failure.  Progress and errors / warnings will be
	 * reported through OperationLog.  In case of a failure, it tries
	 * to rollback changes previously made.  Unfortunately, in some cases
	 * in can't be done.
	 */
	bool applyAndSave(std::string const& text);
	
	/**
	 * \brief Applies and saves the new config.
	 *
	 * Returns true on success and false on failure.  Note that success
	 * doesn't imply the absence of errors or warnings.  Only serious
	 * errors will cause failure.  Progress and errors / warnings will be
	 * reported through OperationLog.  In case of a failure, it tries
	 * to rollback changes previously made.  Unfortunately, in some cases
	 * in can't be done.
	 */
	bool applyAndSave(Config const& config);
	
	/**
	 * \brief Reads file's modification time.
	 *
	 * On failure, returns FileVersion::undefined().
	 */
	FileVersion readFileVersion() const;
private:
	bool applySave(Config& new_config,
		ConfigFileStructure& new_structure, std::string const& text);
	
	static bool rebind(Config const& config);
	
	static bool rebindWithMessage(Config const& config, char const* msg);
	
	std::string m_filePath;
	ConfigFileStructure m_fileStructure;
};

} // namespace GtkGUI

#endif
